﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

param
(
    [Parameter(Mandatory=$true, Position = 0)]
    [string]
    $VsixPath
)

$scriptPath = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
$locateVsixInstallerPath = Join-Path $scriptDirectoryPath "LocateVSIXInstaller.exe"

function Test-DevEnvExists
{
    return (Get-Process devenv -ErrorAction SilentlyContinue).Length -ne 0
}

function Get-ProductMajorVersion([string] $path)
{
    return (Get-Item $path).VersionInfo.ProductMajorPart
}

if (-not (Test-Path $VsixPath))
{
    Write-Host ".VSIX file cannot be found at `"$VsixPath`""
    exit 1
}

# VSIXInstaller のパスを取得
$vsixInstallerPath = & $locateVsixInstallerPath
if ($LastExitCode -ne 0)
{
    Write-Host "Visual Studio cannot be found. Skipping installation of NintendoSdkVsExtension."
    exit 0
}

# VSIXInstaller の major バージョンを取得
$vsixInstallerMajorVersion = Get-ProductMajorVersion($vsixInstallerPath)
$is15Installer = $vsixInstallerMajorVersion -ge 15

# VSIXInstaller を実行
$argumentList = ("`"$VsixPath`"", "/quiet")

if ($is15Installer)
{
    # 15 のインストーラの場合、devenv.exe が存在するとインストールがブロックする
    # /norepair を与えておくと、すでにインストールされている場合インストールしない （Visual Studio が起動している場合でも先にチェックしてスキップされる）
    $argumentList += "/norepair"
}

$p = Start-Process -Wait -PassThru -FilePath $vsixInstallerPath -ArgumentList $argumentList
switch ($p.ExitCode)
{
    # Successful
    0
    {
        Write-Host "Successfully installed NintendoSdkVsExtension."
        exit 0
    }
    # AlreadyInstalledException
    1001
    {
        Write-Host "NintendoSdkVsExtension is already installed."
        exit 0
    }
    # BlockingProcessesException
    2004
    {
        Write-Host "Failed to install NintendoSdkVsExtension: There is a running instance of Visual Studio or another VSIX installation."
        # 15 のインストーラの場合、Visual Studio が起動しているとインストールがブロックして失敗する。
        # これはよく起こるので、 GUI インストーラにフォールバックする。
        if ($is15Installer)
        {
            Write-Host "Launching the VSIXInstaller GUI..."
            Start-Process $VsixPath
        }
        exit $p.ExitCode
    }
    default 
    {
        Write-Host " Error $($p.ExitCode) returned from the following command:"
        Write-Host "    `"$vsixInstallerPath`" $argumentList"
        Write-Host ""
        Write-Host " - Please check $($Env:TEMP)\VSIXInstaller_*.log for details."
        if ($is15Installer)
        {
            Write-Host " - Please try updating Visual Studio Installer by launching it."
        }
        Write-Host " - Please try to manually install NintendoSdkVsExtension by double-clicking ""$VsixPath""."
        exit $p.ExitCode
    }
}