﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

#include <nn/ioscrypto/crypto_impl.h>
#include <nn/ioscrypto/iosctypes.h>
#include <nn/ioscrypto/iosccert.h>

#include <nn/csl/sha1.h>
#include <nn/csl/sha256.h>
#include <nn/csl/aes.h>

#include <fcntl.h>
#include <errno.h>

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <random>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/crypto.h>
#include <nn/os.h>
#include <nn/spl/spl_Api.h>

#include <nn/bsl/bsl.h>
#include <nn/bsl/bsl_defs.h>
#include <nn/csl/csl.h>
#include "device.h"
#include "root.h"

#include <nn/settings/setting.h>

static s64 cacrlVer = 0;
static s64 crlVer   = 0;
static s64 sfsVer   = 0;
static s64 osVer    = 0;

static IOSCAesKey commonAesKey = {
    0x55, 0xa3, 0xf8, 0x72, 0xbd, 0xc8, 0x0c, 0x55,
    0x5a, 0x65, 0x43, 0x81, 0x13, 0x9e, 0x15, 0x3b,
};

static IOSCAesKey sfsAesKey = {'a','w','e','s','o','m','e',' ','s','f','s',' ','k','e','y','!'};
static IOSCHmacKey sfsHmacKey = "awsome sfs hmac key";

static u64 deviceId;
//static u8 deviceCert[] = NG000001F5_DEVEL_CERT;
static u8 deviceCert[] = NX6100000000000001_DEVEL_CERT;

#if defined(NN_BUILD_CONFIG_OS_WIN)
namespace {
    std::mt19937 g_Random;
    bool g_RandInitialized = false;
}
#endif

IOSCError
IOSCryptoEncryptAESCBC(u8 *aesKey, u8 *ivData,
               u8 *inputData, u32 inputSize,
               u8 *outputData)
{
    if (nn::crypto::EncryptAes128Cbc(outputData, inputSize, aesKey, 16, ivData, 16, inputData, inputSize) == 0) {
        return IOSC_ERROR_FAIL_INTERNAL;
    }
    memcpy(ivData, outputData + inputSize  - 16, 16);
    return IOSC_ERROR_OK;
}

IOSCError
IOSCryptoDecryptAESCBC(u8 *aesKey, u8 *ivData,
               u8 *inputData, u32 inputSize,
               u8 *outputData)
{
    u8 savedIvData[16];
    int useSavedIv = 0;

    /*
     * When decrypting "in-place", the iv for the next block must saved
     * before the decrypt operation.
     */
    if (inputData == outputData) {
        memcpy(savedIvData, inputData + inputSize - 16, 16);
        useSavedIv = 1;
    }

    if (nn::crypto::DecryptAes128Cbc(outputData, inputSize, aesKey, 16, ivData, 16, inputData, inputSize) == 0) {
        return IOSC_ERROR_FAIL_INTERNAL;
    }

    if (useSavedIv) {
        memcpy(ivData, savedIvData, 16);
    } else {
        memcpy(ivData, inputData + inputSize - 16, 16);
    }
    return IOSC_ERROR_OK;
}

IOSCError
IOSCryptoEncryptAESCBCHwKey(u32 hwKeyId, u8 *ivData,
               u8 *inputData, u32 inputSize,
               u8 *outputData)
{
    u8 *aesKey;

    if (hwKeyId == 0) {
        aesKey = commonAesKey;
    } else {
        return IOSC_ERROR_INVALID;
    }

    return IOSCryptoEncryptAESCBC(aesKey, ivData, inputData, inputSize, outputData);
}

IOSCError
IOSCryptoDecryptAESCBCHwKey(u32 hwKeyId, u8 *ivData,
               u8 *inputData, u32 inputSize,
               u8 *outputData)
{
    u8 *aesKey;

    if (hwKeyId == 0) {
        aesKey = commonAesKey;
    } else {
        return IOSC_ERROR_INVALID;
    }

    return IOSCryptoDecryptAESCBC(aesKey, ivData, inputData, inputSize, outputData);
}

IOSCError
IOSCryptoComputeSHA1(IOSCHashContext context,
    u8 *inputData, u32 inputSize,
    u32 chainingFlag,
    IOSCHash hashData)
{
    assert(sizeof(IOSCHashContext) >= sizeof(SHA1Context));

    if (chainingFlag == IOSC_HASH_FIRST) {
        SHA1Reset((SHA1Context *)context);
    } else if (chainingFlag == IOSC_HASH_RESTART) {
        /* Nothing to do in this case */
    }

    if (inputSize > 0) {
        SHA1Input((SHA1Context *)context, inputData, inputSize);
    }

    if (chainingFlag == IOSC_HASH_LAST) {
        if (hashData == NULL) {
            return IOSC_ERROR_INVALID;
        }
        SHA1Result((SHA1Context *)context, hashData);
    } else if (chainingFlag == IOSC_HASH_SUSPEND) {
        /* Nothing to do in this case */
    }

    return IOSC_ERROR_OK;
}


IOSCError
IOSCryptoComputeSHA256(IOSCHashContext context,
    u8 *inputData, u32 inputSize,
    u32 chainingFlag,
    IOSCHash256 hashData)
{
    assert(sizeof(IOSCHashContext) >= sizeof(SHA256Context));

    if (chainingFlag == IOSC_HASH_FIRST) {
        SHA256Reset((SHA256Context *)context);
    } else if (chainingFlag == IOSC_HASH_RESTART) {
        /* Nothing to do in this case */
    }

    if (inputSize > 0) {
        SHA256Input((SHA256Context *)context, inputData, inputSize);
    }

    if (chainingFlag == IOSC_HASH_LAST) {
        if (hashData == NULL) {
            return IOSC_ERROR_INVALID;
        }
        SHA256Result((SHA256Context *)context, hashData);
    } else if (chainingFlag == IOSC_HASH_SUSPEND) {
        /* Nothing to do in this case */
    }

    return IOSC_ERROR_OK;
}


/*
 * RSA public key decrypt
 *
 * Use software version in CSL
 */
IOSCError
IOSCryptoDecryptRSA(u8 *publicKey, u32 keySize,
    u8 *exponent, u32 expSize,
    u8 *inputData, u8 *outputData)
{
    IOSCError rv = IOSC_ERROR_OK;

    (void) CSL_DecryptRsa(publicKey, keySize, exponent, expSize,
            inputData, outputData);

    return rv;
}


void *IOSCryptoAllocAligned(u32 size, u32 alignment)
{
    ( void )(alignment);
    // Malloc is 16 byte aligned, good enough for mostly everything
    return malloc(size);
}

void  IOSCryptoFreeAligned(void *p, u32 alignment)
{
    ( void )(alignment);
    free(p);
}

u8 *IOSCryptoGetRootKey(void)
{
    static const u8 rootPublicKeyDev[] = ROOT_PUBLIC_KEY_DEVEL;
    static const u8 rootPublicKeyProd[] = ROOT_PUBLIC_KEY_PROD;

    bool isUseProdRootPublicKey = nn::escore::GetUseProdETicketKey();
    return isUseProdRootPublicKey ? (u8 *)rootPublicKeyProd : (u8 *)rootPublicKeyDev;
}

u8 *IOSCryptoGetRootKeyExp(void)
{
    static const u8 rootPublicExpDev[] = ROOT_PUBLIC_EXP_DEVEL;
    static const u8 rootPublicExpProd[] = ROOT_PUBLIC_EXP_PROD;

    bool isUseProdRootPublicKey = nn::escore::GetUseProdETicketKey();
    return isUseProdRootPublicKey ? (u8 *)rootPublicExpProd : (u8 *)rootPublicExpDev;
}

IOSCError IOSCryptoIncVersion(IOSCDataHandle handle)
{
    switch(handle) {
    case IOSC_CACRLVER_HANDLE:
        cacrlVer++;
        break;
    case IOSC_FSVER_HANDLE:
        sfsVer++;
        break;
    case IOSC_SIGNERCRLVER_HANDLE:
        crlVer++;
        break;
    case IOSC_BOOTOSVER_HANDLE:
        osVer++;
        break;
    default:
        return IOSC_ERROR_INVALID;
    }
    return IOSC_ERROR_OK;
}

s64 IOSCryptoGetVersion(IOSCDataHandle handle)
{
    switch(handle) {
    case IOSC_CACRLVER_HANDLE:
        return cacrlVer;
    case IOSC_FSVER_HANDLE:
        return sfsVer;
    case IOSC_SIGNERCRLVER_HANDLE:
        return crlVer;
    case IOSC_BOOTOSVER_HANDLE:
        return osVer;
    }

    return IOSC_ERROR_INVALID;
}

IOSCError IOSCryptoGenerateRand(u8 *randBuf, u32 size)
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateRandomBytes(randBuf, size));
#elif defined(NN_BUILD_CONFIG_OS_WIN)
    NN_SDK_ASSERT(g_RandInitialized);
    std::uniform_int_distribution<> rand(0, 0xFF);

    for (u32 i = 0;i < size; i++) {
        randBuf[i] = static_cast<u8>(rand(g_Random));
    }
#else
    NN_ABORT("Unexpected platform.");
#endif
    return IOSC_ERROR_OK;
}

IOSCError
IOSCryptoGetDeviceCert(u8 *cert)
{
    memcpy(cert, deviceCert, sizeof deviceCert);
    return IOSC_ERROR_OK;
}

IOSCError
IOSCryptoSetDeviceCert( const u8 *cert )
{
    memcpy( deviceCert, cert, sizeof deviceCert );
    return IOSC_ERROR_OK;
}

IOSCError
IOSCryptoSetDevicePrivateKey( const u8 *privateKey )
{
    return BSL_InstallObject( IOSC_DEV_SIGNING_KEY_HANDLE, IOSC_SECRETKEY_TYPE, IOSC_ECC233_SUBTYPE,
        privateKey, sizeof( IOSCEccPrivateKey ), NULL );
}

static void
initDeviceKeys(void)
{
    IOSCEccEccCert *cert = (IOSCEccEccCert *)deviceCert;
    int n;

    // Parse device ID from cert
    n = sscanf((char *)cert->head.name.deviceId + 2, "%llx", (u64*)&deviceId);
}


/*
 * Initialization
 */
IOSCError
IOSCryptoInitialize(void)
{
    IOSCError rv = IOSC_ERROR_OK;
    static bool s_Initialized = false;

    /* For general application use -- not highly secure :-) */
    u8 appKey[] = {0xab, 0x01, 0xb9, 0xd8, 0xe1, 0x62, 0x2b, 0x08,
                   0xaf, 0xba, 0xd8, 0x4d, 0xbf, 0xc2, 0xa5, 0x5d};

    if (s_Initialized) {
        return rv;
    }
    initDeviceKeys();

    BSL_KeyStoreInit();

    BSL_InstallObject(IOSC_DEV_ID_HANDLE, IOSC_DATA_TYPE, IOSC_CONSTANT_SUBTYPE,
              NULL, 0, (u8 *)&deviceId);

    BSL_InstallObject(IOSC_FS_ENC_HANDLE, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE,
              sfsAesKey, sizeof sfsAesKey, NULL);
    BSL_InstallObject(IOSC_FS_MAC_HANDLE, IOSC_SECRETKEY_TYPE, IOSC_MAC_SUBTYPE,
              sfsHmacKey, sizeof sfsHmacKey, NULL);
    BSL_InstallObject(IOSC_COMMON_ENC_HANDLE, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE,
              commonAesKey, sizeof commonAesKey, NULL);
    BSL_InstallObject(IOSC_APP_ENC_HANDLE, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE,
              appKey, sizeof appKey, NULL);

    BSL_InstallObject(IOSC_BOOTOSVER_HANDLE, IOSC_DATA_TYPE, IOSC_VERSION_SUBTYPE,
              NULL, 0, (u8 *)&osVer);

    BSL_InstallObject(IOSC_CACRLVER_HANDLE, IOSC_DATA_TYPE, IOSC_VERSION_SUBTYPE,
              NULL, 0, (u8 *)&cacrlVer);

    BSL_InstallObject(IOSC_SIGNERCRLVER_HANDLE, IOSC_DATA_TYPE, IOSC_VERSION_SUBTYPE,
              NULL, 0, (u8 *)&crlVer);

    BSL_InstallObject(IOSC_FSVER_HANDLE, IOSC_DATA_TYPE, IOSC_VERSION_SUBTYPE,
              NULL, 0, (u8 *)&sfsVer);

    /* set up protection for secret keys */
    BSL_SetProtection(IOSC_DEV_SIGNING_KEY_HANDLE, BSL_PROT_NO_EXPORT);
    BSL_SetProtection(IOSC_FS_ENC_HANDLE, BSL_PROT_NO_EXPORT);
    BSL_SetProtection(IOSC_FS_MAC_HANDLE, BSL_PROT_NO_EXPORT);
    BSL_SetProtection(IOSC_COMMON_ENC_HANDLE, BSL_PROT_NO_EXPORT);
    BSL_SetProtection(IOSC_APP_ENC_HANDLE, BSL_PROT_NO_EXPORT);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_SDK_ASSERT(!g_RandInitialized);

    // FIXME: Get appropriate random seeds.
    g_Random.seed(static_cast<unsigned long>(nn::os::GetSystemTick().GetInt64Value()));
    g_RandInitialized = true;
#endif

    s_Initialized = true;
    return rv;
}
