﻿// 文字コード:UTF-8
/// @file

#include <nn/os/os_Argument.h>
#include <nn/oe/oe_SpecificToOceanApis.private.h>

#include "ErrorLogger.hpp"
#include "ErrorLoggerUnitTest.hpp"
#include "InternalArgs.hpp"
#include "InternalArgsUnitTest.hpp"
#include "OfflineCommand.hpp"
#include "SplitCommand.hpp"
#include "SplitCommandUnitTest.hpp"
#include "WebCommand.hpp"

#include "sys/account/Account.hpp"

namespace
{

// 利用可能な全メモリ
const size_t fApplicationMemorySize = 768 * 1024 * 1024;

// 割り当て可能な最大サイズ
const size_t fMallocMemorySize = 512 * 1024 * 1024;

}

//-----------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    ::nn::Result result = ::nn::os::SetMemoryHeapSize(fApplicationMemorySize);
    NN_ASSERT(result.IsSuccess());
    uintptr_t address;
    result = ::nn::os::AllocateMemoryBlock(&address, fMallocMemorySize);
    NN_ASSERT(result.IsSuccess());
    ::nn::init::InitializeAllocator(reinterpret_cast<void*>(address), fMallocMemorySize);
}

//-----------------------------------------------------------------------------
void UnitTest()
{
#if DEBUG_IS_ENABLED
    DEBUG_LOG("\n-UnitTest Start -----------------------------\n\n");

    ::ErrorLoggerUnitTest::UnitTest();
    ::SplitCommandUnitTest::UnitTest();
    ::InternalArgsUnitTest::UnitTest();

    DEBUG_LOG("\n-UnitTest End -----------------------------\n\n");
#endif
}

//-----------------------------------------------------------------------------
void run()
{
    // アカウント準備します。
    ::sys::account::Account account;

    DEBUG_EVAL(UnitTest());

#if DEBUG_IS_ENABLED
    // テスト用の起動引数
    // 1個目は実行ファイルのパスが渡される為ここでは null としています。
#if 0
    const char* args[] = {
        "null",
        "web", "http://www.nintendo.com",
    };
#else
    const char* args[] = {
        "null",
        "offline", "debug://test.htdocs/index.html",
        "--media-player-speed-control-enabled", "true",
    };
#endif

    // 起動引数から有効なコマンドのみに分けます。
    auto count = sizeof(args) / sizeof(args[0]);
    ::SplitCommand splitCmd(count, reinterpret_cast<char**>(&args));
    if (::nn::os::GetHostArgc() > 1) {
        splitCmd.reset(::nn::os::GetHostArgc(), ::nn::os::GetHostArgv());
    }
#else
    // 起動引数から有効なコマンドのみに分けます。
    ::SplitCommand splitCmd(::nn::os::GetHostArgc(), ::nn::os::GetHostArgv());
#endif

    // コマンド側が扱える内部引数に変換します。
    ::InternalArgs internalArgs(splitCmd);

    // Help表示を求められていたら表示する。
    if (internalArgs.displayHelpKind() != InternalArgs::DisplayHelpKind::None) {
        if (internalArgs.displayHelpKind() == InternalArgs::DisplayHelpKind::Web) {
            ::ErrorLogger::LogWebHelp();
        }
        else if (internalArgs.displayHelpKind() == InternalArgs::DisplayHelpKind::Offline) {
            ::ErrorLogger::LogOfflineHelp();
        }
        else {
            ::ErrorLogger::LogHelp();
        }

        // splitComd側からヘルプ表示の要請があるならばユーザーの意思では無いので FAILURE を表示します。
        if (splitCmd.requiresHelp()) {
            NN_LOG("[FAILURE]\n");
        }
        else {
            NN_LOG("[SUCCESS]\n");
        }
        return;
    }

    bool isSuccess = false;
    // 有効なコマンドがあればターゲットに合ったコマンドを発行します。
    if (internalArgs.isValid()) {
        if (internalArgs.showAppletKind() == ::InternalArgs::ShowAppletKind::Web) {
            isSuccess = ::WebCommand::Command(internalArgs);
        }
        else if (internalArgs.showAppletKind() == ::InternalArgs::ShowAppletKind::Offline) {
            isSuccess = ::OfflineCommand::Command(internalArgs);
        }
        else {
            SYS_ASSERT_NOT_REACHED();
        }
    }
    // 有効なコマンドが無ければヘルプメッセージのみ表示して終了します。
    else {
        ::ErrorLogger::LogHelp();
    }

    if (isSuccess) {
        NN_LOG("[SUCCESS]\n");
    }
    else {
        NN_LOG("[FAILURE]\n");
    }
}

//-----------------------------------------------------------------------------
extern "C" void nnMain()
{
    run();
    // @note 自発終了時に Ocean 側がエラーと判断しないよう 終了時に呼び出す必要がある NINTENDOSDK-4635
    ::nn::oe::ExitApplication();
}

//-----------------------------------------------------------------------------
// EOF
