﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <lib/debug/Print.hpp>
#include <lib/ArrayNum.hpp>
#include <lib/Join.hpp>
#include <nn/nn_Abort.h>

//------------------------------------------------------------------------------

/// @addtogroup LIB-Debug
//@{
/// @name 静的アサート
//@{
/// コンパイル時アサート。
#define LIB_COMPILER_ASSERT(aCond) static_assert(aCond, "")

/// 配列の数をチェックするコンパイル時アサート。
#define LIB_ASSERT_ARRAY_NUM(aArray, aNum) \
    LIB_COMPILER_ASSERT(LIB_ARRAY_NUM((aArray)) == (aNum))
//@}

/// @name 実行時アサート
//@{
/// 普通のアサート。
#define LIB_ASSERT(aCond) \
    LIB_ASSERT_MSG_IMPL(aCond, ::lib::debug::Assert::MessageAssertDefault)

/// メッセージ付きアサート。
#define LIB_ASSERT_MSG(aCond, ...) \
    LIB_ASSERT_MSG_IMPL(aCond, __VA_ARGS__)

/// アサートに失敗したら、指定変数の値を出力する。(１変数版)
#define LIB_ASSERT1(aCond, aVariable1) \
    LIB_ASSERT_MSG_IMPL(aCond, ::lib::debug::Assert::FormatAssert1, \
        LIB_SPRINT(aVariable1).readPtr() \
        )

/// アサートに失敗したら、指定変数の値を出力する。(２変数版)
#define LIB_ASSERT2(aCond, aVariable1, aVariable2) \
    LIB_ASSERT_MSG_IMPL(aCond, ::lib::debug::Assert::FormatAssert2, \
        LIB_SPRINT(aVariable1).readPtr(), \
        LIB_SPRINT(aVariable2).readPtr() \
        )

/// アサートに失敗したら、指定変数の値を出力する。(３変数版)
#define LIB_ASSERT3(aCond, aVariable1, aVariable2, aVariable3) \
    LIB_ASSERT_MSG_IMPL(aCond, ::lib::debug::Assert::FormatAssert3, \
        LIB_SPRINT(aVariable1).readPtr(), \
        LIB_SPRINT(aVariable2).readPtr(), \
        LIB_SPRINT(aVariable3).readPtr()  \
        )

/// 有効であるべきポインタへのアサート。
#define LIB_ASSERT_POINTER(aCond) \
    LIB_ASSERT_POINTER_MSG(aCond, ::lib::debug::Assert::MessageAssertPointer)

/// ポインタの正当性を確かめるメッセージつきアサート。
#define LIB_ASSERT_POINTER_MSG(aCond, ...) \
    LIB_ASSERT_POINTER_MSG_IMPL(aCond, __VA_ARGS__)

/// 強制的に止めるアサート。
#define LIB_ASSERT_NOT_REACHED() \
    LIB_ASSERT_MSG(false, ::lib::debug::Assert::MessageAssertNotReach)

/// 強制的に止めるメッセージ付きアサート。
#define LIB_ASSERT_NOT_REACHED_MSG(...) LIB_ASSERT_MSG(false, __VA_ARGS__)

/// aValue1 == aValue2。
#define LIB_ASSERT_EQUAL(aValue1, aValue2) \
    LIB_ASSERT_MSG(((aValue1) == (aValue2)), \
    ::lib::debug::Assert::FormatAssertEqual, \
    LIB_SPRINT_VAR(aValue1).readPtr(), \
    LIB_SPRINT_VAR(aValue2).readPtr() \
    )

/// aValue1 != aValue2。
#define LIB_ASSERT_NOT_EQUAL(aValue1, aValue2) \
    LIB_ASSERT_MSG(((aValue1) != (aValue2)), \
    ::lib::debug::Assert::FormatAssertNotEqual, \
    LIB_SPRINT_VAR(aValue1).readPtr(), \
    LIB_SPRINT_VAR(aValue2).readPtr() \
    )

/// aValue1 < aValue2
#define LIB_ASSERT_LESS(aValue1, aValue2) \
    LIB_ASSERT_MSG(((aValue1) < (aValue2)), \
    ::lib::debug::Assert::FormatAssertLess, \
    LIB_SPRINT_VAR(aValue1).readPtr(), \
    LIB_SPRINT_VAR(aValue2).readPtr() \
    )

/// aValue1 <= aValue2
#define LIB_ASSERT_LESS_EQUAL(aValue1, aValue2) \
    LIB_ASSERT_MSG(((aValue1) <= (aValue2)), \
    ::lib::debug::Assert::FormatAssertLessEqual, \
    LIB_SPRINT_VAR(aValue1).readPtr(), \
    LIB_SPRINT_VAR(aValue2).readPtr() \
    )

/// aValue1 > aValue2
#define LIB_ASSERT_GREATER(aValue1, aValue2) \
    LIB_ASSERT_MSG(((aValue1) > (aValue2)), \
    ::lib::debug::Assert::FormatAssertGreater, \
    LIB_SPRINT_VAR(aValue1).readPtr(), \
    LIB_SPRINT_VAR(aValue2).readPtr() \
    )

/// aValue1 >= aValue2
#define LIB_ASSERT_GREATER_EQUAL(aValue1, aValue2) \
    LIB_ASSERT_MSG(((aValue1) >= (aValue2)), \
    ::lib::debug::Assert::FormatAssertGreaterEqual, \
    LIB_SPRINT_VAR(aValue1).readPtr(), \
    LIB_SPRINT_VAR(aValue2).readPtr() \
    )

/// aMin <= aValue && aValue <= aMax
#define LIB_ASSERT_MIN_MAX(aValue, aMin, aMax) \
    LIB_ASSERT_MSG(((aMin) <= (aValue) && (aValue) <= (aMax)), \
    ::lib::debug::Assert::FormatAssertMinMax, \
    LIB_SPRINT_VAR(aValue).readPtr(), \
    LIB_SPRINT_VAR(aMin).readPtr(), \
    LIB_SPRINT_VAR(aMax).readPtr() \
    )

/// aMin <= aValue && aValue < aTerm
#define LIB_ASSERT_MIN_TERM(aValue, aMin, aTerm) \
    LIB_ASSERT_MSG(((aMin) <= (aValue) && (aValue) < (aTerm)), \
    ::lib::debug::Assert::FormatAssertMinTerm, \
    LIB_SPRINT_VAR(aValue).readPtr(), \
    LIB_SPRINT_VAR(aMin).readPtr(), \
    LIB_SPRINT_VAR(aTerm).readPtr() \
    )

/// Enum用。libコーディングルールに従ってると使える。
#define LIB_ASSERT_ENUM(aTypeName, aValue) \
    LIB_ASSERT_MSG((0 <= (aValue) && (aValue) < (aTypeName::TERM)), \
    ::lib::debug::Assert::FormatAssertEnum, \
    LIB_SPRINT_VAR(aValue).readPtr(), \
    "0", \
    LIB_SPRINT_VAR(aTypeName::TERM).readPtr() \
    )
//@}
//@}

//------------------------------------------------------------------------------
namespace lib {
namespace debug {
/// Assertのための補助関数および変数群。
/// @details
/// 文字列シンボルが多量にコピーされて、コードサイズが肥大するのを防ぐため。
struct Assert
    : private ::lib::NonCreatable
{
    static const char* const MessagePanic;
    static const char* const MessageAssertDefault;
    static const char* const MessageAssertNotReach;
    static const char* const MessageAssertPointer;
    static const char* const FormatAssert1;
    static const char* const FormatAssert2;
    static const char* const FormatAssert3;
    static const char* const FormatAssertEqual;
    static const char* const FormatAssertNotEqual;
    static const char* const FormatAssertLess;
    static const char* const FormatAssertLessEqual;
    static const char* const FormatAssertGreater;
    static const char* const FormatAssertGreaterEqual;
    static const char* const FormatAssertMinMax;
    static const char* const FormatAssertMinTerm;
    static const char* const FormatAssertEnum;

    /// ポインタが有効なアドレスを示していればtrue。
    static bool IsPointerValid(const void* aPointer);
};

}} // namespace

/// Panic関数のエイリアス。
#define LIB_ASSERT_PANIC() \
    do { \
        LIB_LOG_ERROR(::lib::debug::Assert::MessagePanic); \
        NN_ABORT(); \
    } while (false)

/// メッセージ付きアサートの実装。
#define LIB_ASSERT_MSG_IMPL(aCond, ...) \
    do {\
        if (!(aCond)) {\
            LIB_LOG_TRACE("%s:%d Assert:", __FILE__, __LINE__); \
            LIB_LOG_TRACE("Function  : %s", __FUNCTION__); \
            LIB_LOG_TRACE("Expression: %s", #aCond); \
            LIB_LOG_TRACE(__VA_ARGS__); \
            LIB_ASSERT_PANIC(); \
        } \
    } while (false)

/// ポインタの正当性を確かめるメッセージつきアサートの実装。
#define LIB_ASSERT_POINTER_MSG_IMPL(aCond, ...) \
    do { \
        const bool valid \
            = ::lib::debug::Assert::IsPointerValid( \
            reinterpret_cast<const void*>(aCond)); \
        if (!valid) { \
            LIB_ASSERT_MSG_IMPL(false, __VA_ARGS__); \
        } \
  } while (false)

// EOF
