﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <lib/Math.hpp>
#include <lib/debug/Assert.hpp>
#include <lib/ShortString.hpp>

LIB_FORWARD_DECLARE_2(lib, class Matrix4x3);
LIB_FORWARD_DECLARE_2(lib, class Matrix4x4);

//------------------------------------------------------------------------------
namespace lib {

/// @addtogroup LIB-Math
//@{
/// ３次元ベクトル
/// @details
/// ・移植元はRVL版。
/// ・宣言の可読性を優先するため、インライン関数は宣言と分けて記述すること。
struct Vector3Pod
{
    /// @name 変数
    //@{
    float x;
    float y;
    float z;
    //@}

    /// @name 列挙型
    //@{
    /// rotate()で使用する軸の種類
    class AxisKind
        : private ::lib::NonCreatable
    {
    public:
        enum EnumType
        {
            X, ///< X軸
            Y, ///< Y軸
            Z ///< Z軸
        };
    };
    //@}

    /// @name 定数
    //@{
    static const Vector3Pod Zero();  ///< 0ベクトル (0, 0, 0)
    static const Vector3Pod One();   ///< X,Y,Zが全て1のベクトル。
    static const Vector3Pod Min();   ///< 最小値のベクトル。
    static const Vector3Pod Max();   ///< 最大値のベクトル。
    static const Vector3Pod UnitX(); ///< X単位ベクトル。
    static const Vector3Pod UnitY(); ///< Y単位ベクトル。
    static const Vector3Pod UnitZ(); ///< Z単位ベクトル。
    static const Vector3Pod NegUnitX(); ///< -X単位ベクトル。
    static const Vector3Pod NegUnitY(); ///< -Y単位ベクトル。
    static const Vector3Pod NegUnitZ(); ///< -Z単位ベクトル。
    //@}

    /// @name 作成関数
    //@{
    inline static const Vector3Pod Create(float aX, float aY, float aZ);
    //@}

    /// @name 変換
    //@{
    const Vector2Pod toXX() const;
    const Vector2Pod toXY() const;
    const Vector2Pod toXZ() const;
    const Vector2Pod toYX() const;
    const Vector2Pod toYY() const;
    const Vector2Pod toYZ() const;
    const Vector2Pod toZX() const;
    const Vector2Pod toZY() const;
    const Vector2Pod toZZ() const;
    const Vector3Pod toXY0() const;
    const Vector3Pod toX0Z() const;
    const Vector3Pod toX00() const;
    const Vector3Pod to0YZ() const;
    const Vector3Pod to0Y0() const;
    const Vector3Pod to00Z() const;
    const Vector3Pod toXXX() const;
    const Vector3Pod toYYY() const;
    const Vector3Pod toZZZ() const;
    const Vector4Pod toXYZ0() const;
    const Vector4Pod toXXXX() const;
    const Vector4Pod toYYYY() const;
    const Vector4Pod toZZZZ() const;
    //@}

    /// @name 値の設定
    //@{
    /// 値を設定する。
    inline void set(float aX, float aY, float aZ);
    //@}

    /// @name 等値比較
    //@{
    /// 等しいか？（許容誤差あり）
    inline bool equalsLoose(const Vector3Pod& aRhs) const;
    /// 等しいか？（許容誤差あり）
    inline bool equalsLoose(const Vector3Pod& aRhs, float aTolerance) const;
    /// 等しいか？（厳密）
    inline bool equalsStrict(const Vector3Pod& aRhs) const;
    /// 等しいか？（厳密）
    inline bool operator==(const Vector3Pod& aRhs) const;
    /// 等しくないか？（厳密）
    inline bool operator!=(const Vector3Pod& aRhs) const;
    /// ０ベクトルか？（許容誤差あり）
    inline bool isZeroLoose() const;
    /// ０ベクトルか？（許容誤差あり）
    inline bool isZeroLoose(float aTolerance) const;
    /// ０ベクトルか？（厳密）
    inline bool isZeroStrict() const;
    /// ０ベクトルか？（厳密）
    inline bool isZero() const;
    //@}

    /// @name 加減乗除
    //@{
    /// ベクトルを加算する。
    inline Vector3Pod& operator+=(const Vector3Pod& aRhs);
    /// ベクトルを減算する。
    inline Vector3Pod& operator-=(const Vector3Pod& aRhs);
    /// ベクトルを乗算する。
    inline Vector3Pod& operator*=(const Vector3Pod& aRhs);
    /// ベクトルを除算する。
    inline Vector3Pod& operator/=(const Vector3Pod& aRhs);
    /// スカラーで乗算する。
    inline Vector3Pod& operator*=(float aScalar);
    /// スカラーで除算する。
    inline Vector3Pod& operator/=(float aScalar);
    /// 行列を乗算する。
    Vector3Pod& operator*=(const Matrix4x3& aMtx);
    /// 行列を乗算する。
    Vector3Pod& operator*=(const Matrix4x4& aMtx);
    /// ベクトルを加算した結果を得る。
    inline const Vector3Pod operator+(const Vector3Pod& aRhs) const;
    /// ベクトルを減算した結果を得る。
    inline const Vector3Pod operator-(const Vector3Pod& aRhs) const;
    /// ベクトルを乗算した結果を得る。
    inline const Vector3Pod operator*(const Vector3Pod& aRhs) const;
    /// ベクトルを除算した結果を得る。
    inline const Vector3Pod operator/(const Vector3Pod& aRhs) const;
    /// スカラーで乗算した結果を得る。
    inline const Vector3Pod operator*(float aScalar) const;
    /// スカラーで除算した結果を得る。
    inline const Vector3Pod operator/(float aScalar) const;
    /// 行列を積算した結果を得る。
    const Vector3Pod operator*(const Matrix4x3& aMtx) const;
    /// 行列を積算した結果を得る。
    const Vector3Pod operator*(const Matrix4x4& aMtx) const;
    /// スクリプト用
    inline const Vector3Pod add(const Vector3Pod& aRhs) const { return *this + aRhs; }
    inline const Vector3Pod sub(const Vector3Pod& aRhs) const { return *this - aRhs; }
    inline const Vector3Pod mul(const Vector3Pod& aRhs) const { return *this * aRhs; }
    inline const Vector3Pod div(const Vector3Pod& aRhs) const { return *this / aRhs; }
    inline const Vector3Pod mul(float aScalar) const { return *this * aScalar; }
    inline const Vector3Pod div(float aScalar) const { return *this / aScalar; }
    inline const Vector3Pod subVec(const Vector3Pod& aRhs) const { return *this - aRhs; }
    inline const Vector3Pod mulSca(const float aScalar) const { return *this * aScalar; }
    inline const Vector3Pod divSca(const float aScalar) const { return *this / aScalar; }
    inline Vector3Pod& mulAssignVec(const Vector3Pod& aRhs) { return *this *= aRhs; }
    inline Vector3Pod& divAssignVec(const Vector3Pod& aRhs) { return *this /= aRhs; }
    inline Vector3Pod& mulAssignSca(const float aScalar) { return *this *= aScalar; }
    inline Vector3Pod& divAssignSca(const float aScalar) { return *this /= aScalar; }
    //@}

    /// @name 選択
    //@{
    /// 指定ベクトルとの各要素の最小値を選択したベクトルを取得する。
    inline const Vector3Pod min(const Vector3Pod& aRhs) const;
    /// 指定ベクトルとの各要素の最大値を選択したベクトルを取得する。
    inline const Vector3Pod max(const Vector3Pod& aRhs) const;
    //@}

    /// @name クランプ
    //@{
    /// aMin以上aMax以下になるようにクランプする。
    inline const Vector3Pod clamp(const Vector3Pod& aMin, const Vector3Pod& aMax) const;
    /// clamp(Zero(), Max()) を取得する。
    inline const Vector3Pod clampPositive() const;
    /// clamp(Min(), Zero()) を取得する。
    inline const Vector3Pod clampNegative() const;
    //@}

    /// @name 符号操作
    //@{
    /// 各要素を正に置き換えたベクトルを取得。
    inline const Vector3Pod abs() const;
    /// 符号を反転させた結果を取得する。
    inline const Vector3Pod neg() const;
    //@}

    /// @name 長さ
    //@{
    /// ベクトルの長さ。
    inline float length() const;
    /// ベクトルの長さの２乗。
    inline float squareLength() const;
    /// ベクトル間の距離。
    inline float distance(const Vector3Pod& aRhs) const;
    /// ベクトル間の距離の２乗。
    inline float squareDistance(const Vector3Pod& aRhs) const;
    /// ベクトルの長さを指定した長さにする
    /// @param aTarget 目標の長さ
    /// @return 変化する前のベクトルの大きさ
    float setLength(float aTarget);
    /// 指定ベクトルの大きさがlimitを超えているか調べ、超過分をカットする
    /// @return 超過していた場合trueを返す
    bool limit(float aLimit);
    //@}

    /// @name 正規化
    //@{
    /// 正規化する。
    /// @return 正規化前のベクトルの長さ。
    /// @details ０ベクトルの場合はアサート失敗。
    float unitAssign();
    /// ０ベクトルチェックありの正規化。
    /// @return 正規化前のベクトルの長さ。
    /// @details ０ベクトルの場合は正規化せず、そのまま。
    inline float permittedUnitAssign();
    /// 正規化されたベクトルを得る。
    inline const Vector3Pod unit() const;
    /// 正規化されているか？
    inline bool isUnit() const;
    //@}


    /// @name 内積・外積
    //@{
    /// 内積
    inline float dot(const Vector3Pod& aRhs) const;
    /// 外積
    inline const Vector3Pod cross(const Vector3Pod& aRhs) const;
    /// 正規化された外積を得る。
    const Vector3Pod unitCross(const Vector3Pod& aRhs) const;
    /// 正規化された外積を得る（０ベクトル許容）。
    /// @return 外積の結果を正規化したベクトル。２ベクトルが平行の時は０ベクトル。
    /// @details もし外積の結果が０ベクトルの時（２ベクトルが平行の時）、０ベクトルを返す。
    const Vector3Pod permittedUnitCross(const Vector3Pod& aRhs) const;
    //@}


    /// @name 方向・角度
    //@{
    /// 逆向きのベクトル
    inline const Vector3Pod operator-() const;
    /// 反転する。
    inline void reverse();
    /// ２ベクトルのなす角のcos。
    float cos(const Vector3Pod& aRhs) const;
    /// ２ベクトルのなす角。
    /// @param aRhs 対象となるもう一つのベクトル。
    /// @return なす角（ラジアン）。
    inline float angle(const Vector3Pod& aRhs) const;
    /// ２つのベクトルが向かい合っているか？
    /// @param aRhs 対象となるもう一つのベクトル。
    /// @return ２ベクトルのなす角が９０度より大きいならtrue。
    inline bool isFace(const Vector3Pod& aRhs) const;
    /// ２つのベクトルは平行か？
    inline bool isParallel(const Vector3Pod& aRhs) const;
    /// ２つのベクトルは垂直か？
    inline bool isVertical(const Vector3Pod& aRhs) const;
    /// ２ベクトルに垂直な方向ベクトル（正規化済み）を得る。
    /// @details
    /// もし２ベクトルが平行ならば、無数に存在する垂直ベクトルから勝手に選択をする。
    /// ２ベクトルが平行でない時は、this->unitCross(rhs)に等しい。
    /// ２つのベクトルはともに０ベクトルであってはならない。
    Vector3Pod getVertical(const Vector3Pod& aRhs) const;
    /// 指定軸に沿って、ベクトルを回転する。
    /// @param aAxis 回転軸を表すベクトル。正規化されている必要はない。
    /// @param aDegree 回転角（1周を360度とする角度）。半時計回りを正とする。
    ///             （aAxisベクトルの先端から根元見たときに半時計回りの方向）
    void rotateDeg(const Vector3Pod& aAxis, float aDegree);
    void rotateDegVec(const Vector3Pod& aAxis, float aDegree); // スクリプト用
    /// 指定軸に沿って、ベクトルを回転する（X,Y,Z軸指定）。
    /// @param aAxisKind 回転軸の種類。
    /// @param aDegree 回転角（360度を1周とする角度）。半時計回りを正とする。
    ///              （aAxisベクトルの先端から根元見たときに半時計回りの方向）
    void rotateDeg(AxisKind::EnumType aAxisKind, float aDegree);
    /// 指定軸に沿って、ベクトルを回転する。
    /// @param aAxis 回転軸を表すベクトル。正規化されている必要はない。
    /// @param aRadian 回転角（ラジアン）。半時計回りを正とする。
    ///               （aAxisベクトルの先端から根元見たときに半時計回りの方向）
    void rotateRad(const Vector3Pod& aAxis, float aRadian);
    void rotateRadVec(const Vector3Pod& aAxis, float aDegree); // スクリプト用
    /// 指定軸に沿って、ベクトルを回転する（X,Y,Z軸指定）。
    /// @param aAxisKind 回転軸の種類。
    /// @param aRadian 回転角（ラジアン）。半時計回りを正とする。
    ///              （aAxisベクトルの先端から根元見たときに半時計回りの方向）
    void rotateRad(AxisKind::EnumType aAxisKind, float aRadian);
    /// 指定角速度で、ベクトルを回転する。
    /// @param aDest 回転目標のベクトル。正規化済みであること。
    /// @param aAngularSpeedDegree 360度を1周とする角速度。[0.0f, 360.0f]
    /// @details
    /// thisベクトルから、aDestベクトルへ向けて、angleSpeedで指定された各速度で
    /// 回転させた結果のベクトル（正規化済み）を得る。
    /// 回転の方向はより近い方向。thisとaDestが平行の場合はアサートでハングする。
    /// thisベクトルとaDestベクトルのなす角がangleSpeed以下なら、（正規化された）aDestベクトルとなる。
    /// thisとaDestは正規化済みであること（もちろんどちらも０ベクトルであってはいけない）。
    /// @return aDest に到達した時に true を返す。
    bool rotateBySpeedDeg(const Vector3Pod& aDest, float aAngularSpeedDegree);
    /// 指定角速度で、ベクトルを回転する。
    /// @param aDest 回転目標のベクトル。正規化済みであること。
    /// @param aAngularSpeedRadian ラジアン単位の角速度。[0.0f, Pi]
    /// @details
    /// thisベクトルから、aDestベクトルへ向けて、angleSpeedで指定された各速度で
    /// 回転させた結果のベクトル（正規化済み）を得る。
    /// 回転の方向はより近い方向。thisとaDestが平行の場合はアサートでハングする。
    /// thisベクトルとaDestベクトルのなす角がangleSpeed以下なら、（正規化された）aDestベクトルとなる。
    /// thisとaDestは正規化済みであること（もちろんどちらも０ベクトルであってはいけない）。
    /// @return aDest に到達した時に true を返す。
    bool rotateBySpeedRad(const Vector3Pod& aDest, float aAngularSpeedRadian);
    /// 指定角速度で、ベクトルを回転する。
    /// @details
    /// thisベクトルとaDestが平行だと外積を使って回転軸が求められない。
    /// その時はaDestとaAxisの外積を取ってそこを回転軸にする。
    /// @return aDest に到達した時に true を返す。
    bool rotateBySpeedChangeDestDeg(
        const Vector3Pod& aDest,
        float aAngularSpeedDegree,
        const Vector3Pod& aAxis = Vector3Pod::UnitY()
        );
    /// 指定角速度で、ベクトルを回転する。
    /// @details
    /// thisベクトルとaDestが平行だと外積を使って回転軸が求められない。
    /// その時はaDestとaAxisの外積を取ってそこを回転軸にする。
    /// @return aDest に到達した時に true を返す。
    bool rotateBySpeedChangeDestRad(
        const Vector3Pod& aDest,
        float aAngularSpeedRadian,
        const Vector3Pod& aAxis = Vector3Pod::UnitY()
        );
    /// thisとaDest（両ベクトルとも正規化済み）の球面補間を求める。(aRatio=0でthis, aRatio=1でaDest)
    /// @param aDest  Targetベクトル
    /// @param aRatio 補間比率
    /// @return      補間結果
    Vector3Pod slerp(const Vector3Pod& aDest, float aRatio) const;
    /// ベクトルを指定軸に沿って回転した時、目標に近づく回転が正回転か、負回転かを得る
    /// @param aDest 目標ベクトル
    /// @param aAxis 回転軸
    /// @return 正回転の時はtrue、負回転の時はfalseを返す
    bool rotSign(const Vector3Pod& aDest, const Vector3Pod& aAxis) const;
    /// ベクトルを法線で表現される平面に対して投影する
    /// @param aNormal 射影する平面の法線
    void projection(const Vector3Pod& aNormal);
    /// ベクトルを法線で表現される平面に対して投影するが、長さはキープ
    /// @param aNormal 射影する平面の法線
    void projectionKeepLength(const Vector3Pod& aNormal);
    /// 法線から傾斜角度（水平面とのなす角度）を求める。
    /// Y成分が負の場合（天井など）は、[π/2, π]の値を返すので注意。
    /// @return 傾斜角度
    float slopeAngle() const;
    /// ベクトルを法線で表現される平面に対して反射する。反射率は１で計算。
    /// @param aNormal 反射する平面の法線
    void reflect(const Vector3Pod& aNormal);
    /// ベクトルを法線で表現される平面に対して反射する
    /// @param aNormal 反射する平面の法線
    /// @param aRatio 反射率
    void reflect(const Vector3Pod& aNormal, float aRatio);
    /// 法線で表現される平面に対して反射したベクトルを得る
    /// 法線で表現される平面に対して反射したベクトルを得る。反射率は１で計算。
    /// @param aNormal 反射する平面の法線
    /// @return 反射結果のベクトル
    Vector3Pod getReflected(const Vector3Pod& aNormal) const;
    /// @param aNormal 反射する平面の法線
    /// @param aRatio 反射率
    /// @return 反射結果のベクトル
    Vector3Pod getReflected(const Vector3Pod& aNormal, float aRatio) const;
    /// 各軸それぞれ倍率が記されたベクトルを用いてスケーリングを行う
    /// @param aScale スケーリングベクトル
    void scale(const Vector3Pod& aScale);
    /// 各軸それぞれ倍率が記されたベクトルを用いてスケーリングを行ったベクトルを得る
    /// @param aScale スケーリングベクトル
    /// @return スケーリング後のベクトル
    Vector3Pod getScaled(const Vector3Pod& aScale) const;
    //@}

    /// @name デバッグ
    //@{
    /// 文字列表現を得る。
    const ::lib::ShortString toShortString() const;
    //std::string toString() const;
    //friend ::std::ostream& operator<<(std::ostream& out, const Vector3Pod& v);
    //@}
};
//@}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::Create(const float aX, const float aY, const float aZ)
{
    Vector3Pod obj;
    obj.set(aX, aY, aZ);
    return obj;
}

//------------------------------------------------------------------------------
void Vector3Pod::set(const float aX, const float aY, const float aZ)
{
    this->x = aX;
    this->y = aY;
    this->z = aZ;
}

//------------------------------------------------------------------------------
bool Vector3Pod::equalsLoose(const Vector3Pod& aRhs) const
{
    return equalsLoose(aRhs, LIB_MATH_EPSILON);
}

//------------------------------------------------------------------------------
bool Vector3Pod::equalsLoose(const Vector3Pod& aRhs, float aTolerance) const
{
    SYS_ASSERT(0.0f <= aTolerance);
    return (
        ::lib::Math::Equals(x, aRhs.x, aTolerance) &&
        ::lib::Math::Equals(y, aRhs.y, aTolerance) &&
        ::lib::Math::Equals(z, aRhs.z, aTolerance)
        );
}

//------------------------------------------------------------------------------
bool Vector3Pod::equalsStrict(const Vector3Pod& aRhs) const
{
    return (x == aRhs.x && y == aRhs.y && z == aRhs.z);
}

//------------------------------------------------------------------------------
bool Vector3Pod::operator==(const Vector3Pod& aRhs) const
{
    return equalsStrict(aRhs);
}

//------------------------------------------------------------------------------
bool Vector3Pod::operator!=(const Vector3Pod& aRhs) const
{
    return !equalsStrict(aRhs);
}

//------------------------------------------------------------------------------
bool Vector3Pod::isZeroLoose() const
{
    return isZeroLoose(LIB_MATH_EPSILON);
}

//------------------------------------------------------------------------------
bool Vector3Pod::isZeroLoose(const float aTolerance) const
{
    return equalsLoose(Zero(), aTolerance);
}

//------------------------------------------------------------------------------
bool Vector3Pod::isZeroStrict() const
{
    return equalsStrict(Zero());
}

//------------------------------------------------------------------------------
bool Vector3Pod::isZero() const
{
    return isZeroStrict();
}

//------------------------------------------------------------------------------
Vector3Pod& Vector3Pod::operator+=(const Vector3Pod& aRhs)
{
    x += aRhs.x;
    y += aRhs.y;
    z += aRhs.z;
    return *this;
}

//------------------------------------------------------------------------------
Vector3Pod& Vector3Pod::operator-=(const Vector3Pod& aRhs)
{
    x -= aRhs.x;
    y -= aRhs.y;
    z -= aRhs.z;
    return *this;
}

//------------------------------------------------------------------------------
Vector3Pod& Vector3Pod::operator*=(const Vector3Pod& aRhs)
{
    x *= aRhs.x;
    y *= aRhs.y;
    z *= aRhs.z;
    return *this;
}

//------------------------------------------------------------------------------
Vector3Pod& Vector3Pod::operator/=(const Vector3Pod& aRhs)
{
    x /= aRhs.x;
    y /= aRhs.y;
    z /= aRhs.z;
    return *this;
}

//------------------------------------------------------------------------------
Vector3Pod& Vector3Pod::operator*=(const float aScalar)
{
    x *= aScalar;
    y *= aScalar;
    z *= aScalar;
    return *this;
}

//------------------------------------------------------------------------------
Vector3Pod& Vector3Pod::operator/=(const float aScalar)
{
    SYS_ASSERT(aScalar != 0.0f);
    return operator*=(1.0f / aScalar);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::operator+(const Vector3Pod& aRhs) const
{
    Vector3Pod tmp = *this;
    return tmp += aRhs;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::operator-(const Vector3Pod& aRhs) const
{
    Vector3Pod tmp = *this;
    return tmp -= aRhs;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::operator*(const Vector3Pod& aRhs) const
{
    Vector3Pod tmp = *this;
    return tmp *= aRhs;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::operator/(const Vector3Pod& aRhs) const
{
    Vector3Pod tmp = *this;
    return tmp /= aRhs;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::operator*(const float aScalar) const
{
    Vector3Pod tmp = *this;
    return tmp *= aScalar;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::operator/(const float aScalar) const
{
    SYS_ASSERT(aScalar != 0.0f);
    Vector3Pod tmp = *this;
    return tmp /= aScalar;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::min(const Vector3Pod& aRhs) const
{
    return Vector3Pod::Create(
        Math::Min(x, aRhs.x),
        Math::Min(y, aRhs.y),
        Math::Min(z, aRhs.z)
        );
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::max(const Vector3Pod& aRhs) const
{
    return Vector3Pod::Create(
        Math::Max(x, aRhs.x),
        Math::Max(y, aRhs.y),
        Math::Max(z, aRhs.z)
        );
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::clamp(
    const Vector3Pod& aMin,
    const Vector3Pod& aMax
    ) const
{
    return max(aMin).min(aMax);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::clampPositive() const
{
    return max(Zero());
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::clampNegative() const
{
    return min(Zero());
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::abs() const
{
    return Vector3Pod::Create(
        Math::Abs(x),
        Math::Abs(y),
        Math::Abs(z)
        );
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::neg() const
{
    Vector3Pod vec = *this;
    vec *= -1.0f;
    return vec;
}

//------------------------------------------------------------------------------
float Vector3Pod::length() const
{
    return Math::Sqrt(squareLength());
}

//------------------------------------------------------------------------------
float Vector3Pod::squareLength() const
{
    return dot(*this);
}

//------------------------------------------------------------------------------
float Vector3Pod::distance(const Vector3Pod& aRhs) const
{
    return Math::Sqrt(squareDistance(aRhs));
}

//------------------------------------------------------------------------------
float Vector3Pod::squareDistance(const Vector3Pod& aRhs) const
{
    return (aRhs - *this).squareLength();
}

//------------------------------------------------------------------------------
float Vector3Pod::permittedUnitAssign()
{
    if (length() == 0.0f) {
        return 0.0f;
    }
    return unitAssign();
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::unit() const
{
    Vector3Pod result = *this;
    result.unitAssign();
    return result;
}

//------------------------------------------------------------------------------
bool Vector3Pod::isUnit() const
{
    return ::lib::Math::Equals(squareLength(), 1.0f);
}

//------------------------------------------------------------------------------
float Vector3Pod::dot(const Vector3Pod& aRhs) const
{
    return x * aRhs.x + y * aRhs.y + z * aRhs.z;
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::cross(const Vector3Pod& aRhs) const
{
    return Vector3Pod::Create(
        y * aRhs.z - z * aRhs.y,
        z * aRhs.x - x * aRhs.z,
        x * aRhs.y - y * aRhs.x
        );
}

//------------------------------------------------------------------------------
const Vector3Pod Vector3Pod::operator-() const
{
    return operator*(-1);
}

//------------------------------------------------------------------------------
void Vector3Pod::reverse()
{
    operator*=(-1);
}

//------------------------------------------------------------------------------
float Vector3Pod::angle(const Vector3Pod& aRhs) const
{
    return ::std::acos(cos(aRhs));
}

//------------------------------------------------------------------------------
bool Vector3Pod::isFace(const Vector3Pod& aRhs) const
{
    // 誤差が出にくい方法として現在のところはcos()を利用しています。(05/09/07)
    float cosValue = cos(aRhs);
    return cosValue < 0.0f;
}

//------------------------------------------------------------------------------
bool Vector3Pod::isParallel(const Vector3Pod& aRhs) const
{
    // 誤差が出にくい方法として現在のところはcos()を利用しています。(05/09/07)
    float cosValue = cos(aRhs);
    return (
        ::lib::Math::Equals(cosValue, -1.0f) ||
        ::lib::Math::Equals(cosValue, 1.0f)
        );
}

//------------------------------------------------------------------------------
bool Vector3Pod::isVertical(const Vector3Pod& aRhs) const
{
    // 誤差が出にくい方法として現在のところはcos()を利用しています。(05/09/07)
    float cosValue = cos(aRhs);
    return (::lib::Math::Equals(cosValue, 0.0f));
}
} // namespace
// EOF
