﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <lib/ShortString.hpp>

namespace lib {
    class Vector2;
}

//------------------------------------------------------------------------------
namespace lib {

/// @addtogroup LIB-Math
//@{
/// 整数型2次元ベクトル構造体。
/// @details メモリ配置は32bit符号有り整数が先頭からx,yの順番に並んでいます。
struct Vector2iPod
{
    /// @name 変数
    //@{
    int x; ///< x。
    int y; ///< y。
    //@}

    /// @name 定数取得
    //@{
    static const Vector2iPod Zero();     ///< 0ベクトル。
    static const Vector2iPod One();      ///< X,Yが全て1のベクトル。
    static const Vector2iPod Min();      ///< 最小値のベクトル。
    static const Vector2iPod Max();      ///< 最大値のベクトル。
    static const Vector2iPod UnitX();    ///< X単位ベクトル。
    static const Vector2iPod UnitY();    ///< Y単位ベクトル。
    static const Vector2iPod NegUnitX(); ///< -X単位ベクトル。
    static const Vector2iPod NegUnitY(); ///< -Y単位ベクトル。
    //@}

    /// @name 変換
    //@{
    const Vector2iPod toX0() const;
    const Vector2iPod to0Y() const;
    const Vector2iPod toXX() const;
    const Vector2iPod toYX() const;
    const Vector2iPod toYY() const;
    const Vector2     toXYf() const;
    const Vector2     toVector2() const { return toXYf(); }
    //@}

    /// @name 値を設定する
    //@{
    void set(int aX, int aY);
    //@}

    /// @name 等価比較
    //@{
    bool equals(const Vector2iPod& aVec) const;       ///< 等価比較。
    //@}

    /// @name 四則演算
    //@{
    const Vector2iPod add(int aVal) const; ///< 各要素に値を加算した結果を取得する。
    const Vector2iPod sub(int aVal) const; ///< 各要素から値を減算した結果を取得する。
    const Vector2iPod mul(int aVal) const; ///< 各要素に値をかけた結果を取得する。
    const Vector2iPod div(int aVal) const; ///< 各要素から値をわった結果を取得する。
    const Vector2iPod add(const Vector2iPod& aVal) const; ///< 対応する値同士を加算した結果を取得する。
    const Vector2iPod sub(const Vector2iPod& aVal) const; ///< 対応する値に対して減算した結果を取得する。
    const Vector2iPod mul(const Vector2iPod& aVal) const; ///< 対応する値同士をかけた結果を取得する。
    const Vector2iPod div(const Vector2iPod& aVal) const; ///< 対応する値に対して割り算をした結果を取得する。
    void addAssign(int aVal); ///< 各要素に値を加算する。
    void subAssign(int aVal); ///< 各要素から値を減算する。
    void mulAssign(int aVal); ///< 各要素に値をかける。
    void divAssign(int aVal); ///< 各要素から値をわる。
    void addAssign(const Vector2iPod& aVal); ///< 対応する値同士を加算する。
    void subAssign(const Vector2iPod& aVal); ///< 対応する値に対して減算する。
    void mulAssign(const Vector2iPod& aVal); ///< 対応する値同士をかける。
    void divAssign(const Vector2iPod& aVal); ///< 対応する値に対して割り算をする。
    //@}

    /// @name 演算子オーバーロード
    //@{
    bool operator==(const Vector2iPod& aRhs) const; ///< equals()。
    bool operator!=(const Vector2iPod& aRhs) const; ///< !equals()。
    const Vector2iPod operator+(int aVal) const; ///< add()。
    const Vector2iPod operator-(int aVal) const; ///< sub()。
    const Vector2iPod operator*(int aVal) const; ///< mul()。
    const Vector2iPod operator/(int aVal) const; ///< div()。
    const Vector2iPod operator+(const Vector2iPod& aVal) const; ///< add()。
    const Vector2iPod operator-(const Vector2iPod& aVal) const; ///< sub()。
    const Vector2iPod operator*(const Vector2iPod& aVal) const; ///< mul()。
    const Vector2iPod operator/(const Vector2iPod& aVal) const; ///< div()。
    const Vector2iPod operator-() const; ///< neg()。
    Vector2iPod& operator+=(int aVal); ///< addAssign()。
    Vector2iPod& operator-=(int aVal); ///< subAssign()。
    Vector2iPod& operator*=(int aVal); ///< mulAssign()。
    Vector2iPod& operator/=(int aVal); ///< divAssign()。
    Vector2iPod& operator+=(const Vector2iPod& aVal); ///< addAssign()。
    Vector2iPod& operator-=(const Vector2iPod& aVal); ///< subAssign()。
    Vector2iPod& operator*=(const Vector2iPod& aVal); ///< mulAssign()。
    Vector2iPod& operator/=(const Vector2iPod& aVal); ///< divAssign()。
    //@}

    /// @name 選択
    //@{
    const Vector2iPod min(const Vector2iPod& aVal) const; ///< 指定のベクトルとの各要素の最小値を選択したベクトルを取得する。
    const Vector2iPod max(const Vector2iPod& aVal) const; ///< 指定のベクトルとの各要素の最大値を選択したベクトルを取得する。
    //@}

    /// @name クランプ
    //@{
    /// aMin以上aMax以下になるようにクランプする
    void clamp(const Vector2iPod& aMin, const Vector2iPod& aMax);
    /// aMin以上aMax以下になるようにクランプしたものを取得する。
    const Vector2iPod getClamped(const Vector2iPod& aMin, const Vector2iPod& aMax) const;
    /// getClamped(Zero(), Max()) を取得する。
    const Vector2iPod getClampedPositive() const;
    /// getClamped(Min(), Zero()) を取得する。
    const Vector2iPod getClampedNegative() const;
    //@}

    /// @name 符号
    //@{
    const Vector2iPod abs() const; ///< 各要素を正に置き換えたベクトルを取得。
    const Vector2iPod neg() const; ///< 符号を反転させた結果を取得する。
    bool isPositive() const; ///< 各要素が0以上か。
    /// Xを反転させる
    void reverseX() { x *= -1; }
    /// Yを反転させる
    void reverseY() { y *= -1; }
    /// Xを反転させた結果を取得する
    Vector2iPod getXReversed() const { Vector2iPod v = *this; v.reverseX(); return v; }
    /// Yを反転させた結果を取得する
    Vector2iPod getYReversed() const { Vector2iPod v = *this; v.reverseY(); return v; }
    //@}

    /// @name 長さ
    //@{
    bool isZero() const;                      ///< 全てが0か。
    //@}

    /// @name ユーティリティ
    //@{
    /// x,y形式の文字列に変換。
    const ::lib::ShortString toShortString() const;
    //@}
};
//@}

} // namespace
// EOF
