﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <lib/Math.hpp>
#include <lib/debug/Assert.hpp>
#include <lib/ShortString.hpp>

namespace lib {
    struct Vector2iPod;
    struct Vector3Pod;
    struct Vector4Pod;
}

//------------------------------------------------------------------------------
namespace lib {


/// @addtogroup LIB-Math
//@{
/// ２次元ベクトル
/// @details
/// ・移植元はRVL版。
/// ・宣言の可読性を優先するため、インライン関数は宣言と分けて記述すること。
struct Vector2Pod
{
    /// @name 変数
    //@{
    float x;
    float y;
    //@}

    /// @name 定数
    //@{
    static const Vector2Pod Zero();  ///< 0ベクトル (0, 0)
    static const Vector2Pod One();   ///< X,Y全て1のベクトル。
    static const Vector2Pod Min();   ///< 最小値のベクトル。
    static const Vector2Pod Max();   ///< 最大値のベクトル。
    static const Vector2Pod UnitX(); ///< X単位ベクトル。
    static const Vector2Pod UnitY(); ///< Y単位ベクトル。
    static const Vector2Pod NegUnitX(); ///< -X単位ベクトル。
    static const Vector2Pod NegUnitY(); ///< -Y単位ベクトル。
    //@}

    /// @name 作成関数
    //@{
    inline static const Vector2Pod Create(float aX, float aY);
    //@}

    /// @name 変換
    //@{
    const Vector2Pod toX0() const;
    const Vector2Pod to0Y() const;
    const Vector2Pod toXX() const;
    const Vector2Pod toYX() const;
    const Vector2Pod toYY() const;
    const Vector3Pod toXY0() const;
    const Vector3Pod toX0Y() const;
    const Vector3Pod toXXX() const;
    const Vector3Pod toYYY() const;
    const Vector4Pod toXY00() const;
    const Vector4Pod toXXXX() const;
    const Vector4Pod toYYYY() const;
    //@}

    /// @name 値を設定する
    //@{
    inline void set(float aX, float aY);
    //@}

    /// @name 等値比較
    //@{
    /// 等しいか？（許容誤差あり）
    inline bool equalsLoose(const Vector2Pod& aRhs) const;
    /// 等しいか？（許容誤差あり）
    inline bool equalsLoose(const Vector2Pod& aRhs, float aTolerance) const;
    /// 等しいか？（厳密）
    inline bool equalsStrict(const Vector2Pod& aRhs) const;
    /// 等しいか？（厳密）
    inline bool operator==(const Vector2Pod& aRhs) const;
    /// 等しくないか？（厳密）
    inline bool operator!=(const Vector2Pod& aRhs) const;
    /// ０ベクトルか？（許容誤差あり）
    inline bool isZeroLoose() const;
    /// ０ベクトルか？（許容誤差あり）
    inline bool isZeroLoose(float aTolerance) const;
    /// ０ベクトルか？（厳密）
    inline bool isZeroStrict() const;
    /// ０ベクトルか？（厳密）
    inline bool isZero() const;
    // スクリプト用
    inline bool equalsLoose1(const Vector2Pod& aRhs) const;
    inline bool equalsLoose2(const Vector2Pod& aRhs, float aTolerance) const;
    //@}

    /// @name 加減乗除
    //@{
    /// ベクトルを加算する。
    inline Vector2Pod& operator+=(const Vector2Pod& aRhs);
    /// ベクトルを減算する。
    inline Vector2Pod& operator-=(const Vector2Pod& aRhs);
    /// ベクトルを乗算する。
    inline Vector2Pod& operator*=(const Vector2Pod& aRhs);
    /// ベクトルを除算する。
    inline Vector2Pod& operator/=(const Vector2Pod& aRhs);
    /// スカラーで乗算する。
    inline Vector2Pod& operator*=(float aScalar);
    /// スカラーで除算する。
    inline Vector2Pod& operator/=(float aScalar);
    /// ベクトルを加算した結果を得る。
    inline const Vector2Pod operator+(const Vector2Pod& aRhs) const;
    /// ベクトルを減算した結果を得る。
    inline const Vector2Pod operator-(const Vector2Pod& aRhs) const;
    /// ベクトルを乗算した結果を得る。
    inline const Vector2Pod operator*(const Vector2Pod& aRhs) const;
    /// ベクトルを除算した結果を得る。
    inline const Vector2Pod operator/(const Vector2Pod& aRhs) const;
    /// スカラーで乗算した結果を得る。
    inline const Vector2Pod operator*(float aScalar) const;
    /// スカラーで除算した結果を得る。
    inline const Vector2Pod operator/(float aScalar) const;
    /// スクリプト用
    inline const Vector2Pod add(const Vector2Pod& aRhs) const { return *this + aRhs; }
    inline const Vector2Pod sub(const Vector2Pod& aRhs) const { return *this - aRhs; }
    inline const Vector2Pod mul(const Vector2Pod& aRhs) const { return *this * aRhs; }
    inline const Vector2Pod div(const Vector2Pod& aRhs) const { return *this / aRhs; }
    inline const Vector2Pod mul(const float aScalar) const { return *this * aScalar; }
    inline const Vector2Pod div(const float aScalar) const { return *this / aScalar; }
    inline const Vector2Pod subVec(const Vector2Pod& aRhs) const { return *this - aRhs; }
    inline const Vector2Pod mulSca(const float aScalar) const { return *this * aScalar; }
    inline const Vector2Pod divSca(const float aScalar) const { return *this / aScalar; }
    inline Vector2Pod& mulAssignVec(const Vector2Pod& aRhs) { return *this *= aRhs; }
    inline Vector2Pod& divAssignVec(const Vector2Pod& aRhs) { return *this /= aRhs; }
    inline Vector2Pod& mulAssignSca(const float aScalar) { return *this *= aScalar; }
    inline Vector2Pod& divAssignSca(const float aScalar) { return *this /= aScalar; }
    //@}

    /// @name 選択
    //@{
    inline const Vector2Pod min(const Vector2Pod& aRhs) const; ///< 指定のベクトルとの各要素の最小値を選択したベクトルを取得する。
    inline const Vector2Pod max(const Vector2Pod& aRhs) const; ///< 指定のベクトルとの各要素の最大値を選択したベクトルを取得する。
    //@}

    /// @name クランプ
    //@{
    /// aMin以上aMax以下になるようにクランプする
    inline void clamp(const Vector2Pod& aMin, const Vector2Pod& aMax);
    /// aMin以上aMax以下になるようにクランプしたものを取得する。
    inline const Vector2Pod getClamped(const Vector2Pod& aMin, const Vector2Pod& aMax) const;
    /// getClamped(Zero(), Max()) を取得する。
    inline const Vector2Pod getClampedPositive() const;
    /// getClamped(Min(), Zero()) を取得する。
    inline const Vector2Pod getClampedNegative() const;
    //@}

    /// @name 符号操作
    //@{
    /// 各要素を正に置き換えたベクトルを取得。
    inline const Vector2Pod abs() const;
    /// 符号を反転させた結果を取得する。
    inline const Vector2Pod neg() const;
    //@}

    /// @name 長さ
    //@{
    /// ベクトルの長さ。
    inline float length() const;
    /// ベクトルの長さの２乗。
    inline float squareLength() const;
    /// ベクトル間の距離。
    inline float distance(const Vector2Pod& aRhs) const;
    /// ベクトル間の距離の２乗。
    inline float squareDistance(const Vector2Pod& aRhs) const;
    /// 指定ベクトルの大きさがlimitを超えているか調べ、超過分をカットする
    bool limit(float aLimit);
    /// 正規化する。
    float unitAssign();
    /// 長さを強制的に設定。
    float setLength(float aTarget);
    //@}

    /// @name 正規化
    //@{
    // ０ベクトルチェックありの正規化。
    // ０ベクトルの場合は正規化せず、そのまま。
    // @return 正規化前のベクトルの長さ。
    inline float permittedUnitAssign();
    /// 正規化されたベクトルを得る。
    inline const Vector2Pod unit() const;
    /// 正規化されているか？
    inline bool isUnit() const;
    //@}

    /// @name 内積・外積
    //@{
    /// 内積
    inline float dot(const Vector2Pod& aRhs) const;
    /// 外積
    inline float cross(const Vector2Pod& aRhs) const;
    //@}

    /// @name 方向・角度
    //@{
    /// 逆向きのベクトル
    inline const Vector2Pod operator-() const;
    /// 反転する。
    inline void reverse();
    /// Xを反転する。
    inline void reverseX();
    /// Yを反転する。
    inline void reverseY();
    /// Xを反転させた結果を取得する
    Vector2Pod getXReversed() const { Vector2Pod v = *this; v.reverseX(); return v; }
    /// Yを反転させた結果を取得する
    Vector2Pod getYReversed() const { Vector2Pod v = *this; v.reverseY(); return v; }
    /// ２ベクトルのなす角のcos。
    float cos(const Vector2Pod& aRhs) const;
    /// ２ベクトルのなす角。
    /// @param aRhs 対象となるもう一つのベクトル。
    /// @return なす角（ラジアン）。
    inline float angle(const Vector2Pod& aRhs) const;
    ///
    float signedAngle(const Vector2Pod& aRhs) const;
    /// ２つのベクトルが向かい合っているか？
    /// @param aRhs 対象となるもう一つのベクトル。
    /// @return ２ベクトルのなす角が９０度より大きいならtrue。
    inline bool isFace(const Vector2Pod& aRhs) const;
    /// ２つのベクトルは平行か？
    inline bool isParallel(const Vector2Pod& aRhs) const;
    /// ２つのベクトルは垂直か？
    inline bool isVertical(const Vector2Pod& aRhs) const;
    /// 垂直なベクトルを得る。
    /// @details 元のベクトルの始点から終点に向かって、左向きのベクトルを返す。
    inline const Vector2Pod getVertical() const;
    /// Z軸回転。
    void rotateDeg(float aDegree);
    /// Z軸回転。
    void rotateRad(float aRadian);
    /// あるベクトルに射影したベクトルを得る。
    /// @param aRhs 射影の基準となるベクトル。生成されるベクトルはこのベクトルに平行なものになる。
    inline const Vector2Pod getProjected(const Vector2Pod& aRhs) const;
    ///
    void project(const Vector2Pod& aRhs);
    /**
     * ある面に対して反射させたベクトルを得る。
     *
     * @param aRhs 反射面の法線。
     */
    inline const Vector2Pod getReflected(const Vector2Pod& aRhs) const;
    inline const Vector2Pod getReflected(const Vector2Pod& aRhs, const float aRatio) const;
    ///
    void reflect(const Vector2Pod& aRhs);
    void reflect(const Vector2Pod& aRhs, const float aRatio);
    // スクリプト用
    void reflect1(const Vector2Pod& aRhs);
    void reflect2(const Vector2Pod& aRhs, const float aRatio);
    //@}

    /// @name スケーリング
    //@{
    void scale(const Vector2Pod& scl);
    Vector2Pod getScaled(const Vector2Pod& scl) const;
    //@}

    /// @name 丸め込み
    //@{
    /// 切り捨てする
    void floor();
    /// 切り捨てした結果を取得する
    Vector2Pod getFloored() const;
    /// 切り捨てした結果を「Vector2i」として取得する
    Vector2iPod getFlooredVector2i() const;
    /// 四捨五入する
    void round();
    /// 四捨五入した結果を取得する
    Vector2Pod getRounded() const;
    /// 四捨五入した結果を「Vector2i」として取得する
    Vector2iPod getRoundedVector2i() const;
    //@}

    /// @name デバッグ
    //@{
    const ::lib::ShortString toShortString() const;
    //@}
};
//@}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::Create(float aX, float aY)
{
    Vector2Pod obj;
    obj.set(aX, aY);
    return obj;
}

//------------------------------------------------------------------------------
void Vector2Pod::set(float aX, float aY)
{
    this->x = aX;
    this->y = aY;
}

//------------------------------------------------------------------------------
bool Vector2Pod::equalsLoose(const Vector2Pod& aRhs) const
{
    return equalsLoose(aRhs, LIB_MATH_EPSILON);
}

//------------------------------------------------------------------------------
bool Vector2Pod::equalsLoose(const Vector2Pod& aRhs, float aTolerance) const
{
    SYS_ASSERT(0.0f <= aTolerance);
    return (::lib::Math::Equals(x, aRhs.x, aTolerance) &&
            ::lib::Math::Equals(y, aRhs.y, aTolerance));
}

//------------------------------------------------------------------------------
bool Vector2Pod::equalsStrict(const Vector2Pod& aRhs) const
{
    return (x == aRhs.x && y == aRhs.y);
}

//------------------------------------------------------------------------------
bool Vector2Pod::operator==(const Vector2Pod& aRhs) const
{
    return equalsStrict(aRhs);
}

//------------------------------------------------------------------------------
bool Vector2Pod::operator!=(const Vector2Pod& aRhs) const
{
    return !equalsStrict(aRhs);
}

//------------------------------------------------------------------------------
bool Vector2Pod::isZeroLoose() const
{
    return isZeroLoose(LIB_MATH_EPSILON);
}

//------------------------------------------------------------------------------
bool Vector2Pod::isZeroLoose(float aTolerance) const
{
    return equalsLoose(Zero(), aTolerance);
}

//------------------------------------------------------------------------------
bool Vector2Pod::isZeroStrict() const
{
    return equalsStrict(Zero());
}

//------------------------------------------------------------------------------
bool Vector2Pod::isZero() const
{
    return isZeroStrict();
}

//------------------------------------------------------------------------------
bool Vector2Pod::equalsLoose1(const Vector2Pod& aRhs) const
{
    return equalsLoose(aRhs);
}

//------------------------------------------------------------------------------
bool Vector2Pod::equalsLoose2(const Vector2Pod& aRhs, float aTolerance) const
{
    return equalsLoose(aRhs, aTolerance);
}

//------------------------------------------------------------------------------
Vector2Pod& Vector2Pod::operator+=(const Vector2Pod& aRhs)
{
    this->x += aRhs.x;
    this->y += aRhs.y;
    return *this;
}

//------------------------------------------------------------------------------
Vector2Pod& Vector2Pod::operator-=(const Vector2Pod& aRhs)
{
    this->x -= aRhs.x;
    this->y -= aRhs.y;
    return *this;
}

//------------------------------------------------------------------------------
Vector2Pod& Vector2Pod::operator*=(const Vector2Pod& aRhs)
{
    this->x *= aRhs.x;
    this->y *= aRhs.y;
    return *this;
}

//------------------------------------------------------------------------------
Vector2Pod& Vector2Pod::operator/=(const Vector2Pod& aRhs)
{
    this->x /= aRhs.x;
    this->y /= aRhs.y;
    return *this;
}

//------------------------------------------------------------------------------
Vector2Pod& Vector2Pod::operator*=(float aScalar)
{
    this->x *= aScalar;
    this->y *= aScalar;
    return *this;
}

//------------------------------------------------------------------------------
Vector2Pod& Vector2Pod::operator/=(float aScalar)
{
    SYS_ASSERT_NOT_EQUAL(aScalar, 0.0f);
    return operator*=(1.0f / aScalar);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::operator+(const Vector2Pod& aRhs) const
{
    return Vector2Pod(*this) += aRhs;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::operator-(const Vector2Pod& aRhs) const
{
    return Vector2Pod(*this) -= aRhs;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::operator*(const Vector2Pod& aRhs) const
{
    return Vector2Pod(*this) *= aRhs;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::operator/(const Vector2Pod& aRhs) const
{
    return Vector2Pod(*this) /= aRhs;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::operator*(float aScalar) const
{
    return Vector2Pod(*this) *= aScalar;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::operator/(float aScalar) const
{
    SYS_ASSERT_NOT_EQUAL(aScalar, 0.0f);
    return Vector2Pod(*this) /= aScalar;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::min(const Vector2Pod& aRhs) const
{
    return Vector2Pod::Create(
        ::lib::Math::Min(x, aRhs.x),
        ::lib::Math::Min(y, aRhs.y)
        );
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::max(const Vector2Pod& aRhs) const
{
    return Vector2Pod::Create(
        ::lib::Math::Max(x, aRhs.x),
        ::lib::Math::Max(y, aRhs.y)
        );
}

//------------------------------------------------------------------------------
void Vector2Pod::clamp(const Vector2Pod& aMin, const Vector2Pod& aMax)
{
    SYS_ASSERT(aMin.x <= aMax.x && aMin.y <= aMax.y);
    *this = max(aMin).min(aMax);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::getClamped(
    const Vector2Pod& aMin,
    const Vector2Pod& aMax
    ) const
{
    Vector2Pod r = *this;
    r.clamp(aMin, aMax);
    return r;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::getClampedPositive() const
{
    return max(Zero());
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::getClampedNegative() const
{
    return min(Zero());
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::abs() const
{
    return Vector2Pod::Create(
        Math::Abs(x),
        Math::Abs(y)
        );
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::neg() const
{
    Vector2Pod vec = *this;
    vec *= -1.0f;
    return vec;
}

//------------------------------------------------------------------------------
float Vector2Pod::length() const
{
    return ::lib::Math::Sqrt(squareLength());
}

//------------------------------------------------------------------------------
float Vector2Pod::squareLength() const
{
    return dot(*this);
}

//------------------------------------------------------------------------------
float Vector2Pod::distance(const Vector2Pod& aRhs) const
{
    return ::lib::Math::Sqrt(squareDistance(aRhs));
}

//------------------------------------------------------------------------------
float Vector2Pod::squareDistance(const Vector2Pod& aRhs) const
{
    return
        (this->x - aRhs.x) * (this->x - aRhs.x) +
        (this->y - aRhs.y) * (this->y - aRhs.y);
}

//------------------------------------------------------------------------------
float Vector2Pod::permittedUnitAssign()
{
    float len = length();
    if (::lib::Math::Equals(len, 0.0f)) {
        return 0.0f;
    }
    operator/=(len);
    return len;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::unit() const
{
    Vector2Pod result(*this);
    result.unitAssign();
    return result;
}

//------------------------------------------------------------------------------
bool Vector2Pod::isUnit() const
{
    return ::lib::Math::Equals(squareLength(), 1.0f);
}

//------------------------------------------------------------------------------
float Vector2Pod::dot(const Vector2Pod& aRhs) const
{
    return this->x * aRhs.x + this->y * aRhs.y;
}

//------------------------------------------------------------------------------
float Vector2Pod::cross(const Vector2Pod& aRhs) const
{
    return this->x * aRhs.y - this->y * aRhs.x;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::operator-() const
{
    return operator*(-1);
}

//------------------------------------------------------------------------------
void Vector2Pod::reverse()
{
    operator*=(-1);
}

//------------------------------------------------------------------------------
void Vector2Pod::reverseX()
{
    x *= -1;
}

//------------------------------------------------------------------------------
void Vector2Pod::reverseY()
{
    y *= -1;
}

//------------------------------------------------------------------------------
float Vector2Pod::angle(const Vector2Pod& aRhs) const
{
    float magFactor = this->length() * aRhs.length();

    // どちらかのベクトルの長さが0の時のなす角は0度とする。
    if (magFactor < LIB_MATH_EPSILON) {
        return 0.0f;
    }

    return ::std::acos(cos(aRhs));
}

//------------------------------------------------------------------------------
bool Vector2Pod::isFace(const Vector2Pod& aRhs) const
{
    // 誤差が出にくい方法として現在のところはcos()を利用しています。(05/09/07)
    float cosValue = cos(aRhs);
    return cosValue < 0.0f;
}

//------------------------------------------------------------------------------
bool Vector2Pod::isParallel(const Vector2Pod& aRhs) const
{
    // 誤差が出にくい方法として現在のところはcos()を利用しています。(05/09/07)
    float cosValue = cos(aRhs);
    return (::lib::Math::Equals(cosValue, -1.0f) ||
            ::lib::Math::Equals(cosValue, 1.0f));
}

//------------------------------------------------------------------------------
bool Vector2Pod::isVertical(const Vector2Pod& aRhs) const
{
    // 誤差が出にくい方法として現在のところはcos()を利用しています。(05/09/07)
    float cosValue = cos(aRhs);
    return (::lib::Math::Equals(cosValue, 0.0f));
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::getVertical() const
{
    Vector2Pod result(*this);
    result.x = -this->y;
    result.y = this->x;
    return result;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::getProjected(const Vector2Pod& aRhs) const
{
    Vector2Pod result(*this);
    result.project(aRhs);
    return result;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::getReflected(const Vector2Pod& aRhs) const
{
    Vector2Pod result(*this);
    result.reflect(aRhs);
    return result;
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::getReflected(const Vector2Pod& aRhs, const float aRatio) const
{
    Vector2Pod result(*this);
    result.reflect(aRhs, aRatio);
    return result;
}

} // namespace
// EOF
