﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <lib/NonCopyable.hpp>
#include <lib/debug/Assert.hpp>

//------------------------------------------------------------------------------
namespace lib {

/// @addtogroup LIB
//@{
/// デストラクタで自動的にdelete[]を発行する、スマートポインタ。
/// @details
/// 配列として確保された(new[]された)ポインタを保持して、デストラクト時に
/// delete[]を呼び出してくれる。
///
/// 注意点としては、([]の付かない)newで確保したポインタに使用しては
/// いけない。そのようなケースにはScopedPtrを使用すること。
///
/// 使用例は、TestUtilScopedArray()を参照。
template <typename TPointee>
class ScopedArray
    : private ::lib::NonCopyable
{
private:
    typedef TPointee* ScopedArray::*SafeBool; ///< SafeBoolイディオムのための型。

public:
    /**
     * コンストラクタ。
     *
     * @param aPtr newで確保したポインタ。現時点で必要なければnullを指定。
     */
    inline explicit ScopedArray(TPointee* aPtr = 0);

    /**
     * デストラクタ。
     *
     * 内部で保持するポインタに対して、delete[]を発行する。
     * nullに対してもdelete[]するのは言語仕様でOK。念のため。
     */
    inline ~ScopedArray();

    /**
     * 内部で保持するポインタを再設定する。
     *
     * 以前に保持していたポインタには、delete[]を発行する。
     */
    inline void reset(TPointee* aPtr = 0);

    /**
     * thisと引数のScopedArrayで、内部で保持するmPtrを交換する。
     *
     * STL等での使用のために用意。
     */
    inline void swap(ScopedArray* aScopedArray);

    /**
     * インデックス参照する。
     *
     * nullだとアサートにかかる。
     * また、範囲外へのアクセスは、未定義動作。
     */
    inline TPointee& operator[](int aIndex) const;

    /**
     * 内部で保持するポインタを返す。
     *
     * 内部のポインタがnullでも、そのままnullが返る。
     * また、所有権の移動も行われないので、get()で保持したポインタを
     * delete[]してはいけない。(ScopedArrayのデストラクタでdelete[]しようとする)
     */
    inline TPointee* get() const;

    /**
     * ポインタがnullでなければtrue、ポインタがnullならfalseを返す。
     *
     * 奇妙な見た目だが、SafeBoolと呼ばれる、安全なoperator bool()相当の
     * 処理をするような仕組み。つまり、通常のポインタのように、
     * 以下のようなコードを記述できる。
     * <code><pre>
     * ScopedArray ptr(...);
     * if (ptr) {
     *   // ptrがnullでない時の処理。
     * }
     * </pre></code>
     */
    inline operator SafeBool() const;

private:
    TPointee* mPtr;
};
//@}

//------------------------------------------------------------------------------
template <typename TPointee>
ScopedArray<TPointee>::ScopedArray(TPointee* const aPtr)
: mPtr(aPtr)
{
}

//------------------------------------------------------------------------------
template <typename TPointee>
ScopedArray<TPointee>::~ScopedArray()
{
    delete[] mPtr;
}

//------------------------------------------------------------------------------
template <typename TPointee>
void ScopedArray<TPointee>::reset(TPointee* const aPtr)
{
    SYS_ASSERT(aPtr == 0 || mPtr != aPtr); // nullでない、同じポインタでresetは禁止。
    ScopedArray tmp(aPtr);
    tmp.swap(this);
    // ここで、tmpがデストラクト。つまり、古いmPtrがdelete[]。
}

//------------------------------------------------------------------------------
template <typename TPointee>
void ScopedArray<TPointee>::swap(ScopedArray* aScopedArray)
{
    SYS_ASSERT_POINTER(aScopedArray);
    TPointee* tmp = aScopedArray->mPtr;
    aScopedArray->mPtr = mPtr;
    mPtr = tmp;
}

//------------------------------------------------------------------------------
template <typename TPointee>
TPointee& ScopedArray<TPointee>::operator[](const int aIndex) const
{
    SYS_ASSERT(mPtr);
    SYS_ASSERT(0 <= aIndex);
    return mPtr[aIndex];
}

//------------------------------------------------------------------------------
template <typename TPointee>
TPointee* ScopedArray<TPointee>::get() const
{
    return mPtr;
}

//------------------------------------------------------------------------------
template <typename TPointee>
ScopedArray<TPointee>::operator SafeBool() const
{
    return mPtr == 0 ? 0 : &ScopedArray::mPtr;
}

} // namespace
// EOF
