﻿// 文字コード:UTF-8
/// @file
#pragma once

#include "lib/DynamicArray.hpp"
#include "lib/DynamicScopedPtrArray.hpp"

//------------------------------------------------------------------------------
namespace lib {

/// オブジェクトプール
/// ・動的なメモリ確保を行わずにオブジェクトを使いまわすのに使う
/// ・オブジェクトのコンストラクタ、デストラクタの代わりの関数は「initialize/finalize」を推奨する
/// ・所有権は、貸し出し中も「ObjectPool」で管理され、返し忘れがあってもデストラクタで「delete」される
template<typename TType>
class ObjectPool
    : private ::lib::NonCopyable
{
public:
    ObjectPool();
    ObjectPool(int aCap);
    ~ObjectPool();
    void reset();
    void reset(int aCap);
    /// 追加する
    /// ・使い始める前に、容量に対応する回数で、「new」したオブジェクトを渡すことを想定している
    /// ・追加したオブジェクトはデストラクタもしくは「reset」で「delete」される
    void add(TType* aObject);
    /// 借りる、在庫が無い場合は「nullptr」が返る
    TType* acquire();
    /// 返却する
    void release(TType* aObject);
private:
    /// 貸し出しを開始できる状態か？
    bool isReady() const { return mObjects.isFull(); }
    ::lib::DynamicScopedPtrArray<TType> mObjects;
    ::lib::DynamicArray<TType*> mPooledObjects;
};

//------------------------------------------------------------------------------
template<typename TType>
ObjectPool<TType>::ObjectPool()
: mObjects()
, mPooledObjects()
{}

//------------------------------------------------------------------------------
template<typename TType>
ObjectPool<TType>::ObjectPool(int aCap)
: ObjectPool()
{
    reset(aCap);
}

//------------------------------------------------------------------------------
template<typename TType>
ObjectPool<TType>::~ObjectPool()
{
}

//------------------------------------------------------------------------------
template<typename TType>
void ObjectPool<TType>::reset()
{
    mPooledObjects.reset();
    mObjects.reset();
}

//------------------------------------------------------------------------------
template<typename TType>
void ObjectPool<TType>::reset(int aCap)
{
    mObjects.reset(aCap);
    mPooledObjects.reset(aCap);
}

//------------------------------------------------------------------------------
template<typename TType>
void ObjectPool<TType>::add(TType* aObject)
{
    SYS_ASSERT_POINTER(aObject);
    SYS_ASSERT(!isReady());
    SYS_ASSERT(mObjects.count() == mPooledObjects.count());
    mObjects.emplace(aObject);
    mPooledObjects.push(aObject);
}

//------------------------------------------------------------------------------
template<typename TType>
TType* ObjectPool<TType>::acquire()
{
    SYS_ASSERT(isReady());
    if (mPooledObjects.isEmpty()) {
        return nullptr;
    }
    return mPooledObjects.pop();
}

//------------------------------------------------------------------------------
template<typename TType>
void ObjectPool<TType>::release(TType* aObject)
{
    SYS_ASSERT_POINTER(aObject);
    SYS_ASSERT(isReady());
    SYS_DEBUG_ASSERT(!mPooledObjects.contains(aObject));
    mPooledObjects.push(aObject);
}


} // namespace

// EOF
