﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <lib/Type.hpp>
#include <lib/debug/Assert.hpp>
#include <lib/NonCopyable.hpp>

//------------------------------------------------------------------------------
namespace lib {

/// @addtogroup LIB-Collection
//@{
/// @brief 最大要素数を指定する可変長配列クラス。
/// @details
/// std::vectorと違い、配列を内部に持つので
/// このクラスからalloc,freeが呼ばれることはない。
///
/// 例：10個までFrameCounterを格納できるMutableArray
/// @code
/// ::lib::MutableArray<FrameCounter, 10> arr;
/// @endcode
///
/// 機能追加などは自由にしていただいて構いません。
template <typename TElement, int TCount>
class MutableArray
{
public:
    typedef ::lib::Array<TElement, TCount> ArrayType;
    typedef typename ArrayType::iterator iterator;
    typedef typename ArrayType::const_iterator const_iterator;

    static const int SIZE = TCount; ///< 最大配列長。
    static const int NOT_FOUND = 0xFFFFFFFF;

    inline MutableArray();

    /// 末尾に追加する。
    inline void add(const TElement& aData);
    /// 指定位置に追加する。
    inline void insert(const int aIndex, const TElement& aData);
    /// 最初に見つかった指定のオブジェクトを除外する。
    inline void remove(const TElement& aData);
    /// 指定のインデックスのデータを除外する。
    inline void removeAtIndex(const int aIndex);
    /// 全てのオブジェクトを除外する。
    inline void clear();

    /// @name データを取得する。
    //@{
    inline const TElement& at(const int aIndex) const;
    inline TElement& at(const int aIndex);
    //@}

    /// 最初に見つかった指定のオブジェクトのインデックスを取得する。
    inline int index(const TElement& aData) const;
    /// 最初に見つかった指定のオブジェクトのインデックスを取得する。見つからなかったらNOT_FOUNDを返す。
    inline int indexLoose(const TElement& aData) const;
    /// 指定のオブジェクトが既に存在するか。
    inline bool isExist(const TElement& aData) const;

    /// @name 要素数を返す。
    //@{
    inline int count() const;
    inline int length() const;
    //@}

    /// 要素は１つもないか。
    inline bool isEmpty() const;
    /// 要素は全部埋まっているか。
    inline bool isFull() const;

    /// @name イテレータを返す。
    //@{
    inline typename ArrayType::iterator begin();
    inline typename ArrayType::iterator end();
    inline typename ArrayType::const_iterator begin() const;
    inline typename ArrayType::const_iterator end() const;
    //@}

    /// @name 配列アクセス。
    //@{
    inline const TElement& operator [](const int aIndex) const;
    inline TElement& operator [](const int aIndex);
    //@}

private:
    int mCount;
    ArrayType mCountainer;
};
//@}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
MutableArray<TElement, TCount>::MutableArray()
: mCount(0)
, mCountainer()
{
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
void MutableArray<TElement, TCount>::add(const TElement& aData)
{
    // チェック
    if (isFull()) {
        SYS_ASSERT_NOT_REACHED();
        return; // fail safe code
    }

    // 追加
    mCountainer[mCount] = aData;
    ++mCount;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
void MutableArray<TElement, TCount>::insert(const int aIndex, const TElement& aData)
{
    // チェック
    if (isFull() || aIndex > count()) {
        SYS_ASSERT_NOT_REACHED();
        return; // fail safe code
    }

    // 挿入
    SYS_ASSERT(aIndex >= 0);
    for (int i = count(); i > aIndex; --i) {
        mCountainer[i] = mCountainer[i - 1];
    }
    mCountainer[aIndex] = aData;
    ++mCount;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
void MutableArray<TElement, TCount>::remove(const TElement& aData)
{
    removeAtIndex(index(aData));
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
void MutableArray<TElement, TCount>::removeAtIndex(const int aIndex)
{
    // チェック
    if (count() <= aIndex) {
        SYS_ASSERT_NOT_REACHED_MSG("index '%d' count '%d'.", aIndex, count());
        return; // fail safe code
    }

    // 削除
    SYS_ASSERT(aIndex >= 0);
    for (int i = aIndex + 1; i < count(); ++i) {
        mCountainer[i - 1] = mCountainer[i];
    }
    --mCount;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
void MutableArray<TElement, TCount>::clear()
{
    mCount = 0;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
const TElement& MutableArray<TElement, TCount>::at(const int aIndex) const
{
    ::lib::ArrayUtil::RangeCheck(aIndex, mCount);
    return mCountainer[aIndex];
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
TElement& MutableArray<TElement, TCount>::at(const int aIndex)
{
    ::lib::ArrayUtil::RangeCheck(aIndex, mCount);
    return mCountainer[aIndex];
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
int MutableArray<TElement, TCount>::index(const TElement& aData) const
{
    const int idx = indexLoose(aData);
    SYS_ASSERT(idx != NOT_FOUND);
    return idx;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
int MutableArray<TElement, TCount>::indexLoose(const TElement& aData) const
{
    for (int i = 0; i < count(); ++i) {
        if (aData == at(i)) {
            return i;
        }
    }
    return NOT_FOUND;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
bool MutableArray<TElement, TCount>::isExist(const TElement& aData) const
{
    return indexLoose(aData) != NOT_FOUND;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
int MutableArray<TElement, TCount>::count() const
{
    return mCount;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
int MutableArray<TElement, TCount>::length() const
{
    return count();
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
bool MutableArray<TElement, TCount>::isEmpty() const
{
    return mCount == 0;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
bool MutableArray<TElement, TCount>::isFull() const
{
    return mCount == SIZE;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
typename MutableArray<TElement, TCount>::ArrayType::iterator MutableArray<TElement, TCount>::begin()
{
    return mCountainer.begin();
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
typename MutableArray<TElement, TCount>::ArrayType::iterator MutableArray<TElement, TCount>::end()
{
    return mCountainer.begin() + static_cast<int>(mCount);
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
typename MutableArray<TElement, TCount>::ArrayType::const_iterator MutableArray<TElement, TCount>::begin() const
{
    return mCountainer.begin();
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
typename MutableArray<TElement, TCount>::ArrayType::const_iterator MutableArray<TElement, TCount>::end() const
{
    return mCountainer.begin() + static_cast<int>(mCount);
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
const TElement& MutableArray<TElement, TCount>::operator [] (const int aIndex) const
{
    return at(aIndex);
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
TElement& MutableArray<TElement, TCount>::operator [] (const int aIndex)
{
    return at(aIndex);
}

} // namespace
// EOF
