﻿// 文字コード:UTF-8
/// @file
#pragma once

#include "lib/Vector3.hpp"

namespace lib {

class Matrix4x4
{
public:
    inline static Matrix4x4 Identity();
    inline static Matrix4x4 Translate(const Vector3& aVec);
    inline static Matrix4x4 Scale(const Vector3& aVec);
    // ・単位行列で生成される
    inline Matrix4x4();
    inline Matrix4x4(
        float a00, float a01, float a02, float a03,
        float a10, float a11, float a12, float a13,
        float a20, float a21, float a22, float a23,
        float a30, float a31, float a32, float a33
        );
    ~Matrix4x4() {}
    inline Matrix4x4& operator*=(const Matrix4x4& aMtx);
    inline const Matrix4x4 operator*(const Matrix4x4& aMtx) const;
    void set(int aCol, int aRow, float aVal) { m[aCol][aRow] = aVal; }
    void setT(const Vector3& aVec) { m[3][0] = aVec.x; m[3][1] = aVec.y; m[3][2] = aVec.z; }
    Vector3 getT() const { return Vector3(m[3][0], m[3][1], m[3][2]); }
    inline bool isTranslateOnly() const;
    float at(int aCol, int aRow) const { return m[aCol][aRow]; }
private:
    float m[4][4];
};

//------------------------------------------------------------------------------
Matrix4x4 Matrix4x4::Identity()
{
    return Matrix4x4(
        1.0f, 0.0f, 0.0f, 0.0f,
        0.0f, 1.0f, 0.0f, 0.0f,
        0.0f, 0.0f, 1.0f, 0.0f,
        0.0f, 0.0f, 0.0f, 1.0f);
}

//------------------------------------------------------------------------------
Matrix4x4 Matrix4x4::Translate(const Vector3& aVec)
{
    return Matrix4x4(
        1.0f, 0.0f, 0.0f, 0.0f,
        0.0f, 1.0f, 0.0f, 0.0f,
        0.0f, 0.0f, 1.0f, 0.0f,
        aVec.x, aVec.y, aVec.z, 1.0f);
}

//------------------------------------------------------------------------------
Matrix4x4 Matrix4x4::Scale(const Vector3& aVec)
{
    return Matrix4x4(
        aVec.x, 0.0f, 0.0f, 0.0f,
        0.0f, aVec.y, 0.0f, 0.0f,
        0.0f, 0.0f, aVec.z, 0.0f,
        0.0f, 0.0f, 0.0f, 1.0f);
}

//------------------------------------------------------------------------------
Matrix4x4::Matrix4x4()
{
    *this = Identity();
}

//------------------------------------------------------------------------------
Matrix4x4::Matrix4x4(
    float a00, float a01, float a02, float a03,
    float a10, float a11, float a12, float a13,
    float a20, float a21, float a22, float a23,
    float a30, float a31, float a32, float a33)
{
    m[0][0] = a00; m[0][1] = a01; m[0][2] = a02; m[0][3] = a03;
    m[1][0] = a10; m[1][1] = a11; m[1][2] = a12; m[1][3] = a13;
    m[2][0] = a20; m[2][1] = a21; m[2][2] = a22; m[2][3] = a23;
    m[3][0] = a30; m[3][1] = a31; m[3][2] = a32; m[3][3] = a33;
}

//------------------------------------------------------------------------------
Matrix4x4& Matrix4x4::operator*=(const Matrix4x4& aMtx)
{
    *this = *this * aMtx;
    return *this;
}

//------------------------------------------------------------------------------
const Matrix4x4 Matrix4x4::operator*(const Matrix4x4& aMtx) const
{
    Matrix4x4 mtx;
    mtx.m[0][0] = m[0][0] * aMtx.m[0][0] + m[0][1] * aMtx.m[1][0] + m[0][2] * aMtx.m[2][0] + m[0][3] * aMtx.m[3][0];
    mtx.m[0][1] = m[0][0] * aMtx.m[0][1] + m[0][1] * aMtx.m[1][1] + m[0][2] * aMtx.m[2][1] + m[0][3] * aMtx.m[3][1];
    mtx.m[0][2] = m[0][0] * aMtx.m[0][2] + m[0][1] * aMtx.m[1][2] + m[0][2] * aMtx.m[2][2] + m[0][3] * aMtx.m[3][2];
    mtx.m[0][3] = m[0][0] * aMtx.m[0][3] + m[0][1] * aMtx.m[1][3] + m[0][2] * aMtx.m[2][3] + m[0][3] * aMtx.m[3][3];

    mtx.m[1][0] = m[1][0] * aMtx.m[0][0] + m[1][1] * aMtx.m[1][0] + m[1][2] * aMtx.m[2][0] + m[1][3] * aMtx.m[3][0];
    mtx.m[1][1] = m[1][0] * aMtx.m[0][1] + m[1][1] * aMtx.m[1][1] + m[1][2] * aMtx.m[2][1] + m[1][3] * aMtx.m[3][1];
    mtx.m[1][2] = m[1][0] * aMtx.m[0][2] + m[1][1] * aMtx.m[1][2] + m[1][2] * aMtx.m[2][2] + m[1][3] * aMtx.m[3][2];
    mtx.m[1][3] = m[1][0] * aMtx.m[0][3] + m[1][1] * aMtx.m[1][3] + m[1][2] * aMtx.m[2][3] + m[1][3] * aMtx.m[3][3];

    mtx.m[2][0] = m[2][0] * aMtx.m[0][0] + m[2][1] * aMtx.m[1][0] + m[2][2] * aMtx.m[2][0] + m[2][3] * aMtx.m[3][0];
    mtx.m[2][1] = m[2][0] * aMtx.m[0][1] + m[2][1] * aMtx.m[1][1] + m[2][2] * aMtx.m[2][1] + m[2][3] * aMtx.m[3][1];
    mtx.m[2][2] = m[2][0] * aMtx.m[0][2] + m[2][1] * aMtx.m[1][2] + m[2][2] * aMtx.m[2][2] + m[2][3] * aMtx.m[3][2];
    mtx.m[2][3] = m[2][0] * aMtx.m[0][3] + m[2][1] * aMtx.m[1][3] + m[2][2] * aMtx.m[2][3] + m[2][3] * aMtx.m[3][3];

    mtx.m[3][0] = m[3][0] * aMtx.m[0][0] + m[3][1] * aMtx.m[1][0] + m[3][2] * aMtx.m[2][0] + m[3][3] * aMtx.m[3][0];
    mtx.m[3][1] = m[3][0] * aMtx.m[0][1] + m[3][1] * aMtx.m[1][1] + m[3][2] * aMtx.m[2][1] + m[3][3] * aMtx.m[3][1];
    mtx.m[3][2] = m[3][0] * aMtx.m[0][2] + m[3][1] * aMtx.m[1][2] + m[3][2] * aMtx.m[2][2] + m[3][3] * aMtx.m[3][2];
    mtx.m[3][3] = m[3][0] * aMtx.m[0][3] + m[3][1] * aMtx.m[1][3] + m[3][2] * aMtx.m[2][3] + m[3][3] * aMtx.m[3][3];
    return mtx;
}

//------------------------------------------------------------------------------
bool Matrix4x4::isTranslateOnly() const
{
    return
        m[0][0] == 1.0f && m[0][1] == 0.0f && m[0][2] == 0.0f && m[0][3] == 0.0f &&
        m[1][0] == 0.0f && m[1][1] == 1.0f && m[1][2] == 0.0f && m[1][3] == 0.0f &&
        m[2][0] == 0.0f && m[2][1] == 0.0f && m[2][2] == 1.0f && m[2][3] == 0.0f &&
        m[3][3] == 1.0f;
}

} // namespace

// EOF

