﻿// 文字コード:UTF-8
/// @file
#include "lib/JsonValue.hpp"

#include "lib/JsonReader.hpp"

//------------------------------------------------------------------------------
namespace lib {

namespace {

const auto fKeyStringCapacity = 128;

}

//------------------------------------------------------------------------------
JsonValue::JsonValue(const ::lib::String& aString)
: JsonValue(aString.buffer(), aString.getLength())
{
}

//------------------------------------------------------------------------------
JsonValue::JsonValue(const char* aString, int aStringLength)
: JsonValue(aString, aString + aStringLength)
{
}

//------------------------------------------------------------------------------
JsonValue::JsonValue()
: JsonValue(nullptr, nullptr)
{
}

//------------------------------------------------------------------------------
JsonValue::~JsonValue()
{
}

//------------------------------------------------------------------------------
bool JsonValue::isNull() const
{
    // @note readValue などで正しい値かチェックできるとより直感的な結果になるかもしれないが、
    //       パース処理の負荷がかかるためひとまずやらない
    return mData == nullptr;
}

//------------------------------------------------------------------------------
Nullable<bool> JsonValue::toBoolOrNull() const
{
    if (isNull()) {
        return {};
    }
    auto value = bool();
    if (!getReader().readBool(&value)) {
        return {};
    }
    return value;
}

//------------------------------------------------------------------------------
Nullable<int> JsonValue::toIntOrNull() const
{
    if (isNull()) {
        return {};
    }
    auto value = int();
    if (!getReader().readInt(&value)) {
        return {};
    }
    return value;
}

//------------------------------------------------------------------------------
Nullable<float> JsonValue::toFloatOrNull() const
{
    if (isNull()) {
        return {};
    }
    auto value = float();
    if (!getReader().readFloat(&value)) {
        return {};
    }
    return value;
}

//------------------------------------------------------------------------------
Nullable<String> JsonValue::toStringOrNull(int aCapacity) const
{
    if (isNull()) {
        return {};
    }
    auto value = String(aCapacity);
    if (!getReader().readString(&value.at(0), value.capacity())) {
        return {};
    }
    return value;
}

//------------------------------------------------------------------------------
JsonValue JsonValue::operator[](const char* aKey) const
{
    auto reader = getReader();
    // オブジェクトではなかった
    if (!reader.readObjectOpening()) {
        return JsonValue();
    }

    // キーを読む
    auto key = String(fKeyStringCapacity);
    while (reader.readKey(&key.at(0), key.capacity())) {
        // 見つかったらその位置を始点としたデータを結果とする
        if (key.equals(aKey)) {
            return JsonValue(reader.currentPosition(), mDataEnd);
        }
        // 次のキーまで進める
        if (!reader.readValue()) {
            return JsonValue();
        }
    }
    return JsonValue();
}

//------------------------------------------------------------------------------
JsonValue::Iterator JsonValue::begin()
{
    auto reader = getReader();
    if (reader.readArrayOpening()) {
        return JsonValue::Iterator(JsonValue(reader.currentPosition(), mDataEnd));
    }
    return JsonValue::Iterator(JsonValue());
}

//------------------------------------------------------------------------------
JsonValue::Iterator JsonValue::begin() const
{
    return const_cast<JsonValue*>(this)->begin();
}

//------------------------------------------------------------------------------
JsonValue::Iterator JsonValue::end()
{
    return JsonValue::Iterator(JsonValue());
}

//------------------------------------------------------------------------------
JsonValue::Iterator JsonValue::end() const
{
    return JsonValue::Iterator(JsonValue());
}

//------------------------------------------------------------------------------
JsonReader JsonValue::getReader() const
{
    return JsonReader(mData, mDataEnd - mData);
}

//------------------------------------------------------------------------------
void JsonValue::Iterator::operator++()
{
    auto reader = mJsonValue.getReader();
    if (!reader.readValue()) {
        mJsonValue = JsonValue();
        return;
    }
    auto position = reader.currentPosition();
    if (reader.readArrayClosing()) {
        mJsonValue = JsonValue();
    }
    else {
        mJsonValue = JsonValue(position, mJsonValue.mDataEnd);
    }
}

} // namespace
// EOF
