﻿// 文字コード:UTF-8
/// @file

#pragma once

#include "lib/IntrusiveList.hpp"

//------------------------------------------------------------------------------
namespace lib {

/// 侵入ツリーノード、子は侵入リストで管理される
template<class TType>
class IntrusiveTreeNode
{
public:
    typedef IntrusiveTreeNode<TType> This;
    IntrusiveTreeNode(TType* aOwner);
    ~IntrusiveTreeNode() { removeSelfFromParentIfPossible(); SYS_ASSERT_MSG(isEmpty(), "子が削除されていない状態で自分が削除されました"); }
    TType& owner() { return mOwner; }
    const TType& owner() const { return mOwner; }
    bool hasParent() const { return mParent != nullptr; }
    This& parent() { SYS_ASSERT(hasParent()); return *mParent; }
    const This& parent() const { SYS_ASSERT(hasParent()); return *mParent; }
    This* parentPtr() { return mParent; }
    const This* parentPtr() const { return mParent; }
    ::lib::IntrusiveList<This>& childNodes() { return mChildNodes; }
    const ::lib::IntrusiveList<This>& childNodes() const { return mChildNodes; }
    bool hasNextSibling() const { return mListNode.hasNext(); }
    This& nextSibling() { return mListNode.next().owner(); }
    const This& nextSibling() const { return const_cast<This*>(this)->nextSibling(); }
    This* nextSiblingPtr() { return (hasNextSibling() ? &nextSibling() : 0); }
    const This* nextSiblingPtr() const { return (hasNextSibling() ? &nextSibling() : 0); }
    bool hasPreviousSibling() const { return mListNode.hasPrevious(); }
    This& previousSibling() { return mListNode.previous().owner(); }
    const This& previousSibling() const { return const_cast<This*>(this)->previousSibling(); }
    This* previousSiblingPtr() { return (hasPreviousSibling() ? &previousSibling() : 0); }
    const This* previousSiblingPtr() const { return (hasPreviousSibling() ? &previousSibling() : 0); }
    This& firstChild() { return mChildNodes.first().owner(); }
    const This& firstChild() const { return const_cast<This*>(this)->firstChild(); }
    This* firstChildPtr() { return (isEmpty() ? 0 : &mChildNodes.first().owner()); }
    const This* firstChildPtr() const { return (isEmpty() ? 0 : &mChildNodes.first().owner()); }
    This& lastChild() { return mChildNodes.last().owner(); }
    const This& lastChild() const { return const_cast<This*>(this)->lastChild(); }
    This* lastChildPtr() { return (isEmpty() ? 0 : &mChildNodes.last().owner()); }
    const This* lastChildPtr() const { return (isEmpty() ? 0 : &mChildNodes.last().owner()); }
    bool isEmpty() const { return mChildNodes.isEmpty(); }
    int getChildCount() const { return mChildNodes.getCount(); }
    This& getChildAt(int aIndex) { return mChildNodes.getAt(aIndex).owner(); }
    const This& getChildAt(int aIndex) const { return const_cast<This*>(this)->getChildAt(aIndex); }

    void appendChild(This* aNode);
    void prependChild(This* aNode);
    void insertAfterSelf(This* aNode);
    void insertBeforeSelf(This* aNode);
    void removeSelfFromParent();
    void removeSelfFromParentIfPossible();
private:
    TType& mOwner;
    This* mParent;
    ::lib::IntrusiveListNode<This> mListNode;
    ::lib::IntrusiveList<This> mChildNodes;
};

//------------------------------------------------------------------------------
template <class TType> IntrusiveTreeNode<TType>::IntrusiveTreeNode(TType* aOwner)
: mOwner(*aOwner)
, mParent(0)
, mListNode(this)
{
    SYS_ASSERT_POINTER(aOwner);
}

//------------------------------------------------------------------------------
template <class TType> void IntrusiveTreeNode<TType>::appendChild(This* aNode)
{
    SYS_ASSERT_POINTER(aNode);
    SYS_ASSERT(!aNode->hasParent());
    mChildNodes.append(&aNode->mListNode);
    aNode->mParent = this;
}

//------------------------------------------------------------------------------
template <class TType> void IntrusiveTreeNode<TType>::prependChild(This* aNode)
{
    SYS_ASSERT_POINTER(aNode);
    SYS_ASSERT(!aNode->hasParent());
    mChildNodes.append(aNode->mListNode);
    aNode->mParent = this;
}

//------------------------------------------------------------------------------
template <class TType> void IntrusiveTreeNode<TType>::insertAfterSelf(This* aNode)
{
    SYS_ASSERT_POINTER(aNode);
    SYS_ASSERT(mParent);
    aNode->mParent = parent();
    mListNode.insertAfterSelf(aNode);
}

//------------------------------------------------------------------------------
template <class TType> void IntrusiveTreeNode<TType>::insertBeforeSelf(This* aNode)
{
    SYS_ASSERT_POINTER(aNode);
    SYS_ASSERT(hasParent());
    aNode->mParent = mParent;
    mListNode.insertBeforeSelf(aNode);
}

//------------------------------------------------------------------------------
template <class TType> void IntrusiveTreeNode<TType>::removeSelfFromParent()
{
    SYS_ASSERT(hasParent());
    mListNode.removeSelf();
    mParent = 0;
}

//------------------------------------------------------------------------------
template <class TType> void IntrusiveTreeNode<TType>::removeSelfFromParentIfPossible()
{
    if (hasParent()) {
        removeSelfFromParent();
    }
}

} // namespace
// EOF
