﻿// 文字コード:UTF-8
/// @file
#pragma once

#include "lib/Math.hpp"

namespace lib {
/// 補間(Interpolation)関係の関数群。
/// @details
/// 以前はMathクラスで定義されていたもの。
class InterpUtil
    : private ::lib::NonCreatable
{
public:
    /// 直線補間パラメータ(0.0～1.0)を
    /// 「両端の傾き０補間」パラメータに変換する
    /// @details
    /// <pre>
    /// 1.0|                        ___---
    ///    |                    __--
    ///    |                  _-
    ///    |                _-
    ///    |              _-
    ///    |            _-
    ///    |          _-
    ///    |      __--
    ///    |___---
    /// 0.0+------------------------------
    ///    0.0                          1.0
    /// </pre>
    static inline float HermiteMinToMin(float aValue0to1);
    /// 直線補間パラメータ(0.0～1.0)を
    /// 「開始端の傾き０終端傾き最大補間」パラメータに変換する
    /// @details
    /// <pre>
    /// 1.0|                            _-
    ///    |                          _-
    ///    |                        _-
    ///    |                      _-
    ///    |                    _-
    ///    |                  _-
    ///    |              __--
    ///    |        ___---
    ///    |____----
    /// 0.0+------------------------------
    ///    0.0                          1.0
    /// </pre>
    static inline float HermiteMinToMax(float aValue0to1);
    /// 直線補間パラメータ(0.0～1.0)を
    /// 「開始端の傾き最大終端傾き０補間」パラメータに変換する
    /// @details
    /// <pre>
    /// 1.0|                      ____----
    ///    |                ___---
    ///    |            __--
    ///    |          _-
    ///    |        _-
    ///    |      _-
    ///    |    _-
    ///    |  _-
    ///    |_-
    /// 0.0+------------------------------
    ///    0.0                          1.0
    /// </pre>
    static inline float HermiteMaxToMin(float aValue0to1);
    /// 線形。入力の値をそのまま返すだけ。引数は[0.0f, 1.0f]の範囲。
    static inline float Linear(float aValue0to1);
    /// 二次曲線。最後のほうでぐっとあがる。引数は[0.0f, 1.0f]の範囲。
    static inline float Square(float aValue0to1);
    /// 二次曲線を上下左右反転したもの。最初のほうでぐっとあがる。引数は[0.0f, 1.0f]の範囲。
    static inline float InvSquare(float aValue0to1);
    /// aValue0to1 = 0.5でピーク値1.0を取り、aValue0to1 = 0 及び 1 で 0 をとるような山型。
    /// 三角波。引数は[0.0f, 1.0f]の範囲。
    static inline float Triangle(float aValue0to1);
    /// 2つの値の線形補間値を得る
    template <class T> static inline const T Linear(const T& aValueAs0,
        const T& aValueAs1, float aValue);
    /// 2つの値の中間値valがどの位の割合の位置にあるかを得る
    static inline float LinearRatio(float aValueAs0, float aValueAs1, float aValue);
    /// 2つの値の中間値valがどの位の割合の位置にあるかを得る
    /// (0.0～1.0間のクランプ機能付き)
    static inline float LinearRatioClamp(float aValueAs0, float aValueAs1,
        float aValue);
    /// マッピング
    /// @details
    /// ある座標空間範囲の値を、他の座標空間範囲に置換する
    /// ある値の範囲を0.0～1.0の間に投影した後、線形補完をかけるという
    /// 定型処理を一つの関数で表現出来ます。
    ///
    /// テンプレートなので投影先の型は自由に変更出来ます。
    ///
    /// @param aValue 基準値、この値を元に0.0から1.0の値に投影される
    /// @param aCurrentMin 投影元、これ以下の値が入った場合はクランプされる
    /// @param aCurrentMax 投影元、これ以上の値が入った場合はクランプされる
    /// @param aTargetMin 投影先、最小値
    /// @param aTargetMax 投影先、最大値
    template <class T> static inline T Mapping(float aValue, float aCurrentMin,
        float aCurrentMax, const T& aTargetMin, const T& aTargetMax);
};

//------------------------------------------------------------------------------
float InterpUtil::HermiteMinToMin(const float aValue0to1)
{
    const float sqt = aValue0to1 * aValue0to1;
    return ((3.0f * sqt) - (2.0f * aValue0to1 * sqt));
}

//------------------------------------------------------------------------------
float InterpUtil::HermiteMinToMax(const float aValue0to1)
 {
    const float sqt = aValue0to1 * aValue0to1;
    return ((1.5f * sqt) - (0.5f * aValue0to1 * sqt));
}

//------------------------------------------------------------------------------
float InterpUtil::HermiteMaxToMin(const float aValue0to1)
{
    return ((-0.5f * aValue0to1 * aValue0to1 * aValue0to1)
        + (1.5f * aValue0to1));
}

//------------------------------------------------------------------------------
float InterpUtil::Linear(const float aValue0to1)
{
    return aValue0to1;
}

//------------------------------------------------------------------------------
float InterpUtil::Square(const float aValue0to1)
{
    return aValue0to1 * aValue0to1;
}

//------------------------------------------------------------------------------
float InterpUtil::InvSquare(const float aValue0to1)
{
    return 1.0f - Square(1.0f - aValue0to1);
}

//------------------------------------------------------------------------------
float InterpUtil::Triangle(const float aValue0to1)
{
    return -::lib::Math::Abs(2 * aValue0to1 - 1) + 1;
}

//------------------------------------------------------------------------------
template <class T> const T InterpUtil::Linear(const T& aValueAs0,
    const T& aValueAs1, const float aValue)
{
    return (aValueAs1 - aValueAs0) * aValue + aValueAs0;
}

//------------------------------------------------------------------------------
float InterpUtil::LinearRatio(const float aValueAs0, const float aValueAs1,
    const float aValue)
{
    return (aValue - aValueAs0) / (aValueAs1 - aValueAs0);
}

//------------------------------------------------------------------------------
float InterpUtil::LinearRatioClamp(const float aValueAs0, const float aValueAs1,
    const float aValue)
{
    return ::lib::Math::LimitMinMax(
        LinearRatio(aValueAs0, aValueAs1, aValue), 0.0f, 1.0f
        );
}

//------------------------------------------------------------------------------
template <class T> T InterpUtil::Mapping(const float aValue, const float aCurrentMin,
    const float aCurrentMax, const T& aTargetMin, const T& aTargetMax)
{
    const float ratio = LinearRatioClamp(aCurrentMin, aCurrentMax, aValue);
    return Linear(aTargetMin, aTargetMax, ratio);
}

} // namespace
// EOF
