﻿// 文字コード:UTF-8
/// @file
#pragma once

namespace lib {

/// 簡易版のboost::function
///
/// Function<bool(int)> func = [](int x) { return x == 0; };
/// func(1);
/// Function<void()> func2 = [this] { memberFunction(); };
/// func2();
///  のようにしてメンバ関数へのコールバックが行える

// @note tagoya (2015/10/16)
//     ラムダ式で int 型を 3 個キャプチャできるサイズにしている (sizeof(void*) + sizeof(int) * 3)
//     ただし、パディングの関係か 24 byte にしないと収まらないので 24 byte にしている
enum { FunctionImplBufferSize = 24 };

template<class FT> class FunctionImpl;
template<class TLambda, class FT> class LambdaFun;
// @memo VC2012では<class FT = void()> と記述してFunction<> を扱うとコンストラクタが未定義と判断されるため
// ひとまずデフォルト引数をやめてFunction<void()> と記述してもらうことに
template<class FT> class Function;

//------------------------------------------------------------------------------
template <class R, class... Args>
class FunctionImpl<R(Args...)>
{
public:
    virtual ~FunctionImpl<R(Args...)>() {}
    virtual FunctionImpl<R(Args...)>* clone(void*) const = 0;
    virtual R call(Args... args) const = 0;
    static void operator delete(void*) { SYS_ASSERT_NOT_REACHED(); }
    static void* operator new(size_t) = delete;
    static void operator delete(void*, void*) {}
    static void* operator new(size_t aSize, void* aBuf)
    {
        SYS_ASSERT(aSize <= FunctionImplBufferSize);
        return aBuf;
    }
};

//------------------------------------------------------------------------------
template <class TLambda, class R, class... Args>
class LambdaFun<TLambda, R(Args...)>
    : public FunctionImpl<R(Args...)>
{
public:
    LambdaFun(TLambda aLambdaFun)
    : mLambdaFun(aLambdaFun)
    {}
    virtual FunctionImpl<R(Args...)>* clone(void* aBuf) const
    {
        return new(aBuf)LambdaFun<TLambda, R(Args...)>(mLambdaFun);
    }
    virtual R call(Args... args) const { return mLambdaFun(args...); }

private:
    TLambda mLambdaFun;
};

//------------------------------------------------------------------------------
template <class R, class... Args>
class Function<R(Args...)>
{
public:
    Function()
    : mImpl(nullptr)
    {
    }
    /// @note tagoya (2016/03/23)
    ///   Function = nullptr で初期化できるようにするために Function(::std::nullptr_t) を用意。
    Function(::std::nullptr_t)
    : mImpl(nullptr)
    {
    }
    /// @note tagoya (2016/03/23)
    ///   Function = NULL や Function = 0 を許容しないために Function(int aPtr) は呼び出せないようにする。
    ///   何もしないと template <class TLambda> Function(const TLambda& aLambda) から Function(int) が実体化してしまうので、
    ///   明示的に関数を delete しておく
    Function(int aPtr) = delete;

    Function(const Function<R(Args...)>& aFunc)
    : mImpl(nullptr)
    {
        if (aFunc.mImpl) {
            mImpl = aFunc.mImpl->clone(mImplBuf);
        }
    }
    Function(const FunctionImpl<R(Args...)>& aFuncImpl)
    : mImpl(nullptr)
    {
        mImpl = aFuncImpl.clone(mImplBuf);
    }
    template <class TLambda>
    Function(const TLambda& aLambda)
    : mImpl(nullptr)
    {
        mImpl = LambdaFun<TLambda, R(Args...)>(aLambda).clone(mImplBuf);
    }
    //------------------------------------------------------------------------------
    ~Function()
    {
        clear();
    }
    //------------------------------------------------------------------------------
    void operator=(const Function<R(Args...)>& aFunc)
    {
        clear();
        mImpl = (aFunc.mImpl) ? aFunc.mImpl->clone(mImplBuf) : nullptr;
    }
    void operator=(const FunctionImpl<R(Args...)>& aFuncImpl)
    {
        clear();
        mImpl = aFuncImpl.clone(mImplBuf);
    }
    template <class TLambda>
    void operator=(const TLambda& aLambda)
    {
        clear();
        mImpl = LambdaFun<TLambda, R(Args...)>(aLambda).clone(mImplBuf);
    }
    //------------------------------------------------------------------------------
    void clear()
    {
        if (mImpl) {
            mImpl->~FunctionImpl<R(Args...)>();
            mImpl = nullptr;
        }
    }
    bool isValid() const { return mImpl != nullptr; }
    bool isEmpty() const { return mImpl == nullptr; }
    R call(Args... args) const { SYS_ASSERT(mImpl); return mImpl->call(args...); }
    R callIfValid(Args... args) const { return isValid() ? call(args...) : R(); }
    R operator()(Args... args) const { return call(args...); }

private:
    FunctionImpl<R(Args...)>* mImpl;
    uint8_t mImplBuf[FunctionImplBufferSize];
};

} // namespace
// EOF
