﻿// 文字コード:UTF-8
/// @file
#pragma once

#include "lib/NonCopyable.hpp"

//------------------------------------------------------------------------------
namespace lib {

//@{
/// 明示的に生成・破棄するシングルトンインターフェースクラス。
/// @details
/// このクラスを継承したクラスは
/// ２つ以上インスタンスを作成することができなくなります。
///
/// テンプレート引数に自分のクラスを指定してください。
template <typename TClass>
class ExplicitSingleton
    : private ::lib::NonCopyable
{
public:
    /// インスタンスを取得する。
    inline static TClass& Instance();

    /// インスタンスが存在するかどうか取得する。
    inline static bool Exists();

    /// インスタンスをセット
    ///
    /// @attention コンストラクタ内で必ず呼ぶこと
    /// @note 子クラスの初期化が終わった後でセットしたい
    inline static void SetInstance(TClass* aObject);

    /// インスタンスのアンセット
    ///
    /// @attention デストラクタ内で必ず呼ぶこと
    inline static void UnsetInstance();

protected:
    inline ExplicitSingleton();
    inline ~ExplicitSingleton();

private:
    static TClass* mObject;
};
//@}

template <typename TClass>
TClass* ExplicitSingleton<TClass>::mObject = 0;

//------------------------------------------------------------------------------
template <typename TClass>
ExplicitSingleton<TClass>::ExplicitSingleton()
{
    // オブジェクトは登録されていない必要がある。
    SYS_ASSERT(mObject == 0);
}

//------------------------------------------------------------------------------
template <typename TClass>
ExplicitSingleton<TClass>::~ExplicitSingleton()
{
    // オブジェクトは手動で登録解除する必要がある。
    SYS_ASSERT(!Exists());
}

//------------------------------------------------------------------------------
template <typename TClass>
TClass& ExplicitSingleton<TClass>::Instance()
{
    SYS_ASSERT(Exists());
    return *mObject;
}

//------------------------------------------------------------------------------
template <typename TClass>
bool ExplicitSingleton<TClass>::Exists()
{
    return mObject != 0;
}

//------------------------------------------------------------------------------
template <typename TClass>
void ExplicitSingleton<TClass>::SetInstance(TClass* aObject)
{
    SYS_ASSERT(!Exists());
    mObject = aObject;
}

//------------------------------------------------------------------------------
template <typename TClass>
void ExplicitSingleton<TClass>::UnsetInstance()
{
    SYS_ASSERT(Exists());
    mObject = 0;
}

} // namespace
// EOF
