﻿// 文字コード:UTF-8
/// @file
#pragma once

//------------------------------------------------------------------------------
#include <lib/Type.hpp>
#include <lib/ShortString.hpp>

//------------------------------------------------------------------------------
namespace lib {
    struct Vector4Pod;
}
namespace lib {
    struct Color4bPod;
}

//------------------------------------------------------------------------------
namespace lib {

//@{
/// 32bit浮動小数をRGBAの順に並べた色構造体。
struct Color4Pod
{
public:
    /// @name 変数
    //@{
    float r; ///< Red。
    float g; ///< Green。
    float b; ///< Blue。
    float a; ///< Alpha。
    //@}

    /// @name 定数取得
    //@{
    static const Color4Pod Zero(); ///< 0の色。
    static const Color4Pod One(); ///< R,G,B,Aが全て1の色。
    static const Color4Pod UnitR(); ///< Rだけ1の色。
    static const Color4Pod UnitG(); ///< Gだけ1の色。
    static const Color4Pod UnitB(); ///< Bだけ1の色。
    static const Color4Pod UnitA(); ///< Aだけ1の色。
    static const Color4Pod Black(); ///< 黒色(0, 0, 0, 1)。
    static const Color4Pod White(); ///< 白色(1, 1, 1, 1)。
    //@}

    /// @name 変換
    //@{
    const Color4Pod toRRRR() const; ///< R,G,B,Aを全てRの値に変換。
    const Color4Pod toGGGG() const; ///< R,G,B,Aを全てGの値に変換。
    const Color4Pod toBBBB() const; ///< R,G,B,Aを全てBの値に変換。
    const Color4Pod toAAAA() const; ///< R,G,B,Aを全てAの値に変換。
    const Color4bPod toRGBAb() const;
    const ::lib::Vector4Pod toVector4() const;
    //@}

    /// @name 等価比較
    //@{
    bool equals(const Color4Pod& aRHS) const; ///< 許容誤差を考慮した等価比較。
    bool equalsStrict(const Color4Pod& aRHS) const; ///< 許容誤差を許容しない等価比較。
    //@}

    /// @name 四則演算
    //@{
    const Color4Pod add(float aVal) const; ///< 各要素に値を加算した結果を取得する。
    const Color4Pod sub(float aVal) const; ///< 各要素から値を減算した結果を取得する。
    const Color4Pod mul(float aVal) const; ///< 各要素に値をかけた結果を取得する。
    const Color4Pod div(float aVal) const; ///< 各要素から値をわった結果を取得する。
    const Color4Pod add(const Color4Pod& aVal) const; ///< 対応する値同士を加算した結果を取得する。
    const Color4Pod sub(const Color4Pod& aVal) const; ///< 対応する値に対して減算した結果を取得する。
    const Color4Pod mul(const Color4Pod& aVal) const; ///< 対応する値同士をかけた結果を取得する。
    const Color4Pod div(const Color4Pod& aVal) const; ///< 対応する値に対して割り算をした結果を取得する。
    const Color4Pod neg() const; ///< 符号を反転させた結果を取得する。
    void addAssign(float aVal); ///< 各要素に値を加算する。
    void subAssign(float aVal); ///< 各要素から値を減算する。
    void mulAssign(float aVal); ///< 各要素に値をかける。
    void divAssign(float aVal); ///< 各要素から値をわる。
    void addAssign(const Color4Pod& aVal); ///< 対応する値同士を加算する。
    void subAssign(const Color4Pod& aVal); ///< 対応する値に対して減算する。
    void mulAssign(const Color4Pod& aVal); ///< 対応する値同士をかける。
    void divAssign(const Color4Pod& aVal); ///< 対応する値に対して割り算をする。
    //@}

    /// @name 演算子オーバーロード
    //@{
    const Color4Pod operator+(float aVal) const; ///< add()。
    const Color4Pod operator-(float aVal) const; ///< sub()。
    const Color4Pod operator*(float aVal) const; ///< mul()。
    const Color4Pod operator/(float aVal) const; ///< div()。
    const Color4Pod operator+(const Color4Pod& aVal) const; ///< add()。
    const Color4Pod operator-(const Color4Pod& aVal) const; ///< sub()。
    const Color4Pod operator*(const Color4Pod& aVal) const; ///< mul()。
    const Color4Pod operator/(const Color4Pod& aVal) const; ///< div()。
    const Color4Pod operator-() const; ///< negate()。
    Color4Pod& operator+=(float aVal); ///< addAssign()。
    Color4Pod& operator-=(float aVal); ///< subAssign()。
    Color4Pod& operator*=(float aVal); ///< mulAssign()。
    Color4Pod& operator/=(float aVal); ///< divAssign()。
    Color4Pod& operator+=(const Color4Pod& aVal); ///< addAssign()。
    Color4Pod& operator-=(const Color4Pod& aVal); ///< subAssign()。
    Color4Pod& operator*=(const Color4Pod& aVal); ///< mulAssign()。
    Color4Pod& operator/=(const Color4Pod& aVal); ///< divAssign()。
    //@}

    /// @name ユーティリティ
    //@{
    /// "R,G,B,A"形式の文字列に変換。
    const ::lib::ShortString toShortString() const;
    //@}
};
//@}

} // namespace
// EOF
