﻿// 文字コード:UTF-8
/// @file
/// @author nakano
#pragma once

#include <lib/debug/Assert.hpp>
#include <lib/Array.hpp>
#include <lib/DeleteObject.hpp>
#include <lib/NonCopyable.hpp>
#include <memory> // for ::std::auto_ptr

//----------------------------------------------------------
namespace lib {

/// @addtogroup LIB-Collection
//@{
/// デストラクタで全要素をdeleteする可変長ポインタ配列クラス。
/// @tparam TType 格納する型（ポインタ記号は不要）
/// @tparam N 配列の最大長。
///
/// @details
/// PointerArrayとの違い@n
/// ・Nullのポインタの格納を許可しない@n
/// ・↑のため、配列のサイズ取得に対する処理コストが安い。
///
/// 例 : JObjのポインタを10個まで扱えるAutoDeleteArray
/// @code
/// AutoDeleteArray<
///   ::lib::JObj // ポインタ記号は不要
///   , 10 // 10個
///   > arr;
/// @endcode
///
/// 機能追加などは自由にしていただいて構いません。
template <typename TType, int N>
class AutoDeleteArray
    : public ::lib::NonCopyable
{
public:
    static const int MaxSize = N; ///< 最大配列長。

    typedef ::lib::Array<TType*, N> ArrayType;

    /// コンストラクタ。
    inline AutoDeleteArray();
    /// デストラクタ。全要素をdeleteする。
    inline ~AutoDeleteArray();

    /// 全削除
    inline void clear();

    /// 末尾に追加する。（破棄責任をこのクラスに委譲します）
    inline void add(TType* aData);
    /// 末尾に追加する。(std::auto_ptrバージョン）
    inline void add(::std::auto_ptr<TType> aData);

    /// @name index番目のデータを取得する。
    //@{
    inline const TType& at(const int aIndex) const;
    inline TType& at(const int aIndex);
    //@}

    /// @name 最後のデータを取得する。
    //@{
    const TType& back() const { return at(count() - 1); }
    TType& back() { return at(count() - 1); }
    //@}

    /// 最初に見つかった指定のオブジェクトのインデックスを取得する。
    inline int index(const TType& aData) const;

    /// @name 長さを返す。
    //@{
    int count() const { return mCount; }
    //@}

    /// 要素は１つもないか。
    bool isEmpty() const { return mCount == 0; }

    /// 要素は全部埋まっているか。
    bool isFull() const { return mCount == MaxSize; }

    /// @name イテレータを返す。
    //@{
    typename ArrayType::iterator begin() { return mContainer.begin(); }
    typename ArrayType::iterator end() { return mContainer.elems + mCount; }
    typename ArrayType::const_iterator begin() const { return mContainer.begin(); }
    typename ArrayType::const_iterator end() const { return mContainer.elems + mCount; }
    //@}

    /// @name 配列アクセス。
    //@{
    const TType& operator [] (const int index) const { return at(index); }
    TType& operator [] (const int index) { return at(index); }
    //@}

private:
    int mCount;
    ArrayType mContainer;
};
//@}

//------------------------------------------------------------------------------
template <typename TType, int N>
AutoDeleteArray<TType, N>::AutoDeleteArray()
: mCount(0)
, mContainer()
{
}

//------------------------------------------------------------------------------
template <typename TType, int N>
AutoDeleteArray<TType, N>::~AutoDeleteArray()
{
    clear();
}

//------------------------------------------------------------------------------
template <typename TType, int N>
void AutoDeleteArray<TType, N>::clear()
{
    while (0 < mCount) {
        const uint32_t index = mCount - 1;
        TType* ptr = mContainer[index];
        mContainer[index] = 0;
        --mCount;

        delete ptr;
    }
}

//------------------------------------------------------------------------------
template <typename TType, int N>
void AutoDeleteArray<TType, N>::add(TType* aData)
{
    SYS_ASSERT(count() < mContainer.count());
    mContainer[mCount] = aData;
    ++mCount;
}

//------------------------------------------------------------------------------
template <typename TType, int N>
void AutoDeleteArray<TType, N>::add(::std::auto_ptr<TType> aData)
{
    add(aData.release());
}

//------------------------------------------------------------------------------
template <typename TType, int N>
const TType& AutoDeleteArray<TType, N>::at(const int aIndex) const
{
    SYS_ASSERT(aIndex < count());
    TType* ptr = 0;
    ptr = mContainer[aIndex];
    return *ptr;
}

//------------------------------------------------------------------------------
template <typename TType, int N>
TType& AutoDeleteArray<TType, N>::at(const int aIndex)
{
    SYS_ASSERT(aIndex < count());
    TType* ptr = 0;
    ptr = mContainer[aIndex];
    return *ptr;
}

//------------------------------------------------------------------------------
template <typename TType, int N>
int AutoDeleteArray<TType, N>::index(const TType& aData) const
{
    for (int i = 0; i < count(); ++i) {
        if (&aData == &at(i)) {
            return i;
        }
    }
    SYS_ASSERT_NOT_REACHED();
    return 0;
}

} // namespace
//------------------------------------------------------------------------------
// EOF
