﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <algorithm>
#include <cstddef>
#include <iterator>
#include <stdexcept>
#include <lib/debug/Assert.hpp>
#include <lib/ArrayUtil.hpp>

//------------------------------------------------------------------------------
namespace lib {

/// @addtogroup LIB-Collection
//@{
/// std::tr1::arrayに、デフォルトコンストラクタを追加したもの。
/// @details
/// std::tr1::arrayだと、POD型を指定した場合、全要素が未初期化になる。
/// そこで、デフォルトコンストラクタでメンバ初期化子を追加した。
/// そうすることでPOD型を指定しても0で初期化されるようにした。
template <typename TElement, int TCount>
class Array
{
public:
    /// @name 変数
    //@{
    /// std::tr1::array互換で公開変数に。
    /// @details 非PODになってしまったので公開にする意味はないはずだけど。
    TElement elems[TCount];
    //@}

    /// @name iteratorの定義（名前付けはSTL準拠）
    //@{
    typedef TElement* iterator; ///< 非constのイテレータ。
    typedef const TElement* const_iterator; ///< constのイテレータ。
    //@}

    /// @name 定数
    //@{
    static const int Size = TCount; ///< 最大配列長。
    static const int NotFound = -1; ///< index検索で見つからなかったときに返す値。
    //@}

    /// @name コンストラクタ
    //@{
    Array() : elems() {}
    //@}

    /// @name データ取得
    //@{
    /// 指定番目のオブジェクトを取得。
    inline const TElement& at(const int aIndex) const;
    /// 指定番目のオブジェクトを取得。
    inline TElement& at(const int aIndex);
    /// 最初に見つかった指定のオブジェクトのインデックスを取得する。
    inline int index(const TElement& aData) const;
    /// 最初に見つかった指定のオブジェクトのインデックスを取得する。見つからなかったらNotFoundを返す。
    inline int indexLoose(const TElement& aData) const;
    /// 指定のオブジェクトが既に存在するか。
    inline bool isExist(const TElement& aData) const;
    //@}

    /// @name 要素数
    //@{
    /// 常に最大数を返す要素数。
    inline int count() const;
    //@}

    /// @name イテレータ取得
    //@{
    inline iterator begin();
    inline iterator end();
    inline const_iterator begin() const;
    inline const_iterator end() const;
    //@}

    /// @name 配列アクセス
    //@{
    inline const TElement& operator [](const int aIndex) const;
    inline TElement& operator [](const int aIndex);
    //@}
};
//@}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
const TElement& Array<TElement, TCount>::at(const int aIndex) const
{
    ::lib::ArrayUtil::RangeCheck(aIndex, TCount);
    return elems[aIndex];
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
TElement& Array<TElement, TCount>::at(const int aIndex)
{
    ::lib::ArrayUtil::RangeCheck(aIndex, TCount);
    return elems[aIndex];
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
int Array<TElement, TCount>::index(const TElement& aData) const
{
    const int idx = indexLoose(aData);
    SYS_ASSERT(idx != NotFound);
    return idx;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
int Array<TElement, TCount>::indexLoose(const TElement& aData) const
{
    for (int i = 0; i < count(); ++i) {
        if (aData == at(i)) {
            return i;
        }
    }
    return NotFound;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
bool Array<TElement, TCount>::isExist(const TElement& aData) const
{
    return indexLoose(aData) != NotFound;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
int Array<TElement, TCount>::count() const
{
    return TCount;
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
typename Array<TElement, TCount>::iterator Array<TElement, TCount>::begin()
{
    return &elems[0];
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
typename Array<TElement, TCount>::iterator Array<TElement, TCount>::end()
{
    return &elems[TCount];
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
typename Array<TElement, TCount>::const_iterator Array<TElement, TCount>::begin() const
{
    return &elems[0];
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
typename Array<TElement, TCount>::const_iterator Array<TElement, TCount>::end() const
{
    return &elems[TCount];
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
const TElement& Array<TElement, TCount>::operator [] (const int aIndex) const
{
    return at(aIndex);
}

//------------------------------------------------------------------------------
template <typename TElement, int TCount>
TElement& Array<TElement, TCount>::operator [] (const int aIndex)
{
    return at(aIndex);
}

} // namespace
// EOF
