﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/systm.h>
#include <sys/queue.h>
#include <sys/lock.h>
#include <sys/malloc.h>
#include <sys/kernel.h>

int               hz;
int               tick;
uint64_t          cputicks_per_hz;
sbintime_t        tick_sbt;
#ifndef __SigloBSD__
volatile time_t   time_uptime;
volatile time_t   time_second;
#endif // __SigloBSD__
struct bintime    boottimebin;

#ifdef __SigloBSD__
// Check that we can handle the rollover (for signed 32b millisecond). Approximately at 22 days.
// #define NN_SOCKET_S32bMillS_Rollover_Check
#ifdef NN_SOCKET_S32bMillS_Rollover_Check
const int rollover                    = 0x7fffffff;  // signed 32b
const int rollover_chk_delta_start_ms = 13000; // time before the rollover, in milliseconds
int rollover_offset_ticks;                     // to artifially bump the system timer. For us, units of ticks is milliseconds
#endif // NN_SOCKET_S32bMillS_Rollover_Check
#endif // __SigloBSD__

#ifdef __SigloBSD__
volatile time_t get_current_milliseconds()
{
    return (time_t) nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMilliSeconds();
}
#endif //  __SigloBSD__

volatile int get_current_ticks()
{
    int result = (volatile int)(nn::os::GetSystemTick().GetInt64Value() / cputicks_per_hz);
#ifdef __SigloBSD__
#ifdef NN_SOCKET_S32bMillS_Rollover_Check
    result += rollover_offset_ticks;
#endif // NN_SOCKET_S32bMillS_Rollover_Check
#endif // __SigloBSD__
    return result;
}

void microtime(struct timeval *tv)
{
    uint64_t microseconds = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMicroSeconds();
    tv->tv_sec  = microseconds / 1000000;
    tv->tv_usec = microseconds % 1000000;
    return;
}

void getmicrotime(struct timeval *tv)
{
    uint64_t microseconds = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMicroSeconds();
    tv->tv_sec  = microseconds / 1000000;
    tv->tv_usec = microseconds % 1000000;
    return;
}

void getmicrouptime(struct timeval *tv)
{
    uint64_t microseconds = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMicroSeconds();
    tv->tv_sec  = microseconds / 1000000;
    tv->tv_usec = microseconds % 1000000;
    return;
}

static void timevalfix(struct timeval *t1)
{
    if (t1->tv_usec < 0) {
        t1->tv_sec--;
        t1->tv_usec += 1000000;
    }
    if (t1->tv_usec >= 1000000) {
        t1->tv_sec++;
        t1->tv_usec -= 1000000;
    }
}

void timevaladd(struct timeval *t1, const struct timeval *t2)
{
    t1->tv_sec  += t2->tv_sec;
    t1->tv_usec += t2->tv_usec;
    timevalfix(t1);
}

void timevalsub(struct timeval *t1, const struct timeval *t2)
{
    t1->tv_sec  -= t2->tv_sec;
    t1->tv_usec -= t2->tv_usec;
    timevalfix(t1);
}

/*
 * If ppsratecheck() is called more than maxpps times in a given one second
 * period, the function will return 0, indicating that we exceeded the
 * limit.  If we are below the limit, the function will return 1.  If maxpps
 * is set to 0, the function will always return 0 (no packets/events are
 * permitted).  Negative maxpps indicates that rate limitation is disabled,
 * and ppsratecheck() will always return 1.
 */
int ppsratecheck(struct timeval *lasttime, int *curpps, int maxpps)
{
    /*
     * Reset the last time and counter if this is the first call
     * or more than a second has passed since the last update of
     * lasttime.
     */
    int now = ticks;
    if (lasttime->tv_sec == 0 || (now - lasttime->tv_sec) >= hz) {
        lasttime->tv_sec = now;
        *curpps = 1;
        return (maxpps != 0);
    } else {
        (*curpps)++;            /* NB: ignore potential overflow */
        return (maxpps < 0 || *curpps < maxpps);
    }
}

/* The ratecheck() function provides a simple time interval check which can
 * be used when implementing time-based rate-limited actions.  If the dif-
 * ference between the current monotonically-increasing system time
 * (mono_time) and lasttime is less than the value given by the mininterval
 * argument, zero is returned.  Otherwise, lasttime is set to the current
 * time and a non-zero value is returned.
 */
int ratecheck(struct timeval *lasttime, const struct timeval *mininterval)
{
    struct timeval tv, delta;
    int rv = 0;

    getmicrouptime(&tv);            /* NB: 10ms precision */
    delta = tv;
    timevalsub(&delta, lasttime);

    /*
     * check for 0,0 is so that the message will be seen at least once,
     * even if interval is huge.
     */
    if (timevalcmp(&delta, mininterval, >=) ||
        (lasttime->tv_sec == 0 && lasttime->tv_usec == 0)) {
        *lasttime = tv;
        rv = 1;
    }

    return (rv);
}

/*
 * Check that a proposed value to load into the .it_value or
 * .it_interval part of an interval timer is acceptable, and
 * fix it to have at least minimal value (i.e. if it is less
 * than the resolution of the clock, round it up.)
 */
int
itimerfix(struct timeval *tv)
{
    if (tv->tv_sec < 0 || tv->tv_usec < 0 || tv->tv_usec >= 1000000)
        return (EINVAL);
    if (tv->tv_sec == 0 && tv->tv_usec != 0 && tv->tv_usec < tick / 16)
        tv->tv_usec = (u_int)tick / 16;
    return (0);
}

uint64_t tvtohz(struct timeval *tv)
{
    return (tv->tv_sec * hz) + (tv->tv_usec / tick);
}

sbintime_t microseconds_to_sbt(int64_t us)
{
    struct timeval tv;
    tv.tv_sec  = us / (1000 * 1000);
    tv.tv_usec = us - (tv.tv_sec * (1000 * 1000));
    return tvtosbt(tv);
}

int64_t sbt_to_microseconds(sbintime_t sbt)
{
    struct timeval tv = sbttotv(sbt);
    int64_t usecs = tv.tv_sec;
    usecs *= 1000 * 1000;
    usecs += tv.tv_usec;
    return usecs;
}

void bintime(struct bintime *bt)
{
    uint64_t nanoseconds = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetNanoSeconds();
    bt->sec  = nanoseconds / 1000000000;
    bt->frac = nanoseconds % 1000000000;
    return;
}

void getbinuptime(struct bintime *bt)
{
    uint64_t nanoseconds = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetNanoSeconds();
    bt->sec  = nanoseconds / 1000000000;
    bt->frac = nanoseconds % 1000000000;
    return;
}

uint64_t cputicks_to_usec(int64_t cpu_ticks)
{
    return (cpu_ticks * tick);
}

int64_t sbintime_to_cputicks(sbintime_t time)
{
    struct timeval tv = sbttotv(time);
    return (tv.tv_sec * hz) + (tv.tv_usec / tick);
}

void DELAY(int usecs)
{
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(usecs));
}

static void time_sysinit(void *dummy)
{
    hz              = 1000;
    cputicks_per_hz = nn::os::ConvertToTick(nn::TimeSpan::FromSeconds(1)).GetInt64Value() / hz;
    tick_sbt        = SBT_1S / hz;
    tick            = 1000000 / hz;

#ifdef __SigloBSD__
#ifdef NN_SOCKET_S32bMillS_Rollover_Check
    int now = nn::os::GetSystemTick().GetInt64Value() / cputicks_per_hz;
    rollover_offset_ticks = rollover - now - rollover_chk_delta_start_ms;

    printf("\n\n\n\n*** %s ***\n", __FUNCTION__);
    printf("cputicks_per_hz             = %d\n", cputicks_per_hz);
    printf("rollover_chk_delta_start_ms = %d\n", rollover_chk_delta_start_ms);
    printf("now                         = 0x%8x\n", now);
    printf("rollover_offset_ticks       = 0x%8x\n\n\n", rollover_offset_ticks);
#endif //  NN_SOCKET_S32bMillS_Rollover_Check
#endif //__SigloBSD__
}

SYSINIT(time_sysinit, SI_SUB_PORTINIT, SI_ORDER_FIRST, time_sysinit, NULL);

FORCE_LINK_THIS(_time_);

}
