﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/ucred.h>
#include <sys/proc.h>
#include <siglo/thread.h>

/* currently credentials are the same for all threads
 * we copy them from the init thread's cred structs
 */
extern struct thread  thread0;

/* 1) creates SIGLO thread
 * 2) links it to bsd thread structure
 * 3) assigns credentials
 */
int CreateNetworkThread(
                        NetworkThread* threadExternal,
                        void (*thread_function)(void*),
                        void* argument,
                        void* stack,
                        size_t stacksize,
                        int priority,
                        int cpu)
{
    NetworkThreadInternal* thread = (NetworkThreadInternal*)threadExternal;
    nn::Result rval;

#ifdef SMP
    if (cpu >= 0)
#else
    if (0)
#endif
    {
        rval = nn::os::CreateThread(
                        &(thread->siglo_thread),
                        thread_function,
                        argument,
                        stack,
                        stacksize,
                        priority,
                        cpu);
    } else {
        /* os will pick the core */
        rval = nn::os::CreateThread(
                        &(thread->siglo_thread),
                        thread_function,
                        argument,
                        stack,
                        stacksize,
                        priority);
    }

    if (rval.IsSuccess()) {
        /* default credentials, eventually replace with per process creds */
        thread->magic = BSD_THREAD_MAGIC;
        thread->bsd_thread.td_proc      = thread0.td_proc;
        thread->bsd_thread.td_ucred     = thread0.td_ucred;
        thread->bsd_thread.td_sel       = NULL;
        thread->bsd_thread.td_retval[0] = 0;
        thread->bsd_thread.td_retval[1] = 0;
        thread->bsd_thread.td_errno     = 0;
        return 0;
    } else {
        NN_SDK_LOG("[bsdnet]: failed to create thread - %d:%d\n", rval.GetModule(), rval.GetDescription());
    }

    return -1;
}

void StartNetworkThread(NetworkThread* threadExternal)
{
    NetworkThreadInternal* thread = (NetworkThreadInternal*)threadExternal;
    NN_ABORT_UNLESS(thread->magic == BSD_THREAD_MAGIC);
    nn::os::StartThread(&(thread->siglo_thread));
}

void SetNetworkThreadNamePointer(NetworkThread* threadExternal, const char* threadName)
{
    NetworkThreadInternal* thread = (NetworkThreadInternal*)threadExternal;
    nn::os::SetThreadNamePointer(&(thread->siglo_thread), threadName);
}

void DestroyNetworkThread(NetworkThread* threadExternal)
{
    NetworkThreadInternal* thread = (NetworkThreadInternal*)threadExternal;
    NN_ABORT_UNLESS(thread->magic == BSD_THREAD_MAGIC);
    nn::os::DestroyThread(&(thread->siglo_thread));
    thread->magic = 0;
}

void WaitNetworkThread(NetworkThread* threadExternal)
{
    NetworkThreadInternal* thread = (NetworkThreadInternal*)threadExternal;
    NN_ABORT_UNLESS(thread->magic == BSD_THREAD_MAGIC);
    nn::os::WaitThread(&(thread->siglo_thread));
}

int* NetworkThreadErrno()
{
    /* call's errno is stored in td_errno of NetworkThreadInternal.bsd_thread */
    return &curthread->td_errno;
}

}
