﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/sx.h>
#include <sys/systm.h>
#include <sys/lock.h>
#include <sys/kernel.h>
#include <sys/malloc.h>

static MALLOC_DEFINE(M_SXLOCK, "isxlock", "SX lock");

struct sxlock_internal {
    nn::os::MutexType mutex;
    nn::os::EventType reader_unlocked;
    volatile int reader_count;
};

static void lock_sx(struct lock_object *lock, uintptr_t how)
{
    if (how) {
        sx_slock((struct sx*)lock);
    } else {
        sx_xlock((struct sx*)lock);
    }
}

static uintptr_t unlock_sx(struct lock_object *lock)
{
    struct sx* sx = (struct sx*)lock;
    if (((struct sxlock_internal*)sx->siglo_ptr)->reader_count > 0) {
        sx_sunlock(sx);
        return true;
    } else {
        sx_xunlock(sx);
    }
    return false;
}

struct lock_class lock_class_sx = {
    .lc_name     = "sx",
    .lc_flags    = 0,
    .lc_assert   = NULL,
    .lc_ddb_show = NULL,
    .lc_lock     = lock_sx,
    .lc_owner    = NULL,
    .lc_unlock   = unlock_sx
};

void sx_init_flags(struct sx *sx, const char *name, int opts)
{
    struct sxlock_internal* lock =
        (struct sxlock_internal*)malloc(
                sizeof(struct sxlock_internal),
                M_SXLOCK,
                M_ZERO | M_WAITOK);
    NN_ABORT_UNLESS(lock != NULL);
    nn::os::InitializeMutex(&lock->mutex, opts & SX_RECURSE ? true : false, 0);
    nn::os::InitializeEvent(&lock->reader_unlocked, false, nn::os::EventClearMode_AutoClear);
    sx->siglo_ptr = (void*)lock;
    lock_init(&sx->lock_object, &lock_class_sx, name, NULL, 0);
}

void sx_init(struct sx *sx, const char *name)
{
    sx_init_flags(sx, name, 0);
}

void sx_destroy(struct sx *sx)
{
    struct sxlock_internal* lock = (struct sxlock_internal*)(sx->siglo_ptr);
    nn::os::FinalizeMutex(&lock->mutex);
    nn::os::FinalizeEvent(&lock->reader_unlocked);
    free((void*)sx->siglo_ptr, M_SXLOCK);
}

void sx_slock(struct sx *sx)
{
    struct sxlock_internal* lock = (struct sxlock_internal*)(sx->siglo_ptr);
    nn::os::LockMutex(&lock->mutex);
    __atomic_add_fetch(&lock->reader_count, 1, __ATOMIC_RELAXED);
    nn::os::UnlockMutex(&lock->mutex);
}

void sx_xlock(struct sx *sx)
{
    struct sxlock_internal* lock = (struct sxlock_internal*)(sx->siglo_ptr);
    nn::os::LockMutex(&lock->mutex);
    while (lock->reader_count > 0) {
        nn::os::WaitEvent(&lock->reader_unlocked);
    }
}

int sx_slock_sig(struct sx *sx)
{
    sx_slock(sx);
    return 0;
}

int sx_xlock_sig(struct sx *sx)
{
    sx_xlock(sx);
    return 0;
}

int sx_try_slock(struct sx *sx)
{
    struct sxlock_internal* lock = (struct sxlock_internal*)(sx->siglo_ptr);
    if (nn::os::TryLockMutex(&lock->mutex)) {
        __atomic_add_fetch(&lock->reader_count, 1, __ATOMIC_RELAXED);
        nn::os::UnlockMutex(&lock->mutex);
        return true;
    }
    return false;
}

int sx_try_xlock(struct sx *sx)
{
    struct sxlock_internal* lock = (struct sxlock_internal*)(sx->siglo_ptr);
    if (nn::os::TryLockMutex(&lock->mutex)) {
        if (lock->reader_count == 0) {
            return true;
        }
        nn::os::UnlockMutex(&lock->mutex);
    }
    return false;
}

void sx_sunlock(struct sx *sx)
{
    struct sxlock_internal* lock = (struct sxlock_internal*)(sx->siglo_ptr);
    __atomic_sub_fetch(&lock->reader_count, 1, __ATOMIC_RELAXED);
    nn::os::SignalEvent(&lock->reader_unlocked);
}

void sx_xunlock(struct sx *sx)
{
    struct sxlock_internal* lock = (struct sxlock_internal*)(sx->siglo_ptr);
    nn::os::UnlockMutex(&lock->mutex);
}

void sx_sysinit(void *arg)
{
    struct sx_args *sxargs = (struct sx_args*)arg;
    sx_init_flags((struct sx*)sxargs->sa_sx, sxargs->sa_desc, sxargs->sa_flags);
    return;
}

}
