﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>


extern "C" {
#include <stdarg.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/proc.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/syscallsubr.h>
#include <sys/sysproto.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/_null.h>
#include <sys/errno.h>
#include <sys/select.h>
#include <sys/poll.h>
#include <sys/fcntl.h>
#include <sys/filio.h>
#include <siglo/client.h>

SYSINIT(placeholder, SI_SUB_DUMMY, SI_ORDER_ANY, NULL, NULL);
SET_DECLARE(sysinit_set, struct sysinit);
struct sysinit **sysinit, **sysinit_end;

size_t   mempool_size = 0;
uint8_t* mempool = NULL;

#define SOCKET_ERROR(e, v)              \
({                                      \
    int err = (e);                      \
    int out = (v);                      \
    client_release_ref();               \
    if (err != 0) {                     \
        curthread->td_errno = err;      \
        out = -1;                       \
    }                                   \
    out;                                \
})

int socketduplicate(int s, uint64_t socket_owner_pid, uint64_t pid)
{
    // 1. hold reference to both current process and the process from which socket is duplicated,
    //    we want to avoid losing either struct while duplicating socket.
    // 2. make call then release both current and owner process,
    //    note that ref to current process is released inside SOCKET_ERROR macro
    int result;
    struct socketduplicate_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.owner_proc   = &(client_acquire_ref(socket_owner_pid)->cl_proc);
    args.s            = s;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    result = SOCKET_ERROR(sys_socketduplicate(td, &args), td->td_retval[0]);
    client_release_ref_by_pid(socket_owner_pid);
    return result;
}

int socketgetpeername(int s, struct sockaddr* name, socklen_t* namelen, uint64_t pid)
{
    struct getpeername_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.fdes         = s;
    args.asa          = name;
    args.alen         = namelen;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_getpeername(td, &args), 0);
}

int socketgetsockname(int s, struct sockaddr* name, socklen_t* namelen, uint64_t pid)
{
    struct getsockname_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.fdes         = s;
    args.asa          = name;
    args.alen         = namelen;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_getsockname(td, &args), 0);
}

int socketconnect(int s, const struct sockaddr* name, socklen_t namelen, uint64_t pid)
{
    struct connect_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.name         = (caddr_t)name;
    args.namelen      = (int)namelen;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_connect(td, &args), 0);
}

int socketsocket(int domain, int type, int protocol, uint64_t pid)
{
    struct socket_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.domain       = domain;
    args.type         = type;
    args.protocol     = protocol;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_socket(td, &args), td->td_retval[0]);
}
#ifdef __SigloBSD__
int socketsocketexempt(int domain, int type, int protocol, uint64_t pid)
{
    struct socket_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.domain = domain;
    args.type = type;
    args.protocol = protocol;
    td->td_retval[0] = 0;
    td->td_ucred = &cl->cl_ucred;
    td->td_proc = &cl->cl_proc;
    return SOCKET_ERROR(sys_socket_exempt(td, &args), td->td_retval[0]);
}
#endif
int socketbind(int s, const struct sockaddr* name, socklen_t namelen, uint64_t pid)
{
    struct bind_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.name         = (caddr_t)name;
    args.namelen      = (int)namelen;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_bind(td, &args), 0);
}

int socketlisten(int s, int backlog, uint64_t pid)
{
    struct listen_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.backlog      = backlog;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_listen(td, &args), 0);
}

int socketaccept(int s, struct sockaddr* name, socklen_t* namelen, uint64_t pid)
{
    struct accept_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.name         = name;
    args.anamelen     = namelen;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_accept(td, &args), td->td_retval[0]);
}

ssize_t socketsendto(int s, const void* buf, size_t len, int flags, const struct sockaddr* dest, socklen_t destlen, uint64_t pid)
{
    struct sendto_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.buf          = (caddr_t)buf;
    args.len          = len;
    args.flags        = flags;
    args.to           = (caddr_t)dest;
    args.tolen        = (int)destlen;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_sendto(td, &args), td->td_retval[0]);
}

ssize_t socketsendmsg(int s, const struct msghdr* msg, int flags, uint64_t pid)
{
    struct sendmsg_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.msg          = (struct msghdr*)msg;
    args.flags        = flags;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_sendmsg(td, &args), td->td_retval[0]);
}

#ifdef __SigloBSD__
ssize_t socketsendmmsg(int s, const struct mmsghdr* msgvec, size_t vlen, int flags, uint64_t pid)
{
    struct sendmmsg_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.msgvec       = (struct mmsghdr*)msgvec;
    args.vlen         = vlen;
    args.flags        = flags;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_sendmmsg(td, &args), td->td_retval[0]);
}
#endif

ssize_t socketrecvfrom(int s, void* buf, size_t len, int flags, struct sockaddr* src, socklen_t* fromlen, uint64_t pid)
{
    struct recvfrom_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.buf          = (caddr_t)buf;
    args.len          = len;
    args.flags        = flags;
    args.from         =  src;
    args.fromlenaddr  = fromlen;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_recvfrom(td, &args), td->td_retval[0]);
}

ssize_t socketrecvmsg(int s, struct msghdr* msg, int flags, uint64_t pid)
{
    struct recvmsg_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.msg          = msg;
    args.flags        = flags;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_recvmsg(td, &args), td->td_retval[0]);
}

#ifdef __SigloBSD__
ssize_t socketrecvmmsg(int s, struct mmsghdr* msgvec, size_t vlen, int flags, timespec *timeout, uint64_t pid)
{
    struct recvmmsg_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.msgvec       = msgvec;
    args.vlen         = vlen;
    args.flags        = flags;
    args.timeout      = timeout;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_recvmmsg(td, &args), td->td_retval[0]);
}
#endif

int socketshutdown(int s, int how, uint64_t pid)
{
    struct shutdown_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.how          = how;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_shutdown(td, &args), 0);
}

int socketgetsockopt(int s, int level, int optname, void* optval, socklen_t* optlen, uint64_t pid)
{
    struct getsockopt_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.level        = level;
    args.name         = optname;
    args.val          = (caddr_t)optval;
    args.avalsize     = (int*)optlen;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_getsockopt(td, &args), 0);
}

int socketsetsockopt(int s, int level, int optname, const void* optval, socklen_t optlen, uint64_t pid)
{
    struct setsockopt_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.s            = s;
    args.level        = level;
    args.name         = optname;
    args.val          = (caddr_t)optval;
    args.valsize      = optlen;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_setsockopt(td, &args), 0);
}

ssize_t socketrecv(int s, void* buf, size_t len, int flags, uint64_t pid)
{
    return socketrecvfrom(s, buf, len, flags, NULL, NULL, pid);
}

ssize_t socketsend(int s, const void* buf, size_t len, int flags, uint64_t pid)
{
    return socketsendto(s, buf, len, flags, NULL, 0, pid);
}

int socketioctl(int s, uint32_t cmd, void* data, size_t len, uint64_t pid)
{
    struct ioctl_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.fd           = s;
    args.com          = cmd;
    args.data         = (caddr_t)data;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_ioctl(td, &args), td->td_retval[0]);
}

int socketclose(int s, uint64_t pid)
{
    struct close_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.fd           = s;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_close(td, &args), td->td_retval[0]);
}

int socketwrite(int s, const void *buf, size_t nbyte, uint64_t pid)
{
    struct write_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.fd           = s;
    args.buf          = buf;
    args.nbyte        = nbyte;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_write(td, &args), td->td_retval[0]);
}

int socketread(int s, void *buf, size_t nbyte, uint64_t pid)
{
    struct read_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.fd           = s;
    args.buf          = buf;
    args.nbyte        = nbyte;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_read(td, &args), td->td_retval[0]);
}

int socketselect(int nfds, fd_set *rfds, fd_set *wfds, fd_set *efds, struct timeval *timeout, uint64_t pid)
{
    struct select_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.nd           = nfds;
    args.in           = rfds;
    args.ou           = wfds;
    args.ex           = efds;
    args.tv           = (const struct timeval*)timeout;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_select(td, &args), td->td_retval[0]);
}

int socketpoll(struct pollfd* fds, nfds_t nfds, int timeout, uint64_t pid)
{
    struct poll_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.fds          = fds;
    args.nfds         = nfds;
    args.timeout      = timeout;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_poll(td, &args), td->td_retval[0]);
}

int socketfcntl(int s, int cmd, int iocmd, uint64_t pid)
{
    int rval, clean_iocmd;
    switch (cmd)
    {
    case F_SETFL:
        clean_iocmd = O_NONBLOCK & iocmd;
        rval = socketioctl(s, FIONBIO, (caddr_t)&clean_iocmd, sizeof(clean_iocmd), pid);
        break;
    case F_GETFL:
        rval = socketioctl(s, NN_GET_SOCKFL, (caddr_t)&iocmd, sizeof(iocmd), pid);
        if (-1 == rval)
        {
            goto bail;
        }
        else if (iocmd & SS_NBIO)
        {
            rval = O_NONBLOCK;
        }
        else
        {
            rval = 0;
        };
        break;
    default:
        rval  = -1;
        curthread->td_errno = EOPNOTSUPP;
        break;
    }
bail:
    return rval;
}

int socketopen(const char *path, int flags, uint64_t pid)
{
    struct open_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.path         = (char*)path;
    args.flags        = flags;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys_open(td, &args), td->td_retval[0]);
}

int socketsysctl(int *mib, size_t miblength, void *oldvalue, size_t *oldlength, void *newvalue, size_t newlength, uint64_t pid)
{
    struct sysctl_args args;
    struct thread *td = curthread;
    struct client *cl = client_acquire_ref(pid);
    args.name         = mib;
    args.namelen      = miblength;
    args.oldctl       = oldvalue;
    args.oldlenp      = oldlength;
    args.newctl       = newvalue;
    args.newlen       = newlength;
    td->td_retval[0]  = 0;
    td->td_ucred      = &cl->cl_ucred;
    td->td_proc       = &cl->cl_proc;
    return SOCKET_ERROR(sys___sysctl(td, &args), td->td_retval[0]);
}

void NetworkInit(void* malloc_area, size_t malloc_area_size)
{
    struct sysinit **sipp;
    struct sysinit **xipp;
    struct sysinit *save;

    printf("[net] BSD sockets \n");

    mempool = (uint8_t*)malloc_area;
    mempool_size = malloc_area_size;

    if (sysinit == NULL) {
        sysinit     = SET_BEGIN(sysinit_set);
        sysinit_end = SET_LIMIT(sysinit_set);
    }
    /* re-order start up entries by their init order */
    for (sipp = sysinit; sipp < sysinit_end; sipp++) {
        for (xipp = sipp + 1; xipp < sysinit_end; xipp++) {
            if ((*sipp)->subsystem < (*xipp)->subsystem ||
                ((*sipp)->subsystem == (*xipp)->subsystem &&
                 (*sipp)->order <= (*xipp)->order))
            {
                continue;       /* skip*/
            }
            save = *sipp;
            *sipp = *xipp;
            *xipp = save;
        }
    }
    for (sipp = sysinit; sipp < sysinit_end; sipp++) {
        if ((*sipp)->subsystem == SI_SUB_DUMMY) {
            continue;
        }
        if ((*sipp)->subsystem == SI_SUB_DONE) {
            continue;
        }
#if BSDDEBUG
        printf("[net] %s\n", (*sipp)->fname);
#endif
        (*((*sipp)->func))((*sipp)->udata);
        (*sipp)->subsystem = SI_SUB_DONE;
    }
#if BSDDEBUG
    printf("[net] done\n");
#endif
    return;
}

}
