﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/mem/mem_StandardAllocator.h>

extern "C"
{

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/sysctl.h>
#include <sys/eventhandler.h>
#include <opt/opt_malloc.h>

/* Default memory allocator */
extern size_t   mempool_size;
extern uint8_t* mempool;
nn::mem::StandardAllocator g_Allocator;

#if MALLOC_COLLECT_STATS

static SLIST_HEAD(slisthead, malloc_type) malloc_list = SLIST_HEAD_INITIALIZER(malloc_list);
static unsigned long malloc_counter = 0;

/* This is placed just below the pointer returned to the user,
 * contains requested size and offset to the 'actual' start of the chunk.
 */
typedef struct
{
    unsigned long size;
    unsigned long offset;
} __packed malloc_header;

static void *malloc_add_record(struct malloc_type *m_type, void *addr, unsigned long size, unsigned long offset)
{
    ((malloc_header*)addr)->size   = size;
    ((malloc_header*)addr)->offset = offset;
    critical_enter();
    m_type->ks_allocated += size;
    m_type->ks_numallocated++;
    critical_exit();
    return (malloc_header*)addr + 1;
}

static void *malloc_remove_record(struct malloc_type *m_type, void *addr)
{
    unsigned long size   = ((malloc_header*)addr - 1)->size;
    unsigned long offset = ((malloc_header*)addr - 1)->offset;
    critical_enter();
    m_type->ks_freed += size;
    m_type->ks_numfreed++;
    critical_exit();
    return (uint8_t*)addr - offset;
}

static void *lmalloc(unsigned long size, struct malloc_type *m_type, int flags)
{
    nn::mem::StandardAllocator *allocator = (nn::mem::StandardAllocator*)(m_type->ks_allocator);
    void* addr = allocator->Allocate(size + sizeof(malloc_header));
    if (addr != NULL) {
        addr = malloc_add_record(m_type, addr, size, sizeof(malloc_header));
        if (flags & M_ZERO) {
            memset(addr, 0x0, size);
        }
    }
    return addr;
}

static void lfree(void *addr, struct malloc_type *m_type)
{
    if (addr != NULL) {
        nn::mem::StandardAllocator *allocator = (nn::mem::StandardAllocator*)(m_type->ks_allocator);
        addr = malloc_remove_record(m_type, addr);
        allocator->Free(addr);
    }
}

static void *lmemalign(unsigned long align, unsigned long size, struct malloc_type *type, int flags)
{
    nn::mem::StandardAllocator *allocator = (nn::mem::StandardAllocator*)(type->ks_allocator);
    void *addr = allocator->Allocate(size + align, align);
    if (addr != NULL) {
        addr = malloc_add_record(type, (uint8_t*)addr + align - sizeof(malloc_header), size, align);
        if (flags & M_ZERO) {
            memset(addr, 0x0, size);
        }
    }
    return addr;
}

static void *lrealloc(void *addr, unsigned long size, struct malloc_type *type, int flags)
{
    nn::mem::StandardAllocator *allocator = (nn::mem::StandardAllocator*)(type->ks_allocator);
    addr = allocator->Reallocate(malloc_remove_record(type, addr), size + sizeof(malloc_header));
    if (addr != NULL) {
        addr = malloc_add_record(type, addr, size, sizeof(malloc_header));
    }
    return addr;
}

static int malloc_sysctl_handler(SYSCTL_HANDLER_ARGS)
{
    struct   malloc_type *entry;
    uint64_t malloc_used = 0;
    char*    malloc_profile;
    int      result = -1;
    int      offset;

    critical_enter();

    #define PROFILE_LINE_LENGTH 200

    size_t malloc_profile_length  = (malloc_counter + 10) * PROFILE_LINE_LENGTH;
    char  *malloc_profile_pointer = malloc_profile = (char*)g_Allocator.Allocate(malloc_profile_length);

    if (malloc_profile != NULL)
    {
        offset = snprintf(
                        malloc_profile_pointer,
                        malloc_profile_length,
                        "\nMalloc Stats:\n"
                        " Allocated - total allocated during runtime (Bytes)\n"
                        " Freed     - total freed during runtime (Bytes)\n"
                        " NumAllocs - total # of allocation requests\n"
                        " NumFrees  - total # of free requests\n"
                        " Used      - memory in use (Bytes)\n"
                        "%10s: %10s: %10s: %10s: %10s: %10s\n",
                        "Allocated",
                        "Freed",
                        "NumAllocs",
                        "NumFrees",
                        "Used",
                        "Description");

        malloc_profile_length  -= offset;
        malloc_profile_pointer += offset;

        SLIST_FOREACH(entry, &malloc_list, ks_entry) {

            offset = snprintf(
                        malloc_profile_pointer,
                        malloc_profile_length,
                        "%10ju: %10ju: %10ju: %10ju: %10ju: %s\n",
                        entry->ks_allocated,
                        entry->ks_freed,
                        entry->ks_numallocated,
                        entry->ks_numfreed,
                        entry->ks_allocated - entry->ks_freed,
                        entry->ks_longdesc);

            if (offset >= malloc_profile_length) {
                break;
            }

            malloc_profile_length  -= offset;
            malloc_profile_pointer += offset;
            malloc_used += (entry->ks_allocated - entry->ks_freed);
        }

        snprintf(
            malloc_profile_pointer,
            malloc_profile_length,
            "Total memory used: %ju (Bytes)\n",
            malloc_used);

        result = SYSCTL_OUT(req, malloc_profile, strlen(malloc_profile) + 1);

        g_Allocator.Free(malloc_profile);
    }

    critical_exit();

    return result;
}

static void lmalloc_init(void *v)
{
    struct malloc_type *m_type = (struct malloc_type*)v;
    if (m_type->ks_allocator == NULL) {
        m_type->ks_allocator = (void*)&g_Allocator;
    }
    m_type->ks_allocated    = 0;
    m_type->ks_freed        = 0;
    m_type->ks_numallocated = 0;
    m_type->ks_numfreed     = 0;
    critical_enter();
    malloc_counter++;
    SLIST_INSERT_HEAD(&malloc_list, m_type, ks_entry);
    critical_exit();
    return;
}

static void lmalloc_uninit(void *v)
{
    struct malloc_type *m_type = (struct malloc_type*)v;
    critical_enter();
    SLIST_REMOVE(&malloc_list, m_type, malloc_type, ks_entry);
    malloc_counter--;
    critical_exit();
    return;
}

SYSCTL_PROC(_kern, OID_AUTO, memory, CTLTYPE_STRING | CTLFLAG_RD,
        NULL, 0, malloc_sysctl_handler, "A", "malloc statistics");

#else // !MALLOC_COLLECT_STATS

static void *lmalloc(unsigned long size, struct malloc_type *type, int flags)
{
    nn::mem::StandardAllocator *allocator = (nn::mem::StandardAllocator*)(type->ks_allocator);
    void* addr = allocator->Allocate(size);
    if (addr != NULL && flags & M_ZERO) {
        memset(addr, 0x0, size);
    }
    return addr;
}

static void lfree(void *addr, struct malloc_type *type)
{
    if (addr != NULL) {
        nn::mem::StandardAllocator *allocator = (nn::mem::StandardAllocator*)(type->ks_allocator);
        allocator->Free(addr);
    }
}

static void *lmemalign(unsigned long align, unsigned long size, struct malloc_type *type, int flags)
{
    nn::mem::StandardAllocator *allocator = (nn::mem::StandardAllocator*)(type->ks_allocator);
    void *addr = allocator->Allocate(size, align);
    if (addr != NULL && flags & M_ZERO) {
        memset(addr, 0x0, size);
    }
    return addr;
}

static void *lrealloc(void *addr, unsigned long size, struct malloc_type *type, int flags)
{
    nn::mem::StandardAllocator *allocator = (nn::mem::StandardAllocator*)(type->ks_allocator);
    return allocator->Reallocate(addr, size);
}

static void lmalloc_init(void *v)
{
    struct malloc_type *m_type = (struct malloc_type*)v;
    if (m_type->ks_allocator == NULL) {
        m_type->ks_allocator = (void*)&g_Allocator;
    }
    return;
}

static void lmalloc_uninit(void *v)
{
    return;
}

#endif // !MALLOC_COLLECT_STATS

/* Even though M_WAITOK is supported, it may not be very useful
 * in our environemnt. Memory allocations in the stack are mostly static.
 * It is unlikely that desired memory is freed up. It may make more sense
 * to just terminate the process when out of memory condition is detected...
 *
 * For NET team to review.
 */

static void mem_signal_init(struct malloc_type *type)
{
    nn::mem::StandardAllocator *allocator = (nn::mem::StandardAllocator*)(type->ks_allocator);
    type->ks_waitercount = 0;
    type->ks_waitercond  = allocator->Allocate(sizeof(nn::os::ConditionVariableType));
    type->ks_waiterlock  = allocator->Allocate(sizeof(nn::os::MutexType));
    if (type->ks_waitercond == NULL || type->ks_waiterlock == NULL) {
        /* unlikely to happen... */
        panic("out of memory\n");
    }
    nn::os::InitializeMutex((nn::os::MutexType*)(type->ks_waiterlock), false, 0);
    nn::os::InitializeConditionVariable((nn::os::ConditionVariableType*)(type->ks_waitercond));
}

static void mem_signal_wait(struct malloc_type *type)
{
    EVENTHANDLER_INVOKE(vm_lowmem, 0);
    nn::os::LockMutex((nn::os::MutexType*)type->ks_waiterlock);
    type->ks_waitercount++;
    nn::os::WaitConditionVariable(
            (nn::os::ConditionVariableType*)type->ks_waitercond,
            (nn::os::MutexType*)type->ks_waiterlock);
    type->ks_waitercount--;
    nn::os::UnlockMutex((nn::os::MutexType*)type->ks_waiterlock);
}

static void mem_signal_send(struct malloc_type *type)
{
    if (type->ks_waitercount) {
        nn::os::BroadcastConditionVariable((nn::os::ConditionVariableType*)type->ks_waitercond);
    }
}

static void mem_signal_uninit(struct malloc_type *type)
{
    nn::mem::StandardAllocator *allocator = (nn::mem::StandardAllocator*)type->ks_allocator;
    if (type->ks_waiterlock != NULL) {
        nn::os::FinalizeMutex((nn::os::MutexType*)(type->ks_waiterlock));
        allocator->Free(type->ks_waiterlock);
        type->ks_waiterlock = NULL;
    }
    if (type->ks_waitercond != NULL) {
        nn::os::FinalizeConditionVariable((nn::os::ConditionVariableType*)type->ks_waitercond);
        allocator->Free(type->ks_waitercond);
        type->ks_waitercond = NULL;
    }
}

void *malloc(unsigned long size, struct malloc_type *type, int flags)
{
    void *addr;
    do {
        addr = lmalloc(size, type, flags);
        if (addr == NULL && (size != 0) && (flags & M_WAITOK)) {
            mem_signal_wait(type);
        } else {
            break;
        }
    } while (1);
    return addr;
}

void free(void *addr, struct malloc_type *type)
{
    lfree(addr, type);
    mem_signal_send(type);
}

void *memalign(unsigned long align, unsigned long size, struct malloc_type *type, int flags)
{
    void *addr;
    do {
        addr = lmemalign(align, size, type, flags);
        if (addr == NULL && (size != 0) && (flags & M_WAITOK)) {
            mem_signal_wait(type);
        } else {
            break;
        }
    } while (1);
    return addr;
}

void *realloc(void *addr, unsigned long size, struct malloc_type *type, int flags)
{
    void *newaddr;
    do {
        newaddr = lrealloc(addr, size, type, flags);
        if (newaddr == NULL && (size != 0) && (flags & M_WAITOK)) {
            mem_signal_wait(type);
        } else {
            break;
        }
    } while (1);
    return newaddr;
}

void malloc_init(void *v)
{
    lmalloc_init(v);
    mem_signal_init((struct malloc_type*)v);
}

void malloc_uninit(void *v)
{
    mem_signal_uninit((struct malloc_type*)v);
    lmalloc_uninit(v);
}

static void malloc_sysinit(void *dummy)
{
    /* initialize main memory allocator */
    g_Allocator.Initialize(mempool, mempool_size);
}

SYSINIT(malloc_sysinit, SI_SUB_PORTINIT, SI_ORDER_FIRST, malloc_sysinit, NULL);

}
