﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/file.h>
#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/libkern.h>
#include <sys/sysproto.h>
#include <sys/filedesc.h>

static LIST_HEAD(, cdev) si_head;
static struct mtx device_mutex;
static MALLOC_DEFINE(M_CDEVPDATA, "DEVFSP", "Metainfo for cdev-fp data");

struct cdev_privdata {
    struct file *cdpd_fp;
    void        *cdpd_data;
    void       (*cdpd_dtr)(void *);
};

int devfs_set_cdevpriv(void *priv, cdevpriv_dtr_t priv_dtr)
{
    struct file *fp;
    struct cdev_privdata *p;
    int error;

    fp = curthread->td_fpop;
    if (fp == NULL) {
        return ENOENT;
    }
    p = (struct cdev_privdata*)malloc(sizeof(struct cdev_privdata), M_CDEVPDATA, M_WAITOK);
    p->cdpd_data = priv;
    p->cdpd_dtr  = priv_dtr;
    p->cdpd_fp   = fp;
    if (fp->f_cdevpriv == NULL) {
        fp->f_cdevpriv = (void*)p;
        error = 0;
    } else {
        free(p, M_CDEVPDATA);
        error = EBUSY;
    }
    return error;
}

int devfs_get_cdevpriv(void **datap)
{
    struct file *fp;
    struct cdev_privdata *p;
    int error;

    fp = curthread->td_fpop;
    if (fp == NULL) {
        return EBADF;
    }
    p = (cdev_privdata*)fp->f_cdevpriv;
    if (p != NULL) {
        error = 0;
        *datap = p->cdpd_data;
    } else {
        error = ENOENT;
    }
    return error;
}

void devfs_destroy_cdevpriv(struct cdev_privdata *p)
{
    p->cdpd_fp->f_cdevpriv = NULL;
    (p->cdpd_dtr)(p->cdpd_data);
    free(p, M_CDEVPDATA);
}

void devfs_fpdrop(struct file *fp)
{
    struct cdev_privdata *p;

    if ((p = (cdev_privdata *)(fp->f_cdevpriv)) == NULL)
    {
        return;
    }
    devfs_destroy_cdevpriv(p);
}


const char *devtoname(struct cdev *dev)
{
    return (dev->si_name);
}

struct cdev *make_dev(
    struct cdevsw *devsw,
    int    unit,
    uid_t  uid,
    gid_t  gid,
    int    perms,
    const  char *fmt, ...)
{
    struct cdev* si;
    char   tmp[SPECNAMELEN];
    va_list ap;
    si = (struct cdev*)malloc(sizeof(struct cdev), M_CDEVPDATA, M_ZERO | M_WAITOK);
    if (si != NULL) {
        si->si_devuid = uid;
        si->si_devgid = gid;
        si->si_devsw  = devsw;
        si->si_drv0   = unit;
        va_start(ap, fmt);
        snprintf(tmp, sizeof(tmp), "/dev/%s", fmt);
        vsnprintf(si->si_name, sizeof(si->si_name), tmp, ap);
        va_end(ap);
        mtx_lock(&device_mutex);
        LIST_INSERT_HEAD(&si_head, si, si_list);
        mtx_unlock(&device_mutex);
#if BSDDEBUG
        printf("[net_init]: %s\n", si->si_name);
#endif
    }
    return si;
}

struct cdev *make_dev_alias(
    struct cdev *pdev,
    const char  *fmt, ...)
{
    struct cdev* si;
    char   tmp[SPECNAMELEN];
    va_list ap;
    si = (struct cdev*)malloc(sizeof(struct cdev), M_CDEVPDATA, M_ZERO | M_WAITOK);
    if (si != NULL) {
        si->si_devuid = pdev->si_devuid;
        si->si_devgid = pdev->si_devgid;
        si->si_devsw  = pdev->si_devsw;
        si->si_drv0   = pdev->si_drv0;
        va_start(ap, fmt);
        snprintf(tmp, sizeof(tmp), "/dev/%s", fmt);
        vsnprintf(si->si_name, sizeof(si->si_name), tmp, ap);
        va_end(ap);
        mtx_lock(&device_mutex);
        LIST_INSERT_HEAD(&si_head, si, si_list);
        mtx_unlock(&device_mutex);
#if BSDDEBUG
        printf("[net_init]: %s\n", si->si_name);
#endif
    }
    return si;
}

static int dev_read(
    struct file  *fp,
    struct uio   *uio,
    struct ucred *active_cred,
    int flags,
    struct thread *td)
{
    struct cdev* si = (struct cdev*)fp->f_data;
    curthread->td_fpop = fp;
    return si->si_devsw->d_read ?
           si->si_devsw->d_read(si, uio, 0) :
           (-1);
}

static int dev_write(
    struct file  *fp,
    struct uio   *uio,
    struct ucred *active_cred,
    int flags,
    struct thread *td)
{
    struct cdev* si = (struct cdev*)fp->f_data;
    curthread->td_fpop = fp;
    return si->si_devsw->d_write ? si->si_devsw->d_write(si, uio, 0) : -1;
}

static int dev_ioctl(
    struct file   *fp,
    u_long         com,
    void          *data,
    struct ucred  *active_cred,
    struct thread *td)
{
    struct cdev* si = (struct cdev*)fp->f_data;
    curthread->td_fpop = fp;
    return si->si_devsw->d_ioctl ?
           si->si_devsw->d_ioctl(si, com, (caddr_t)data, 0, td) :
           (-1);
}

static int dev_poll(
    struct file   *fp,
    int            events,
    struct ucred  *active_cred,
    struct thread *td)
{
    struct cdev* si = (struct cdev*)fp->f_data;
    curthread->td_fpop = fp;
    return si->si_devsw->d_poll ?
           si->si_devsw->d_poll(si, events, td) :
           (-1);
}

static int dev_kqfilter(
    struct file  *fp,
    struct knote *kn)
{
    struct cdev* si = (struct cdev*)fp->f_data;
    curthread->td_fpop = fp;
    return si->si_devsw->d_kqfilter ?
           si->si_devsw->d_kqfilter(si, kn) :
           (-1);
}

static int dev_close(
    struct file   *fp,
    struct thread *td)
{
    int rval;
    struct cdev* si = (struct cdev*)fp->f_data;
    curthread->td_fpop = fp;
    rval = si->si_devsw->d_close ? si->si_devsw->d_close(si, 0, 0, td) : 0;
    if (fp->f_cdevpriv != NULL) {
        devfs_fpdrop(fp);
    }
    return rval;
}

struct fileops deviceops = {
    .fo_read        = dev_read,
    .fo_write       = dev_write,
    .fo_truncate    = NULL,
    .fo_ioctl       = dev_ioctl,
    .fo_poll        = dev_poll,
    .fo_kqfilter    = dev_kqfilter,
    .fo_stat        = NULL,
    .fo_close       = dev_close,
    .fo_chmod       = NULL,
    .fo_chown       = NULL,
    .fo_sendfile    = NULL
};

#ifndef _SYS_SYSPROTO_H_
struct open_args {
    char* path;
    int   flags;
    int   mode;
};
#endif
int sys_open(
    struct thread    *td,
    struct open_args *uap)
{
    struct cdev* si;
    struct file* fp;
    int    fd;
    int    err = -1;

    mtx_lock(&device_mutex);
    LIST_FOREACH(si, &si_head, si_list) {
        /* look in registered paths */
        if (strncmp(si->si_name, uap->path, sizeof(si->si_name)) == 0) {
            if ((err = falloc(td, &fp, &fd, uap->flags)) == 0) {
                curthread->td_fpop = fp;
                finit(fp, uap->flags, DTYPE_VNODE, si, &deviceops);
                if ((err = si->si_devsw->d_open(si, 0, 0, td)) == 0) {
                    td->td_retval[0] = fd;
                } else {
                    fdclose(td->td_proc->p_fd, fp, fd, td);
                }
                fdrop(fp, td);
            }
            break;
        }
    }
    mtx_unlock(&device_mutex);
    return err;
}

#ifndef _SYS_SYSPROTO_H_
struct close_args {
        int     fd;
};
#endif
int
sys_close(
    struct thread *td,
    struct close_args *uap)
{
    int fd = uap->fd;
    int error;
    struct file *fp;
    error = fget_unlocked(NULL, fd, NULL, 0, &fp, NULL);
    if (error != 0) {
        return error;
    }
    fdclose(curproc->p_fd, fp, fd, td);
    fdrop(fp, td);
    return error;
}

static void device_sysinit(void* arg)
{
    mtx_init(&device_mutex, "device list lock", NULL, MTX_DEF);
    LIST_INIT(&si_head);
}

SYSINIT(device_sysinit, SI_SUB_DRIVERS, SI_ORDER_FIRST, device_sysinit, NULL);

}

