﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/nn_SystemThreadDefinition.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/queue.h>
#include <sys/callout.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/rwlock.h>
#include <sys/kernel.h>
#include <siglo/thread.h>

static const int CS_DRAIN_WAIT = 20000;   // total wait time in microseconds
static const int CS_DRAIN_POLL = 5000;    // polling time in microseconds
static const int CS_DRAIN = 0x0001;       // wait allowed.  Used as arg to _callout_stop()

static const int CALLOUT_STACK_SIZE = 8192;
static struct callout_tailq cc_expire;
static struct callout_tailq cc_schedout;
static struct callout       *cc_curr;     // Normaly per CPU, but we are only on one core for NX.  See struct cc_exec in ref. code
static bool                 cc_cancel;    // Normaly per CPU, but we are only on one core for NX.  See struct cc_exec in ref. code
static bool                 cc_waiting;   // Normaly per CPU, but we are only on one core for NX.  See struct cc_exec in ref. code
static NetworkThread        callout_thd;
static nn::os::EventType    callout_event;
static nn::os::MutexType    callout_mutex;
static uint8_t callout_stack[CALLOUT_STACK_SIZE]
    __attribute__((aligned(nn::os::StackRegionAlignment)));

// Debug function for tracing callout over time
void callout_dump()
{
    struct callout *entry, *temp;

    printf("In callout_reset_sbt_on\n");
    printf("callout_dump at %d \n", cputicks);
    LockMutex(&callout_mutex);
    TAILQ_FOREACH_SAFE(entry, &cc_expire, tqe, temp)
    {
        printf("    function: %p timer %d \n", entry->c_func, sbt_to_microseconds(entry->c_time));
    }

    printf("cc_schedout\n");
    TAILQ_FOREACH_SAFE(entry, &cc_schedout, tqe, temp)
    {
        printf("    function: %p timer %d \n", entry->c_func, sbt_to_microseconds(entry->c_time));
    }
    UnlockMutex(&callout_mutex);
}

int callout_pending(struct callout *c)
{
    return (c->c_iflags & CALLOUT_PENDING);
}

int callout_active(struct callout *c)
{
    return (c->c_flags & CALLOUT_ACTIVE);
}


void callout_deactivate(struct callout *c)
{
    c->c_flags &= ~CALLOUT_ACTIVE;
}

int _callout_stop(struct callout *c, int flags)
{
    int wait_time = 0;
    bool use_lock = false;

    if ((flags & CS_DRAIN) == 0 && c->c_lock != NULL) {
        use_lock = true;
    }

    LockMutex(&callout_mutex);

    if (!(c->c_iflags & CALLOUT_PENDING)) {     // not on a list. Possibly about to execute or executing
        c->c_flags  &= ~CALLOUT_ACTIVE;

        // if not current, then either cc_curr changed and/or callout already finished
        if (cc_curr != c) {                     // not currently executing (nor about to be executing)
            cc_waiting = false;
            UnlockMutex(&callout_mutex);
            return 0;                           // cannot tell if not set (-1) nor already completed (0)
        }

        if ((flags & CS_DRAIN) != 0) {          // wait for callout to finish?
            while (cc_curr == c) {
                cc_waiting = true;
                UnlockMutex(&callout_mutex);
                DELAY(CS_DRAIN_POLL);
                LockMutex(&callout_mutex);
                wait_time += CS_DRAIN_POLL;
                // should not be needed, but just in case.  Do not want to block Finalize()
                if (wait_time >= CS_DRAIN_WAIT) {       // if past total desired wait time, CS_DRAIN_WAIT
                    break;                      // do not wait anymore
                }
            }
        } else if (use_lock && !cc_cancel) {    // try to cancel right before it executes
            cc_cancel = true;
            UnlockMutex(&callout_mutex);
            return 1;
        }

        UnlockMutex(&callout_mutex);
        return 0;
    }

    // else, on either the schedout or expire list
    if ((c->c_iflags & CALLOUT_PROCESSED) == 0) {
        QMD_TAILQ_ASSERT_ELM_ON_LIST(callout, &cc_schedout, tqe, c);
        TAILQ_REMOVE(&cc_schedout, c, tqe);
    } else {
        QMD_TAILQ_ASSERT_ELM_ON_LIST(callout, &cc_expire, tqe, c);
        TAILQ_REMOVE(&cc_expire, c, tqe);
    }

    c->c_iflags &= ~CALLOUT_PENDING;
    c->c_flags  &= ~CALLOUT_ACTIVE;

    UnlockMutex(&callout_mutex);
    return 1;
}

int callout_drain(struct callout *c)
{
    return _callout_stop(c, CS_DRAIN);
}

int callout_stop(struct callout *c)
{
    return _callout_stop(c, 0);
}

int callout_schedule(struct callout *c, int to)
{
    return callout_reset_on(c, to, c->c_func, c->c_arg, c->c_cpu);
}

void callout_init(struct callout *c, int mpsafe)
{
    memset((void*)c, 0x00, sizeof(struct callout));
    return;
}

void callout_init_mtx(struct callout *c, struct mtx *mtx, int flags)
{
    callout_init(c, 0);
    c->c_lock   = &(mtx)->lock_object;
    c->c_iflags = flags & (CALLOUT_RETURNUNLOCKED | CALLOUT_SHAREDLOCK);
    return;
}

void callout_init_rw(struct callout *c, struct rwlock *rw, int flags)
{
    callout_init(c, 0);
    c->c_lock   = &(rw)->lock_object;
    c->c_iflags = flags & (CALLOUT_RETURNUNLOCKED | CALLOUT_SHAREDLOCK);
    return;
}

int callout_reset_sbt_on(
    struct callout *c, sbintime_t to, sbintime_t dummy,
    void (*c_func)(void *), void *c_arg, int c_cpu, int flags)
{
    int cancelled = 0;
    struct callout* entry;

    LockMutex(&callout_mutex);

    if (cc_curr == c) {   // reschedule a callout that is currently in progress?
        if (c->c_lock != NULL && !cc_cancel) {
            cancelled = 1;
            cc_cancel = true;
        }
        if (cc_waiting) { // callout_drain() called.  Do not reschedule.
            UnlockMutex(&callout_mutex);
            return (cancelled);
        }
    }

    if (c->c_iflags & CALLOUT_PENDING) {
        if (!(c->c_iflags & CALLOUT_PROCESSED)) {
            QMD_TAILQ_ASSERT_ELM_ON_LIST(callout, &cc_schedout, tqe, c);
            TAILQ_REMOVE(&cc_schedout, c, tqe);
        } else {
            QMD_TAILQ_ASSERT_ELM_ON_LIST(callout, &cc_expire, tqe, c);
            TAILQ_REMOVE(&cc_expire, c, tqe);
        }
        cancelled = 1;
        // c->c_iflags &= ~CALLOUT_PENDING  // Set in the following.  No need to clear here.
        // c->c_flags  &= ~CALLOUT_ACTIVE;  // Set in the following.  No need to clear here.
    }

    // add callout to the callout schedule list
    sbintime_t nowsbt = microseconds_to_sbt(nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMicroSeconds());
    c->c_time   = nowsbt + to;   // absolute sbintime
    c->c_func   = c_func;
    c->c_arg    = c_arg;
    c->c_cpu    = c_cpu;
    c->c_iflags &= ~CALLOUT_PROCESSED;
    c->c_iflags |= CALLOUT_PENDING;
    c->c_flags  |= CALLOUT_ACTIVE;
    int iterations = 0;
    /* keep list ordered by time */
    TAILQ_FOREACH_REVERSE(entry, &cc_schedout, callout_tailq, tqe) {
        if (c->c_time > entry->c_time) {
            TAILQ_INSERT_AFTER(&cc_schedout, entry, c, tqe);
            goto out;
        }
        // There should never be over 5000 items on this list.
        NN_ABORT_UNLESS(iterations < 5000, "Too many iterations when reverse traversing cc_schedout\n");
        ++iterations;
    }
    TAILQ_INSERT_HEAD(&cc_schedout, c, tqe);
    nn::os::SignalEvent(&callout_event);
out:
    UnlockMutex(&callout_mutex);
    return cancelled;
}

// we assume the callout lock is held
static void callout_process_call_entries()
{
    struct callout* entry;
    sbintime_t nowsbt = microseconds_to_sbt(nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMicroSeconds());
    struct callout *temp;
    int iterations = 0;
    TAILQ_FOREACH_SAFE(entry, &cc_schedout, tqe, temp) {
        if (entry->c_time <= nowsbt) {
            TAILQ_REMOVE(&cc_schedout, entry, tqe);
            entry->c_iflags |= CALLOUT_PROCESSED;
            TAILQ_INSERT_HEAD(&cc_expire, entry, tqe);
        } else {
            break;
        }
        // There should never be over 5000 items on this list.
        NN_ABORT_UNLESS(iterations < 5000, "Too many iterations when traversing cc_schedout\n");
        ++iterations;
    }
    return;
}

static void callout_dispatch_call_entries()
{
    struct callout     *entry;
    struct lock_class  *lock_class;

    void (*c_func)(void *);
    void               *c_arg;
    struct lock_object *c_lock;
    short               c_flags;
    short               c_iflags;
    int iterations = 0;
    LockMutex(&callout_mutex);
    while (!TAILQ_EMPTY(&cc_expire)) {
        entry = TAILQ_FIRST(&cc_expire);
        TAILQ_REMOVE(&cc_expire, entry, tqe);

        c_lock   = entry->c_lock;
        c_func   = entry->c_func;
        c_arg    = entry->c_arg;
        c_iflags = entry->c_iflags;
        c_flags  = entry->c_flags;

        entry->c_iflags &= ~CALLOUT_PENDING;
        cc_curr   = entry;
        cc_cancel = false;
        // There should never be over 5000 items on this list.
        NN_ABORT_UNLESS(iterations < 5000, "Too many iterations when traversing cc_expire\n");
        ++iterations;
        UnlockMutex(&callout_mutex);

        if (c_lock != NULL) {
            lock_class = LOCK_CLASS(c_lock);
            lock_class->lc_lock(c_lock, 0);

            // The callout may have been cancelled/reset while we switched locks.
            if (cc_cancel) {
                lock_class->lc_unlock(c_lock);
                goto skip;
            }

            // The callout cannot be stopped now.
            cc_cancel = true;
        }

        c_func(c_arg);
        if (((c_iflags & CALLOUT_RETURNUNLOCKED) == 0) &&
            (c_lock != NULL)) {
            lock_class->lc_unlock(c_lock);
        }
skip:
        LockMutex(&callout_mutex);
        cc_curr = NULL;
    }
    UnlockMutex(&callout_mutex);

    return;
}

static void callout_thread(void* dummy)
{
    while (1) {
        LockMutex(&callout_mutex);
        if (TAILQ_EMPTY(&cc_schedout)) {
            UnlockMutex(&callout_mutex);
            nn::os::WaitEvent(&callout_event);
        } else {
            callout_process_call_entries();
            UnlockMutex(&callout_mutex);
            callout_dispatch_call_entries();

            LockMutex(&callout_mutex);
            if (!TAILQ_EMPTY(&cc_schedout)) {
                struct   callout *entry = TAILQ_FIRST(&cc_schedout);
                sbintime_t c_time = entry->c_time;
                UnlockMutex(&callout_mutex);
                sbintime_t nowsbt = microseconds_to_sbt(nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMicroSeconds());
                if (c_time > nowsbt) {
                    nn::os::TimedWaitEvent(&callout_event,
                                           nn::TimeSpan::FromMicroSeconds(sbt_to_microseconds(c_time - nowsbt)));
                }
            } else {
                UnlockMutex(&callout_mutex);
            }
        }
    }
    return;
}

static void callout_sysinit(void* dummy)
{
    TAILQ_INIT(&cc_expire);
    TAILQ_INIT(&cc_schedout);
    nn::os::InitializeMutex(&callout_mutex, false, 0);
    nn::os::InitializeEvent(&callout_event, false, nn::os::EventClearMode_AutoClear);
    cc_curr    = NULL;
    cc_cancel  = false;
    cc_waiting = false;
    NN_ABORT_UNLESS(CreateNetworkThread(
                        &callout_thd,
                        &callout_thread,
                        NULL,
                        callout_stack,
                        sizeof(callout_stack),
                        NN_SYSTEM_THREAD_PRIORITY(socket, CoreStackCallout),
                        0) == 0);
    SetNetworkThreadNamePointer(&callout_thd, NN_SYSTEM_THREAD_NAME(socket, CoreStackCallout));
    StartNetworkThread(&callout_thd);
    return;
}

SYSINIT(callout_sysinit, SI_SUB_PORTINIT, SI_ORDER_ANY, callout_sysinit, NULL);

}

