﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/svc/svc_Base.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nn/drivers/eth/driverEth.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/kernel.h>
#include <siglo/network.h>

#include <drivers/eth/eth_if_drv.h>

using namespace nn::drivers::eth;

namespace nn { namespace bsdnet {

static uint8_t s_MacAddress[NN_NET_MAC_ADDRESS_SIZE] = { 0 };
static bool s_DevAttached = false;

static void FrameReceivedCallBack(const void *frameData, int frameBytes)
{
    eth_if_drv_receive_frame(frameData, frameBytes);
}

static void SendCompletedCallBack(void *frameData)
{
    eth_if_drv_send_complete();
}

static void InterfaceStateCallback(nn::drivers::eth::InterfaceState ifState)
{
    if ( !s_DevAttached && (ifState != InterfaceState_NoDevice))
    {
        GetMacAddress(s_MacAddress);
        ETH_IF_DRV_LOG_INFO("InterfaceStateCallback s_MacAddress %x%x%x%x%x%x\n",
                            s_MacAddress[0], s_MacAddress[1], s_MacAddress[2],
                            s_MacAddress[3], s_MacAddress[4], s_MacAddress[5]);
        eth_if_drv_attach(nn::bsdnet::s_MacAddress);
        s_DevAttached = true;
        RegisterSendCompletedCallBack(SendCompletedCallBack);
        RegisterFrameReceivedCallBack(FrameReceivedCallBack);
        if (ifState != InterfaceState_LinkUp) return;
    }
    switch (ifState)
    {
    case InterfaceState_NoDevice:
        eth_if_drv_detach();
        break;
    case InterfaceState_LinkDown:
        eth_if_drv_linkdown();
        break;
    case InterfaceState_LinkUp:
        eth_if_drv_linkup();
        break;
    default:
        break;
    }
}

static int EthIfDrvInitialize(void)
{
    int rv = 0;
    Result result;
    if ( !(result = Initialize()).IsSuccess())
    {
        rv = EIO;
    }
    return rv;
}

static int EthIfDrvFinalize(void)
{
    int rv = 0;
    nn::Result result;
    if ( !(result = Finalize()).IsSuccess())
    {
        rv = EIO;
    }
    return rv;
}

static bool EthIfDrvSend(int segments, const unsigned char **segment_buffers, size_t *segment_sizes)
{
    return nn::drivers::eth::Send(segments, (uint8_t **)segment_buffers, segment_sizes).IsSuccess();
}

} // namespace nn
} // namespace bsdnet

extern "C"
{

static void eth_if_drv_init(void *dummy)
{
    int rv = 0;
    do
    {
        if ((rv = nn::bsdnet::EthIfDrvInitialize()) < 0)
        {
            ETH_IF_DRV_CHECK_RV_ERROR(rv);
            break;
        }
        RegisterInterfaceStateCallBack(nn::bsdnet::InterfaceStateCallback);
    }while (false);
}

int eth_if_drv_finalize()
{
    int rv = 0;
    do
    {
        if ((rv = eth_if_drv_detach()) < 0)
        {
            ETH_IF_DRV_CHECK_RV_ERROR(rv);
            break;
        }
        if ((rv = nn::bsdnet::EthIfDrvFinalize()) < 0)
        {
            ETH_IF_DRV_CHECK_RV_ERROR(rv);
            break;
        }
    }while (false);
    return rv;
}

bool eth_if_drv_send(int segments, const unsigned char *segment_buffers[], size_t segment_sizes[])
{
    return nn::bsdnet::EthIfDrvSend(segments, segment_buffers, segment_sizes);
}

void eth_if_drv_log(const char *format, ...)
{
    char buffer[256];
    va_list ap;
    va_start(ap, format);
    nn::util::VSNPrintf(buffer, sizeof(buffer), format, ap);
    NN_SDK_LOG(buffer);
    va_end(ap);
}

SYSINIT(eth_if_drv_init, SI_SUB_LAST, SI_ORDER_ANY, eth_if_drv_init, NULL);

FORCE_LINK_THIS(_eth_if_drv_);

} //extern "C"


