﻿/*-
 * Copyright (c) 1999 Poul-Henning Kamp.
 * Copyright (c) 2009 James Gritton.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#ifndef _SYS_JAIL_H_
#define _SYS_JAIL_H_

#include <sys/types.h>
#include <sys/queue.h>
#include <sys/sysctl.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/_task.h>
#include <sys/osd.h>
#include <opt/opt_inet6.h>

struct ucred;
struct in_addr;
struct in6_addr;
struct sockaddr;
/*
 * This structure describes a prison.  It is pointed to by all struct
 * ucreds's of the inmates.  pr_ref keeps track of them and is used to
 * delete the struture when the last inmate is dead.
 *
 * Lock key:
 *   (a) allprison_lock
 *   (p) locked by pr_mtx
 *   (c) set only during creation before the structure is shared, no mutex
 *       required to read
 *   (d) set only during destruction of jail, no mutex needed
 */
struct prison {
    unsigned     pr_flags;                      /* (p) PR_* flags */
    struct mtx   pr_mtx;
    char         pr_hostname[MAXHOSTNAMELEN];   /* (p) jail hostname */
};

/* Flag bits set via options */
#define PR_PERSIST          0x00000001    /* Can exist without processes */
#define PR_HOST             0x00000002    /* Virtualize hostname et al */
#define PR_IP4_USER         0x00000004    /* Restrict IPv4 addresses */
#define PR_IP6_USER         0x00000008    /* Restrict IPv6 addresses */
#define PR_VNET             0x00000010    /* Virtual network stack */
#define PR_IP4_DISABLE      0x00000020    /* Disable IPv4 */
#define PR_IP6_DISABLE      0x00000040    /* Disable IPv6 */
#define PR_IP4_SADDRSEL     0x00000080    /* Do IPv4 src addr sel. or use the */
                                          /* primary jail address. */
#define PR_IP6_SADDRSEL     0x00000100    /* Do IPv6 src addr sel. or use the */
                                          /* primary jail address. */

/* Internal flag bits */
#define PR_REMOVE           0x01000000    /* In process of being removed */
#define PR_IP4              0x02000000    /* IPv4 restricted or disabled */
                                          /* by this jail or an ancestor */
#define PR_IP6              0x04000000    /* IPv6 restricted or disabled */
                                          /* by this jail or an ancestor */

/* Flags for pr_allow */
#define PR_ALLOW_SET_HOSTNAME   0x0001
#define PR_ALLOW_SYSVIPC        0x0002
#define PR_ALLOW_RAW_SOCKETS    0x0004
#define PR_ALLOW_CHFLAGS        0x0008
#define PR_ALLOW_MOUNT          0x0010
#define PR_ALLOW_QUOTAS         0x0020
#define PR_ALLOW_SOCKET_AF      0x0040
#define PR_ALLOW_MOUNT_DEVFS    0x0080
#define PR_ALLOW_MOUNT_NULLFS   0x0100
#define PR_ALLOW_MOUNT_ZFS      0x0200
#define PR_ALLOW_MOUNT_PROCFS   0x0400
#define PR_ALLOW_MOUNT_TMPFS    0x0800
#define PR_ALLOW_ALL            0x0fff

/*
 * OSD methods
 */
#define PR_METHOD_CREATE        0
#define PR_METHOD_GET           1
#define PR_METHOD_SET           2
#define PR_METHOD_CHECK         3
#define PR_METHOD_ATTACH        4
#define PR_MAXMETHOD            5

int jailed(struct ucred *cred);
int jailed_without_vnet(struct ucred *);
int prison_flag(struct ucred *, unsigned);
int prison_equal_ip4(struct prison *, struct prison *);
int prison_get_ip4(struct ucred *cred, struct in_addr *ia);
int prison_local_ip4(struct ucred *cred, struct in_addr *ia);
int prison_remote_ip4(struct ucred *cred, struct in_addr *ia);
int prison_check_ip4(struct ucred *cred, struct in_addr *ia);
int prison_saddrsel_ip4(struct ucred *, struct in_addr *);

#ifdef INET6
int prison_equal_ip6(struct prison *, struct prison *);
int prison_get_ip6(struct ucred *, struct in6_addr *);
int prison_local_ip6(struct ucred *, struct in6_addr *, int);
int prison_remote_ip6(struct ucred *, struct in6_addr *);
int prison_check_ip6(struct ucred *, struct in6_addr *);
int prison_saddrsel_ip6(struct ucred *, struct in6_addr *);
#endif

int prison_check_af(struct ucred *cred, int af);
int prison_if(struct ucred *cred, struct sockaddr *sa);

#endif /* !_SYS_JAIL_H_ */
