﻿/*-
 * Copyright (c) 1990, 1993
 *    The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *    @(#)filedesc.h    8.1 (Berkeley) 6/2/93
 * $FreeBSD$
 */

#ifndef _SYS_FILEDESC_H_
#define _SYS_FILEDESC_H_

#include <sys/caprights.h>
#include <sys/queue.h>
#include <sys/event.h>
#include <sys/lock.h>
#include <sys/priority.h>
#include <sys/seq.h>
#include <sys/sx.h>
#include <sys/file.h>
#include <machine/_limits.h>

struct filecaps {
    cap_rights_t     fc_rights;    /* per-descriptor capability rights */
    u_long          *fc_ioctls;    /* per-descriptor allowed ioctls */
    int16_t          fc_nioctls;   /* fc_ioctls array size */
    uint32_t         fc_fcntls;    /* per-descriptor allowed fcntls */
};

struct filedescent {
    struct file    *fde_file;    /* pointer to file structure for open file */
    struct filecaps fde_caps;    /* per-descriptor rights */
    uint8_t         fde_flags;   /* per-process open file flags */
    seq_t           fde_seq;     /* keep file and caps in sync */
};
#define fde_rights      fde_caps.fc_rights
#define fde_fcntls      fde_caps.fc_fcntls
#define fde_ioctls      fde_caps.fc_ioctls
#define fde_nioctls     fde_caps.fc_nioctls
#define fde_change_size (offsetof(struct filedescent, fde_seq))

/*
 * This structure is used for the management of descriptors.  It may be
 * shared by multiple processes.
 */
#define NDSLOTTYPE    u_long

struct filedesc {
    struct      filedescent *fd_ofiles;     /* open files */
    int         fd_nfiles;                  /* number of open files allocated */
    NDSLOTTYPE *fd_map;                     /* bitmap of free fds */
    int         fd_lastfile;                /* high-water mark of fd_ofiles */
    int         fd_freefile;                /* approx. next free file */
    u_short     fd_cmask;                   /* mask for file creation */
    u_short     fd_refcnt;                  /* thread reference count */
    u_short     fd_holdcnt;                 /* hold count on structure + mutex */
    struct      sx fd_sx;                   /* protects members of this struct */
};
#define    fd_seq(fdp, fd)    (&(fdp)->fd_ofiles[(fd)].fde_seq)

/*
 * Per-process open flags.
 */
#define UF_EXCLOSE      0x1                 /* auto-close on exec */

#ifdef _KERNEL

/* Flags for do_dup() */
#define DUP_FIXED       0x1                 /* Force fixed allocation. */
#define DUP_FCNTL       0x2                 /* fcntl()-style errors. */
#define DUP_CLOEXEC     0x4                 /* Atomically set FD_CLOEXEC. */

/* Lock a file descriptor table. */
#define FILEDESC_LOCK_INIT(fdp)         sx_init(&(fdp)->fd_sx, "filedesc structure")
#define FILEDESC_LOCK_DESTROY(fdp)      sx_destroy(&(fdp)->fd_sx)
#define FILEDESC_LOCK(fdp)              (&(fdp)->fd_sx)
#define FILEDESC_XLOCK(fdp)             sx_xlock(&(fdp)->fd_sx)
#define FILEDESC_XUNLOCK(fdp)           sx_xunlock(&(fdp)->fd_sx)
#define FILEDESC_SLOCK(fdp)             sx_slock(&(fdp)->fd_sx)
#define FILEDESC_SUNLOCK(fdp)           sx_sunlock(&(fdp)->fd_sx)
#define FILEDESC_LOCK_ASSERT(fdp)       sx_assert(&(fdp)->fd_sx, SX_LOCKED | SX_NOTRECURSED)
#define FILEDESC_XLOCK_ASSERT(fdp)      sx_assert(&(fdp)->fd_sx, SX_XLOCKED | SX_NOTRECURSED)
#define FILEDESC_UNLOCK_ASSERT(fdp)     sx_assert(&(fdp)->fd_sx, SX_UNLOCKED)

struct thread;

int  falloc(struct thread *td, struct file **resultfp, int *resultfd, int flags);
void fdclose(struct filedesc *fdp, struct file *fp, int idx, struct thread *td);
int  fdshutdown_all(struct filedesc* fdp, int* pcount, int forced);
int  fget_unlocked(
        struct filedesc *fdp, int fd, cap_rights_t *needrightsp,
        int needfcntl, struct file **fpp, cap_rights_t *haverightsp);
int  fget_write(struct thread *td, int fd, cap_rights_t *rightsp, struct file **fpp);
int  fget_read(struct thread *td, int fd, cap_rights_t *rightsp, struct file **fpp);
int  fget(struct thread *td, int fd, cap_rights_t *rightsp, struct file **fpp);

/* Requires a FILEDESC_{S,X}LOCK held and returns without a ref. */
static __inline struct file *
fget_locked(struct filedesc *fdp, int fd)
{
    FILEDESC_LOCK_ASSERT(fdp);
    if (fd < 0 || fd > fdp->fd_lastfile)
        return (NULL);
    return (fdp->fd_ofiles[fd].fde_file);
}

#endif /* _KERNEL */

#endif /* !_SYS_FILEDESC_H_ */
