﻿/*-
 * Copyright (c) 1980, 1986, 1993
 *    The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *    @(#)route.h    8.4 (Berkeley) 1/9/95
 * $FreeBSD$
 */

#ifndef _BSD_INTERNAL_NET_ROUTE_H_
#define _BSD_INTERNAL_NET_ROUTE_H_

#include      <sys/counter.h>
#include_next <net/route.h>

/*
 * Kernel resident routing tables.
 *
 * The routing tables are initialized when interface addresses
 * are set by making entries for all directly connected interfaces.
 */

/*
 * A route consists of a destination address, a reference
 * to a routing entry, and a reference to an llentry.
 * These are often held by protocols in their control
 * blocks, e.g. inpcb.
 */
struct route {
    struct	rtentry *ro_rt;
    struct	llentry *ro_lle;
    struct	in_ifaddr *ro_ia;
    int		ro_flags;
    struct	sockaddr ro_dst;
};

#ifndef RNF_NORMAL
#include <net/radix.h>
#ifdef RADIX_MPATH
#include <net/radix_mpath.h>
#endif
#endif

extern u_int rt_numfibs;                    /* number of usable routing tables */
extern u_int rt_add_addr_allfibs;           /* Announce interfaces to all fibs */

#if defined(_KERNEL) || defined(_WANT_RTENTRY)
struct rtentry {
    struct    radix_node rt_nodes[2];       /* tree glue, and other values */
    /*
     * XXX struct rtentry must begin with a struct radix_node (or two!)
     * because the code does some casts of a 'struct radix_node *'
     * to a 'struct rtentry *'
     */
#define rt_key(r)       (*((struct sockaddr **)(&(r)->rt_nodes->rn_key)))
#define rt_mask(r)      (*((struct sockaddr **)(&(r)->rt_nodes->rn_mask)))
    struct          sockaddr *rt_gateway;   /* value */
    struct          ifnet *rt_ifp;          /* the answer: interface to use */
    struct          ifaddr *rt_ifa;         /* the answer: interface address to use */
    int             rt_flags;               /* up/down?, host/net */
    int             rt_refcnt;              /* # held references */
    u_int           rt_fibnum;              /* which FIB */
    u_long          rt_mtu;                 /* MTU for this path */
    u_long          rt_weight;              /* absolute weight */
    u_long          rt_expire;              /* lifetime for route, e.g. redirect */
#define rt_endzero  rt_pksent
    counter_u64_t   rt_pksent;              /* packets sent using this route */
    struct mtx      rt_mtx;                 /* mutex for routing entry */
};
#endif /* _KERNEL || _WANT_RTENTRY */

#ifdef _KERNEL

#define RT_LINK_IS_UP(ifp)    (!((ifp)->if_capabilities & IFCAP_LINKSTATE) \
                              || (ifp)->if_link_state == LINK_STATE_UP)

#define RT_LOCK_INIT(_rt) \
        mtx_init(&(_rt)->rt_mtx, "rtentry", NULL, MTX_DEF | MTX_DUPOK)
#define RT_LOCK(_rt)            mtx_lock(&(_rt)->rt_mtx)
#define RT_UNLOCK(_rt)          mtx_unlock(&(_rt)->rt_mtx)
#define RT_LOCK_DESTROY(_rt)    mtx_destroy(&(_rt)->rt_mtx)
#define RT_LOCK_ASSERT(_rt)     mtx_assert(&(_rt)->rt_mtx, MA_OWNED)

#define RT_UNLOCK_COND(_rt)                                 \
     do {                                                   \
        if (mtx_owned(&(_rt)->rt_mtx))                      \
            mtx_unlock(&(_rt)->rt_mtx);                     \
     } while (0)


#define RT_ADDREF(_rt)                                      \
     do {                                                   \
        RT_LOCK_ASSERT(_rt);                                \
        KASSERT((_rt)->rt_refcnt >= 0,                      \
                ("negative refcnt %d", (_rt)->rt_refcnt));  \
        (_rt)->rt_refcnt++;                                 \
    } while (0)

#define RT_REMREF(_rt)                                      \
    do {                                                    \
        RT_LOCK_ASSERT(_rt);                                \
        KASSERT((_rt)->rt_refcnt > 0,                       \
                ("bogus refcnt %d", (_rt)->rt_refcnt));     \
        (_rt)->rt_refcnt--;                                 \
    } while (0)

#define RTFREE_LOCKED(_rt)                                  \
    do {                                                    \
        if ((_rt)->rt_refcnt <= 1)                          \
           rtfree(_rt);                                     \
        else {                                              \
           RT_REMREF(_rt);                                  \
           RT_UNLOCK(_rt);                                  \
        }                                                   \
        /* guard against invalid refs */                    \
        _rt = 0;                                            \
    } while (0)

#define RTFREE(_rt)                                         \
    do {                                                    \
        RT_LOCK(_rt);                                       \
        RTFREE_LOCKED(_rt);                                 \
    } while (0)

#define RO_RTFREE(_ro)                                      \
    do {                                                    \
        if ((_ro)->ro_rt) {                                 \
            if ((_ro)->ro_flags & RT_NORTREF) {             \
                (_ro)->ro_flags &= ~RT_NORTREF;             \
                (_ro)->ro_rt = NULL;                        \
            } else {                                        \
                RT_LOCK((_ro)->ro_rt);                      \
                RTFREE_LOCKED((_ro)->ro_rt);                \
            }                                               \
        }                                                   \
    } while (0)

struct radix_node_head *rt_tables_get_rnh(int, int);

struct ifmultiaddr;

void    rt_ieee80211msg(struct ifnet *, int, void *, size_t);
void    rt_ifannouncemsg(struct ifnet *, int);
void    rt_ifmsg(struct ifnet *);
void    rt_missmsg(int, struct rt_addrinfo *, int, int);
void    rt_missmsg_fib(int, struct rt_addrinfo *, int, int, int);
void    rt_newaddrmsg(int, struct ifaddr *, int, struct rtentry *);
void    rt_newaddrmsg_fib(int, struct ifaddr *, int, struct rtentry *, int);
int     rt_addrmsg(int, struct ifaddr *, int);
int     rt_routemsg(int, struct ifnet *ifp, int, struct rtentry *, int);
void    rt_newmaddrmsg(int, struct ifmultiaddr *);
int     rt_setgate(struct rtentry *, struct sockaddr *, struct sockaddr *);
void    rt_maskedcopy(struct sockaddr *, struct sockaddr *, struct sockaddr *);

int     rtsock_addrmsg(int, struct ifaddr *, int);
int     rtsock_routemsg(int, struct ifnet *ifp, int, struct rtentry *, int);

/*
 * Note the following locking behavior:
 *
 *    rtalloc_ign() and rtalloc() return ro->ro_rt unlocked
 *
 *    rtalloc1() returns a locked rtentry
 *
 *    rtfree() and RTFREE_LOCKED() require a locked rtentry
 *
 *    RTFREE() uses an unlocked entry.
 */

int     rtexpunge(struct rtentry *);
void    rtfree(struct rtentry *);
int     rt_check(struct rtentry **, struct rtentry **, struct sockaddr *);

/* XXX MRT COMPAT VERSIONS THAT SET UNIVERSE to 0 */
/* Thes are used by old code not yet converted to use multiple FIBS */
int     rt_getifa(struct rt_addrinfo *);
void    rtalloc_ign(struct route *ro, u_long ignflags);
void    rtalloc(struct route *ro); /* XXX deprecated, use rtalloc_ign(ro, 0) */
struct  rtentry *rtalloc1(struct sockaddr *, int, u_long);
int     rtinit(struct ifaddr *, int, int);
int     rtioctl(u_long, caddr_t);
void    rtredirect(struct sockaddr *, struct sockaddr *,
        struct sockaddr *, int, struct sockaddr *);
int     rtrequest(int, struct sockaddr *,
        struct sockaddr *, struct sockaddr *, int, struct rtentry **);

/* XXX MRT NEW VERSIONS THAT USE FIBs
 * For now the protocol indepedent versions are the same as the AF_INET ones
 * but this will change..
 */
int     rt_getifa_fib(struct rt_addrinfo *, u_int fibnum);
void    rtalloc_ign_fib(struct route *ro, u_long ignflags, u_int fibnum);
void    rtalloc_fib(struct route *ro, u_int fibnum);
struct  rtentry *rtalloc1_fib(struct sockaddr *, int, u_long, u_int);
int     rtioctl_fib(u_long, caddr_t, u_int);
void    rtredirect_fib(struct sockaddr *, struct sockaddr *,
        struct sockaddr *, int, struct sockaddr *, u_int);
int     rtrequest_fib(int, struct sockaddr *,
        struct sockaddr *, struct sockaddr *, int, struct rtentry **, u_int);
int     rtrequest1_fib(int, struct rt_addrinfo *, struct rtentry **, u_int);

#include <sys/eventhandler.h>
typedef void (*rtevent_arp_update_fn)(void *, struct rtentry *, uint8_t *, struct sockaddr *);
typedef void (*rtevent_redirect_fn)(void *, struct rtentry *, struct rtentry *, struct sockaddr *);
/* route_arp_update_event is no longer generated; see arp_update_event */
EVENTHANDLER_DECLARE(route_arp_update_event, rtevent_arp_update_fn);
EVENTHANDLER_DECLARE(route_redirect_event, rtevent_redirect_fn);
#endif

#endif
