﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include "ntd-test-tls.h"

extern void TestSecondFile(void);

#define MIN_ALIGN nn::os::ThreadStackAlignment

#define STACK_SIZE (3 * MIN_ALIGN)

const size_t            ThreadStackSize = STACK_SIZE;                 // スレッド操作スレッドのスタックサイズ
static NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
static NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック

static nn::os::ThreadType      g_Thread1;
static nn::os::ThreadType      g_Thread2;

extern thread_local short   GlobalShortFirstFile;
extern thread_local int     GlobalIntFirstFile;
static thread_local char    StaticChar = 's';
static thread_local char    *StaticPointer = (char *)"SecondFile";
thread_local int WeakTlsFirstStrongTlsSecond = -1234;
__attribute__((weak)) thread_local int StrongTlsFirstWeakTlsSecond = -4321;
thread_local double         GlobalDoubleSecondFile = 2.0;
thread_local float          GlobalFloatSecondFile;

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#if ULONG_MAX == 0xffffffff
#define TPOFF_K 8
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TPOFF_K 16
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif

static void TestLocal ()
{
    TESTLOG("Entering TestLocal()");

    thread_local short GlobalShortFirstFile = 24;

    TESTCASE_MESSAGE(GlobalShortFirstFile == 24, "GlobalShortFirstFile in TestLocal(): %d", GlobalShortFirstFile);

    TESTLOG("Exiting TestLocal()");
}

/*---------------------------------------------------------------------------*
  Name:         TestThread

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    name of thread as string.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/


static void TestThread(void *arg)
{
    char *self = (char *)__get_tp();
    TESTLOG("TestThread arg is %s", (char *)arg);
    TESTLOG("%s TP is %p", (char *)arg, self);
    int strcmp_result = strcmp(StaticPointer, "SecondFile");
    int char_compare = (StaticChar == 's') ? 1 : 0;

    TESTCASE_MESSAGE(GlobalShortFirstFile == 0, "GlobalShortFirstFile when initialized: %d", GlobalShortFirstFile);
    TESTCASE_MESSAGE(GlobalIntFirstFile == 100, "GlobalIntFirstFile when initialized: %d", GlobalIntFirstFile);
    TESTCASE_MESSAGE(char_compare == 1, "StaticChar char_compare when initialized: %d", char_compare);
    TESTCASE_MESSAGE(strcmp_result == 0, "StaticPointer strcmp result when initialized: %d", strcmp_result);
    TESTCASE_MESSAGE(WeakTlsFirstStrongTlsSecond == -1234, "WeakTlsFirstStrongTlsSecond when initialized: %d", WeakTlsFirstStrongTlsSecond);
    TESTCASE_MESSAGE(StrongTlsFirstWeakTlsSecond == 4321, "StrongTlsFirstWeakTlsSecond when initialized: %d", StrongTlsFirstWeakTlsSecond);
    TESTCASE_MESSAGE(GlobalDoubleSecondFile == 2.0, "GlobalDoubleSecondFile when initialized: %f", GlobalDoubleSecondFile);
    TESTCASE_MESSAGE(GlobalFloatSecondFile == 0.0, "GlobalFloatSecondFile when initialized: %f", GlobalFloatSecondFile);

    TestLocal();

    TESTCASE_MESSAGE(GlobalShortFirstFile == 0, "GlobalShortFirstFile at thread exit: %d", GlobalShortFirstFile);

    TESTLOG("%s is exiting", (char *)arg);
}

/*---------------------------------------------------------------------------*
  Name:         TestSecondFile

  Description:  This is the main function of the demo.

                This demo displays Hello! on the main thread and sub-threads.

  Arguments:    None.

  Returns:      Always zero.

 *---------------------------------------------------------------------------*/
extern void TestSecondFile()
{
    nn::Result      result;

    result = nn::os::CreateThread( &g_Thread1, TestThread, (void *)"TestThreadThree", g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread1." );

    TESTLOG("g_Thread1 = 0x%08zX", (size_t)&g_Thread1);

    result = nn::os::CreateThread( &g_Thread2, TestThread, (void *)"TestThreadFour", g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread2." );

    TESTLOG("g_Thread2 = 0x%08zX", (size_t)&g_Thread2);

    // スレッドの実行を開始する
    nn::os::StartThread( &g_Thread1 );
    TESTLOG("g_Thread1 started");
    nn::os::StartThread( &g_Thread2 );
    TESTLOG("g_Thread2 started");

    TestThread((void *)"main_thread second file");
    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );
    TESTLOG("g_Thread1 did wait");
    nn::os::WaitThread( &g_Thread2 );
    TESTLOG("g_Thread2 did wait");

    // スレッドを破棄する
    nn::os::DestroyThread( &g_Thread1 );
    TESTLOG("g_Thread1 destroyed");
    nn::os::DestroyThread( &g_Thread2 );
    TESTLOG("g_Thread2 destroyed");
}

