﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include "ntd-test-tls.h"

extern void TestSecondFile(void);

#define MIN_ALIGN nn::os::ThreadStackAlignment

#define STACK_SIZE (3 * MIN_ALIGN)

const size_t            ThreadStackSize = STACK_SIZE;                 // スレッド操作スレッドのスタックサイズ
static NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
static NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック

static nn::os::ThreadType      g_Thread1;
static nn::os::ThreadType      g_Thread2;

thread_local short          GlobalShortFirstFile;
thread_local int            GlobalIntFirstFile = 100;
static thread_local char    StaticChar = 'f';
static thread_local char    *StaticPointer = (char *)"FirstFile";
__attribute__((weak)) thread_local int WeakTlsFirstStrongTlsSecond = 1234;
thread_local int StrongTlsFirstWeakTlsSecond = 4321;
extern thread_local double  GlobalDoubleSecondFile;
extern thread_local float   GlobalFloatSecondFile;

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#if ULONG_MAX == 0xffffffff
#define TPOFF_K 8
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TPOFF_K 16
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif

static void TestLocal ()
{
    TESTLOG("Entering TestLocal()");

    thread_local short GlobalShortFirstFile = 23;

    TESTCASE_MESSAGE(GlobalShortFirstFile == 23, "GlobalShortFirstFile in TestLocal(): %d", GlobalShortFirstFile);

    TESTLOG("Exiting TestLocal()");
}

/*---------------------------------------------------------------------------*
  Name:         TestThread

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    name of thread as string.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/


static void TestThread(void *arg)
{
    char *self = (char *)__get_tp();
    TESTLOG("TestThread arg is %s", (char *)arg);
    TESTLOG("%s TP is %p", (char *)arg, self);
    int strcmp_result = strcmp(StaticPointer, "FirstFile");
    int char_compare = (StaticChar == 'f') ? 1 : 0;

    TESTCASE_MESSAGE(GlobalShortFirstFile == 0, "GlobalShortFirstFile when initialized: %d", GlobalShortFirstFile);
    TESTCASE_MESSAGE(GlobalIntFirstFile == 100, "GlobalIntFirstFile when initialized: %d", GlobalIntFirstFile);
    TESTCASE_MESSAGE(char_compare == 1, "StaticChar char_compare when initialized: %d", char_compare);
    TESTCASE_MESSAGE(strcmp_result == 0, "StaticPointer strcmp result when initialized: %d", strcmp_result);
    TESTCASE_MESSAGE(WeakTlsFirstStrongTlsSecond == -1234, "WeakTlsFirstStrongTlsSecond when initialized: %d", WeakTlsFirstStrongTlsSecond);
    TESTCASE_MESSAGE(StrongTlsFirstWeakTlsSecond == 4321, "StrongTlsFirstWeakTlsSecond when initialized: %d", StrongTlsFirstWeakTlsSecond);
    TESTCASE_MESSAGE(GlobalDoubleSecondFile == 2.0, "GlobalDoubleSecondFile when initialized: %f", GlobalDoubleSecondFile);
    TESTCASE_MESSAGE(GlobalFloatSecondFile == 0.0, "GlobalFloatSecondFile when initialized: %f", GlobalFloatSecondFile);

    TestLocal();

    TESTCASE_MESSAGE(GlobalShortFirstFile == 0, "GlobalShortFirstFile at thread exit: %d", GlobalShortFirstFile);

    TESTLOG("%s is exiting", (char *)arg);
}


/*---------------------------------------------------------------------------*
  Name:         main

  Description:  This is the main function of the demo.

                This demo displays Hello! on the main thread and sub-threads.

  Arguments:    None.

  Returns:      Always zero.

 *---------------------------------------------------------------------------*/
extern "C" void nnMain()
{
    nn::Result      result;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);

    // スレッドを生成する
    result = nn::os::CreateThread( &g_Thread1, TestThread, (void *)"TestThreadOne", g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread1." );

    TESTLOG("g_Thread1 = 0x%08zX", (size_t)&g_Thread1);

    result = nn::os::CreateThread( &g_Thread2, TestThread, (void *)"TestThreadTwo", g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread2." );

    TESTLOG("g_Thread2 = 0x%08zX", (size_t)&g_Thread2);

    // スレッドの実行を開始する
    nn::os::StartThread( &g_Thread1 );
    TESTLOG("g_Thread1 started");
    nn::os::StartThread( &g_Thread2 );
    TESTLOG("g_Thread2 started");

    TestSecondFile();

    TestThread((void *)"main_thread first file");
    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );
    TESTLOG("g_Thread1 did wait");
    nn::os::WaitThread( &g_Thread2 );
    TESTLOG("g_Thread2 did wait");

    // スレッドを破棄する
    nn::os::DestroyThread( &g_Thread1 );
    TESTLOG("g_Thread1 destroyed");
    nn::os::DestroyThread( &g_Thread2 );
    TESTLOG("g_Thread2 destroyed");

    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}

