﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include "ntd-test-tls.h"

#define MIN_ALIGN nn::os::ThreadStackAlignment

#define STACK_SIZE (3 * MIN_ALIGN)

const size_t            ThreadStackSize = STACK_SIZE;                 // スレッド操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック

nn::os::ThreadType      g_Thread1;
nn::os::ThreadType      g_Thread2;

int                     GameMinus;
thread_local short      GamePlus;
thread_local char       GamePlus1 = 1;
#if TLS_ALIGNMENT >= 4
thread_local int        GamePlus100 = 100;
#if TLS_ALIGNMENT == 4
__attribute__((aligned(TLS_ALIGNMENT))) thread_local float     GamePlusDouble = 2.0;
#else
__attribute__((aligned(TLS_ALIGNMENT))) thread_local double     GamePlusDouble = 2.0;
#endif
thread_local float      GamePlusFloat;
#endif
static size_t tdata_align_abs, tbss_align_abs, tdata_align_rel, tbss_align_rel, tls_segment_align;

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#if ULONG_MAX == 0xffffffff
#define TPOFF_K 8
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TPOFF_K 16
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif

typedef struct type_tls_data {
    void **dtv;
    void *tls_allocation;
    size_t tls_allocation_size;
    void *tls_dtor_list;
    void **dtv_copy;
} tls_data;

#define GET_TLS_DATA(TP)	((type_tls_data*)(TP + TPOFF_K - sizeof(type_tls_data)))

static void GameUpdatePlusMinus()
{
    GameMinus -= 5;;
    GamePlus += 5;
    GamePlus1 += 1;
#if TLS_ALIGNMENT >= 4
    GamePlus100 += 100;
    GamePlusDouble += 5.0;
    GamePlusFloat -= 5.0;
#endif
}

/*---------------------------------------------------------------------------*
  Name:         TestThreadOne

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    name of thread as string.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/


static void TestThreadOne(void *arg)
{
    int i;
    char *self = (char *)__get_tp();
    tls_data *td = GET_TLS_DATA(self);
    void *module_one_mem = td->dtv[1];
    void *aligned_one_mem = (void *)((((uintptr_t)module_one_mem + (tls_segment_align - 1)) & ~(tls_segment_align - 1)));
    unsigned char *mem;
    int gold_tpoff = (TLS_ALIGNMENT > TPOFF_K) ? TLS_ALIGNMENT : TPOFF_K;
    TESTLOG("TestThreadOne arg is %s", (char *)arg);
    TESTLOG("TestThreadOne TP is %p", self);
    TESTLOG("TestThreadOne td->dtv[1] is %p (should be %p)", module_one_mem, aligned_one_mem);

    TESTCASE_MESSAGE(module_one_mem == aligned_one_mem, "%p != %p", module_one_mem, aligned_one_mem);
    TESTCASE_MESSAGE(GamePlus == 0, "GamePlus when initialized: %d (expected 0)", GamePlus);
    TESTCASE_MESSAGE(GamePlus1 == 1, "GamePlus1 when initialized: %d (expected 1)", GamePlus1);
#if TLS_ALIGNMENT >= 4
    TESTCASE_MESSAGE(GamePlus100 == 100, "GamePlus100 when initialized: %d (expected 100)", GamePlus100);
    TESTCASE_MESSAGE(GamePlusDouble == 2.0, "GamePlusDouble when initialized: %d (expected 2.0)", GamePlusDouble);
    TESTCASE_MESSAGE(GamePlusFloat == 0.0, "GamePlusFloat when initialized: %f (expected 0.0)", GamePlusFloat);
#endif

    char *cp = &GamePlus1;
#if TLS_ALIGNMENT >= 4
    int *ip = &GamePlus100;
#if TLS_ALIGNMENT == 4
    float *dp = &GamePlusDouble;
#else
    double *dp = &GamePlusDouble;
#endif
#endif
    short *sp = &GamePlus;
#if TLS_ALIGNMENT >= 4
    float *fp = &GamePlusFloat;
#endif

#if TLS_ALIGNMENT >= 4
    if (((size_t)cp < (size_t)ip) && ((size_t)cp < (size_t)dp)) {
        TESTLOG("cp: %p", cp);
        if ((size_t)ip < (size_t)dp) {
            TESTLOG("ip: %p", ip);
            TESTLOG("dp: %p", dp);
        } else {
            TESTLOG("dp: %p", dp);
            TESTLOG("ip: %p", ip);
        }
    } else if (((size_t)ip < (size_t)cp) && ((size_t)ip < (size_t)dp)) {
        TESTLOG("ip: %p", ip);
        if ((size_t)cp < (size_t)dp) {
            TESTLOG("cp: %p", cp);
            TESTLOG("dp: %p", dp);
        } else {
            TESTLOG("dp: %p", dp);
            TESTLOG("cp: %p", cp);
        }
    } else {
        TESTLOG("dp: %p", dp);
        if ((size_t)cp < (size_t)ip) {
            TESTLOG("cp: %p", cp);
            TESTLOG("ip: %p", ip);
        } else {
            TESTLOG("ip: %p", ip);
            TESTLOG("cp: %p", cp);
        }
    }
    if ((size_t)sp < (size_t)fp) {
        TESTLOG("sp: %p", sp);
        TESTLOG("fp: %p", fp);
    } else {
        TESTLOG("fp: %p", fp);
        TESTLOG("sp: %p", sp);
    }
    TESTCASE_MESSAGE(((size_t)sp > (size_t)dp), "TLS .tdata and .tbss unexpectedly out of order");
    TESTCASE_MESSAGE(((size_t)cp == ((size_t)self + (size_t)gold_tpoff)), "TLS variable GamePlus1 has wrong TLS address");
    TESTCASE_MESSAGE((((size_t)cp < (size_t)ip) && ((size_t)ip < (size_t)dp)), "TLS .tdata variables unexpectedly out of order");
    TESTCASE_MESSAGE(((size_t)sp < (size_t)fp), "TLS .tbss variables unexpectedly out of order");
    mem = (unsigned char *)(cp);
    mem += (sizeof(char));
    mem = (unsigned char *)((((uintptr_t)mem + (4 - 1)) & ~(4 - 1)));
    TESTCASE_MESSAGE(((size_t)ip == (size_t)mem), "TLS variable GamePlus100 has misaligned address");
    mem += (sizeof(int));
    mem = (unsigned char *)((((uintptr_t)mem + (TLS_ALIGNMENT - 1)) & ~(TLS_ALIGNMENT - 1)));
    TESTCASE_MESSAGE(((size_t)dp == (size_t)mem), "TLS variable GamePlusDouble has misaligned address");
#if TLS_ALIGNMENT == 4
    mem += (sizeof(float));
#else
    mem += (sizeof(double));
#endif
    mem = (unsigned char *)((((uintptr_t)mem + (tbss_align_abs - 1)) & ~(tbss_align_abs - 1)));
    TESTCASE_MESSAGE(((size_t)sp == (size_t)mem), "TLS variable GamePlus has misaligned address");
    mem += (sizeof(short));
    mem = (unsigned char *)((((uintptr_t)mem + (4 - 1)) & ~(4 - 1)));
    TESTCASE_MESSAGE(((size_t)fp == (size_t)mem), "TLS variable GamePlusFloat has misaligned address");
#else
    TESTLOG("cp: %p", cp);
    TESTLOG("sp: %p", sp);
    TESTCASE_MESSAGE(((size_t)sp > (size_t)cp), "TLS .tdata and .tbss unexpectedly out of order");
    TESTCASE_MESSAGE(((size_t)cp == ((size_t)self + (size_t)gold_tpoff)), "TLS variable GamePlus1 has wrong TLS address");
    mem = (unsigned char *)(cp);
    mem += (sizeof(char));
    mem = (unsigned char *)((((uintptr_t)mem + (tbss_align_abs - 1)) & ~(tbss_align_abs - 1)));
    TESTCASE_MESSAGE(((size_t)sp == (size_t)mem), "TLS variable GamePlus has misaligned address");
#endif

    for (i = 0; i < 5; i++)
    {
        TESTLOG("Hello! from TestThreadOne.");
        // normal global data
        GameMinus--;
        // thread local storage data
        GamePlus++;
#if TLS_ALIGNMENT >= 4
        GamePlus100++;
#endif
        GameUpdatePlusMinus();
    }
    TESTCASE_MESSAGE(GamePlus == 30, "GamePlus at thread exit: %d (expected 30)", GamePlus);
    TESTCASE_MESSAGE(GamePlus1 == 6, "GamePlus1 at thread exit: %d (expected 6)", GamePlus1);
#if TLS_ALIGNMENT >= 4
    TESTCASE_MESSAGE(GamePlus100 == 605, "GamePlus100 at thread exit: %d (expected 605)", GamePlus100);
    TESTCASE_MESSAGE(GamePlusDouble == 27.0, "GamePlusDouble at thread exit: %f (expected 27.0)", GamePlusDouble);
    TESTCASE_MESSAGE(GamePlusFloat == -25.0, "GamePlusFloat at thread exit: %f (expected -25.0)", GamePlusFloat);
#endif
}


/*---------------------------------------------------------------------------*
  Name:         TestThreadTwo

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    name of thread as string.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/

static void TestThreadTwo(void *arg)
{
    int i;
    char *self = (char *)__get_tp();
    tls_data *td = GET_TLS_DATA(self);
    void *module_one_mem = td->dtv[1];
    void *aligned_one_mem = (void *)((((uintptr_t)module_one_mem + (tls_segment_align - 1)) & ~(tls_segment_align - 1)));
    unsigned char *mem;
    int gold_tpoff = (TLS_ALIGNMENT > TPOFF_K) ? TLS_ALIGNMENT : TPOFF_K;
    TESTLOG("TestThreadTwo arg is %s", (char *)arg);
    TESTLOG("TestThreadTwo TP is %p", self);
    TESTLOG("TestThreadTwo td->dtv[1] is %p (should be %p)", module_one_mem, aligned_one_mem);

    TESTCASE_MESSAGE(module_one_mem == aligned_one_mem, "%p != %p", module_one_mem, aligned_one_mem);
    TESTCASE_MESSAGE(GamePlus == 0, "GamePlus when initialized: %d (expected 0)", GamePlus);
    TESTCASE_MESSAGE(GamePlus1 == 1, "GamePlus1 when initialized: %d (expected 1)", GamePlus1);
#if TLS_ALIGNMENT >= 4
    TESTCASE_MESSAGE(GamePlus100 == 100, "GamePlus100 when initialized: %d (expected 100)", GamePlus100);
    TESTCASE_MESSAGE(GamePlusDouble == 2.0, "GamePlusDouble when initialized: %d (expected 2.0)", GamePlusDouble);
    TESTCASE_MESSAGE(GamePlusFloat == 0.0, "GamePlusFloat when initialized: %f (expected 0.0)", GamePlusFloat);
#endif

    char *cp = &GamePlus1;
#if TLS_ALIGNMENT >= 4
    int *ip = &GamePlus100;
#if TLS_ALIGNMENT == 4
    float *dp = &GamePlusDouble;
#else
    double *dp = &GamePlusDouble;
#endif
#endif
    short *sp = &GamePlus;
#if TLS_ALIGNMENT >= 4
    float *fp = &GamePlusFloat;
#endif

#if TLS_ALIGNMENT >= 4
    if (((size_t)cp < (size_t)ip) && ((size_t)cp < (size_t)dp)) {
        TESTLOG("cp: %p", cp);
        if ((size_t)ip < (size_t)dp) {
            TESTLOG("ip: %p", ip);
            TESTLOG("dp: %p", dp);
        } else {
            TESTLOG("dp: %p", dp);
            TESTLOG("ip: %p", ip);
        }
    } else if (((size_t)ip < (size_t)cp) && ((size_t)ip < (size_t)dp)) {
        TESTLOG("ip: %p", ip);
        if ((size_t)cp < (size_t)dp) {
            TESTLOG("cp: %p", cp);
            TESTLOG("dp: %p", dp);
        } else {
            TESTLOG("dp: %p", dp);
            TESTLOG("cp: %p", cp);
        }
    } else {
        TESTLOG("dp: %p", dp);
        if ((size_t)cp < (size_t)ip) {
            TESTLOG("cp: %p", cp);
            TESTLOG("ip: %p", ip);
        } else {
            TESTLOG("ip: %p", ip);
            TESTLOG("cp: %p", cp);
        }
    }
    if ((size_t)sp < (size_t)fp) {
        TESTLOG("sp: %p", sp);
        TESTLOG("fp: %p", fp);
    } else {
        TESTLOG("fp: %p", fp);
        TESTLOG("sp: %p", sp);
    }
    TESTCASE_MESSAGE(((size_t)sp > (size_t)dp), "TLS .tdata and .tbss unexpectedly out of order");
    TESTCASE_MESSAGE(((size_t)cp == ((size_t)self + (size_t)gold_tpoff)), "TLS variable GamePlus1 has wrong TLS address");
    TESTCASE_MESSAGE((((size_t)cp < (size_t)ip) && ((size_t)ip < (size_t)dp)), "TLS .tdata variables unexpectedly out of order");
    TESTCASE_MESSAGE(((size_t)sp < (size_t)fp), "TLS .tbss variables unexpectedly out of order");
    mem = (unsigned char *)(cp);
    mem += (sizeof(char));
    mem = (unsigned char *)((((uintptr_t)mem + (4 - 1)) & ~(4 - 1)));
    TESTCASE_MESSAGE(((size_t)ip == (size_t)mem), "TLS variable GamePlus100 has misaligned address");
    mem += (sizeof(int));
    mem = (unsigned char *)((((uintptr_t)mem + (TLS_ALIGNMENT - 1)) & ~(TLS_ALIGNMENT - 1)));
    TESTCASE_MESSAGE(((size_t)dp == (size_t)mem), "TLS variable GamePlusDouble has misaligned address");
#if TLS_ALIGNMENT == 4
    mem += (sizeof(float));
#else
    mem += (sizeof(double));
#endif
    mem = (unsigned char *)((((uintptr_t)mem + (tbss_align_abs - 1)) & ~(tbss_align_abs - 1)));
    TESTCASE_MESSAGE(((size_t)sp == (size_t)mem), "TLS variable GamePlus has misaligned address");
    mem += (sizeof(short));
    mem = (unsigned char *)((((uintptr_t)mem + (4 - 1)) & ~(4 - 1)));
    TESTCASE_MESSAGE(((size_t)fp == (size_t)mem), "TLS variable GamePlusFloat has misaligned address");
#else
    TESTLOG("cp: %p", cp);
    TESTLOG("sp: %p", sp);
    TESTCASE_MESSAGE(((size_t)sp > (size_t)cp), "TLS .tdata and .tbss unexpectedly out of order");
    TESTCASE_MESSAGE(((size_t)cp == ((size_t)self + (size_t)gold_tpoff)), "TLS variable GamePlus1 has wrong TLS address");
    mem = (unsigned char *)(cp);
    mem += (sizeof(char));
    mem = (unsigned char *)((((uintptr_t)mem + (tbss_align_abs - 1)) & ~(tbss_align_abs - 1)));
    TESTCASE_MESSAGE(((size_t)sp == (size_t)mem), "TLS variable GamePlus has misaligned address");
#endif

    for (i = 0; i < 5; i++)
    {
        TESTLOG("Hello! from TestThreadTwo.");
        // normal global data
        GameMinus--;
        // thread local storage data
        GamePlus++;
#if TLS_ALIGNMENT >= 4
        GamePlus100++;
#endif
        GameUpdatePlusMinus();
    }
    TESTCASE_MESSAGE(GamePlus == 30, "GamePlus at thread exit: %d (expected 30)", GamePlus);
    TESTCASE_MESSAGE(GamePlus1 == 6, "GamePlus1 at thread exit: %d (expected 6)", GamePlus1);
#if TLS_ALIGNMENT >= 4
    TESTCASE_MESSAGE(GamePlus100 == 605, "GamePlus100 at thread exit: %d (expected 605)", GamePlus100);
    TESTCASE_MESSAGE(GamePlusDouble == 27.0, "GamePlusDouble at thread exit: %f (expected 27.0)", GamePlusDouble);
    TESTCASE_MESSAGE(GamePlusFloat == -25.0, "GamePlusFloat at thread exit: %f (expected -25.0)", GamePlusFloat);
#endif
}

/*---------------------------------------------------------------------------*
  Name:         main

  Description:  This is the main function of the demo.

                This demo displays Hello! on the main thread and sub-threads.

  Arguments:    None.

  Returns:      Always zero.

 *---------------------------------------------------------------------------*/
extern "C" void nnMain()
{
    nn::Result      result;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);
    TESTLOG("__EX_start = %p", (unsigned char *)__EX_start);
    TESTLOG("__tdata_start = %p", (unsigned char *)__tdata_start);
    TESTLOG("__tdata_end = %p", (unsigned char *)__tdata_end);
    TESTLOG("__tdata_align_abs = %p", (unsigned char *)__tdata_align_abs);
    TESTLOG("__tdata_align_rel = %p", (unsigned char *)__tdata_align_rel);
    TESTLOG("__tbss_start = %p", (unsigned char *)__tbss_start);
    TESTLOG("__tbss_end = %p", (unsigned char *)__tbss_end);
    TESTLOG("__tbss_align_abs = %p", (unsigned char *)__tbss_align_abs);
    TESTLOG("__tbss_align_rel = %p", (unsigned char *)__tbss_align_rel);
    TESTLOG("TLS_ALIGNMENT is %d\n", TLS_ALIGNMENT);
    tdata_align_abs = (size_t)__tdata_align_abs - (size_t)__EX_start;
    tdata_align_rel = (size_t)__tdata_align_rel - (size_t)__tdata_start;
    TESTCASE_MESSAGE(tdata_align_abs == tdata_align_rel, "Inconsistent alignment for ,tdata abs and rel");
    tbss_align_abs = (size_t)__tbss_align_abs - (size_t)__EX_start;
    tbss_align_rel = (size_t)__tbss_align_rel - (size_t)__tbss_start;
    TESTCASE_MESSAGE(tbss_align_abs == tbss_align_rel, "Inconsistent alignment for ,tbss abs and rel");
    // ld and lld make an empty .tdata section  have alignment 1; gold does 0 alignment
    if (tdata_align_abs > 1) {
        TESTCASE_MESSAGE(tdata_align_abs >= tbss_align_abs, "Unexpected .tbss alignment greater than .tdata alignment");
    }
    tls_segment_align = (tdata_align_abs > tbss_align_abs) ? tdata_align_abs : tbss_align_abs;
    TESTCASE_MESSAGE(tls_segment_align == TLS_ALIGNMENT, "Unexpected TLS segment alignment");

    TESTCASE_MESSAGE(GameMinus == 0, "GameMinus when initialized: %d (expected 0)", GameMinus);
    // スレッドを生成する
    result = nn::os::CreateThread( &g_Thread1, TestThreadOne, (void *)"1", g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread1." );

    TESTLOG("g_Thread1 = 0x%08zX", (size_t)&g_Thread1);

    result = nn::os::CreateThread( &g_Thread2, TestThreadTwo, (void *)"2", g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread2." );

    TESTLOG("g_Thread2 = 0x%08zX", (size_t)&g_Thread2);

    TestThreadOne((void *)"main_thread");
    // スレッドの実行を開始する
    nn::os::StartThread( &g_Thread1 );
    TESTLOG("g_Thread1 started");
    nn::os::StartThread( &g_Thread2 );
    TESTLOG("g_Thread2 started");

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );
    TESTLOG("g_Thread1 did wait");
    nn::os::WaitThread( &g_Thread2 );
    TESTLOG("g_Thread2 did wait");

    // スレッドを破棄する
    nn::os::DestroyThread( &g_Thread1 );
    TESTLOG("g_Thread1 destroyed");
    nn::os::DestroyThread( &g_Thread2 );
    TESTLOG("g_Thread2 destroyed");

    TESTCASE_MESSAGE(GameMinus == -90, "GameMinus at main exit: %d (expected -90)", GameMinus);

    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}

