﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <new>
#include <nn/fs.h>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/ro.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/os/os_SdkMemoryAllocatorForThreadLocal.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include "ntd-test-tls.h"

typedef void* (*NroEntryPoint)(void);
extern "C" void *CallNro1(void);
extern "C" void *CallNro2(void);
extern "C" void *CallNro3(void);
extern "C" void *CallNro4(void);
extern "C" void *CallNro5(void);
extern "C" void *CallNro6(void);
extern "C" void *CallNro7(void);
extern "C" void *CallNro8(void);
extern "C" void *CallNro9(void);
extern "C" void *CallNro10(void);
extern "C" void *CallNro11(void);
extern "C" void *CallNro12(void);
extern "C" void *CallNro13(void);
extern "C" void *CallNro14(void);
extern "C" void *CallNro15(void);
extern "C" void *CallNro16(void);
extern "C" void *CallNro17(void);
extern "C" void *CallNro18(void);
extern "C" void *CallNro19(void);
extern "C" void *CallNro20(void);
extern "C" void *CallNro21(void);
extern "C" void *CallNro22(void);
extern "C" void *CallNro23(void);
extern "C" void *CallNro24(void);
extern "C" void *CallNro25(void);
extern "C" void *CallNro26(void);
extern "C" void *CallNro27(void);
extern "C" void *CallNro28(void);
extern "C" void *CallNro29(void);
extern "C" void *CallNro30(void);
extern "C" void *CallNro31(void);
extern "C" void *CallNro32(void);
extern "C" void *CallNro33(void);
extern "C" void *CallNro34(void);
extern "C" void *CallNro35(void);
extern "C" void *CallNro36(void);
extern "C" void *CallNro37(void);
extern "C" void *CallNro38(void);
extern "C" void *CallNro39(void);
extern "C" void *CallNro40(void);

extern "C" void *__get_tp(void);

extern void *__dso_handle;  // this is defined in Chris/Sources/Libraries/rocrt/rocrt.cpp

#define MIN_ALIGN nn::os::ThreadStackAlignment

#define STACK_SIZE (3 * MIN_ALIGN)
#define NUM_THREADS 6

const size_t            ThreadStackSize = STACK_SIZE;                 // スレッド操作スレッドのスタックサイズ
typedef struct thread_stack {
    NN_OS_ALIGNAS_THREAD_STACK char  stack[ ThreadStackSize ];   // 1つ目のスレッドのスタック
} thread_stack;
thread_stack            g_ThreadStack[NUM_THREADS];

nn::os::ThreadType      g_Thread[NUM_THREADS];

typedef void (*ThreadFuncPtr)(void *);
static void TestThread1(void *arg);
static void TestThread2(void *arg);
static void TestThread3(void *arg);
static void TestThread4(void *arg);
static void TestThread5(void *arg);
static void TestThread6(void *arg);

static ThreadFuncPtr TestThread[NUM_THREADS] = {
    TestThread1,
    TestThread2,
    TestThread3,
    TestThread4,
    TestThread5,
    TestThread6
};

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#if ULONG_MAX == 0xffffffff
#define TPOFF_K 8
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TPOFF_K 16
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif
#define NRR_NAME "rom:/.nrr/" TEST_ROOT_NAME ".nrr"

#define NRO1_NAME "rom:/nro/" STRINGIZE_VALUE_OF(TLS_NRO1_NAME) ".nro"
#define NRO2_NAME "rom:/nro/" STRINGIZE_VALUE_OF(TLS_NRO2_NAME) ".nro"
#define NRO3_NAME "rom:/nro/" STRINGIZE_VALUE_OF(TLS_NRO3_NAME) ".nro"
#define NRO4_NAME "rom:/nro/" STRINGIZE_VALUE_OF(TLS_NRO4_NAME) ".nro"
#define NRO5_NAME "rom:/nro/" STRINGIZE_VALUE_OF(TLS_NRO5_NAME) ".nro"
#define NRO6_NAME "rom:/nro/" STRINGIZE_VALUE_OF(TLS_NRO6_NAME) ".nro"

static const size_t MaxFileSize  = 0x400000;
static long         num_dtrs_to_call = 0;
static char*        cacheBuffer;
static size_t       cacheSize = 0;
static int          num_HogeConstructorCalls = 0;
static int          num_HogeDestructorCalls = 0;

extern thread_local int nro_var1;
extern thread_local double nro_var2;
extern thread_local double nro_var3;
extern thread_local short nro_var4;
extern thread_local char *nro_var5;
extern thread_local char nro_string[20];
extern thread_local char *nro_var6;

static size_t ReadAll(void** pOut, size_t bufferSize, const char* path, size_t alignment)
{
    void *p;
    nn::Result result;
    nn::fs::FileHandle file;
    result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    int64_t fileSize;
    int32_t test;
    result = nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_LESS(fileSize, static_cast<int64_t>(bufferSize));
    test = static_cast<int32_t>(fileSize);
    NN_ASSERT_EQUAL(static_cast<int64_t>(test), fileSize);

    p = aligned_alloc(alignment, test);
    NN_ASSERT_NOT_NULL(p);
    TESTLOG("ReadAll: path \"%s\" required %d byts to read its content", path, test);

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, file, 0, p, bufferSize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_EQUAL(static_cast<int64_t>(readSize), fileSize);
    *pOut = p;

    nn::fs::CloseFile(file);

    return readSize;
}

static void Initialize_Ro_And_Rom()
{
    nn::Result result;

    TESTLOG("Initialize_Ro_And_Rom calling nn::ro::Initialize()");
    nn::ro::Initialize();

    // ファイルシステムのメタデータキャッシュに必要なバッファサイズを修得
    TESTLOG("Initialize_Ro_And_Rom calling nn::fs::QueryMountRomCacheSize(&cacheSize)");
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());
    TESTLOG("Initialize_Ro_And_Rom cacheSize is %zd", cacheSize);

    // キャッシュバッファを確保
    TESTLOG("Initialize_Ro_And_Rom calling new(std::nothrow) char[cacheSize]");
    cacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(cacheBuffer);

    // ファイルシステムをマウント
    TESTLOG("Initialize_Ro_And_Rom calling nn::fs::MountRom(\"rom\", cacheBuffer, cacheSize)");
    result = nn::fs::MountRom("rom", cacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

static void Finalize_Ro_And_Rom()
{
    TESTLOG("Finalize_Ro_And_Rom calling nn::ro::Finalize()");
    nn::ro::Finalize();

    // アンマウントする
    TESTLOG("Finalize_Ro_And_Rom calling nn::fs::Unmount(\"rom\")");
    nn::fs::Unmount("rom");
    delete[] cacheBuffer;
    cacheBuffer = NULL;
}

class Hoge
{
public:
    nn::Result result;
    void* nro;
    void* bss;
    void* nrr;
    size_t imageSize;
    size_t bufferSize;
    size_t nrrSize;
    nn::ro::Module module;
    nn::ro::RegistrationInfo info;
    const char *nrr_name;
    const char *nro_name;
    const char *nro_entry_name;
    uintptr_t addr;

    Hoge(const char *NrrName, const char *NroName) {
        nrr_name = NrrName;
        nro_name = NroName;

        num_HogeConstructorCalls++;
        nrr = 0;
        TESTLOG("Hoge: calling ReadAll(&nrr, MaxFileSize, \"%s\")", NrrName);
        nrrSize = ReadAll(&nrr, MaxFileSize, NrrName, nn::os::MemoryPageSize);
        TESTLOG("load_nro_module nrrSize is %zd", nrrSize);

        // nrr を登録
        TESTLOG("Hoge: calling nn::ro::RegisterModuleInfo(&info, nrr)");
        result = nn::ro::RegisterModuleInfo(&info, nrr);
        NN_ASSERT(result.IsSuccess());

        nro = 0;
        TESTLOG("Hoge: calling ReadAll(nro, MaxFileSize, \"%s\")", NroName);
        imageSize = ReadAll(&nro, MaxFileSize, NroName, nn::os::MemoryPageSize);
        TESTLOG("load_nro_module imageSize is %zd", imageSize);

        TESTLOG("Hoge: calling nn::ro::GetBufferSize(&bufferSize, nro)");
        result = nn::ro::GetBufferSize(&bufferSize, nro);
        NN_ASSERT(result.IsSuccess());
        TESTLOG("Hoge: bufferSize for bss is %zd", bufferSize);
        if (bufferSize != 0)
        {
            TESTLOG("Hoge: calling bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize)");
            bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize);
        }
        else
        {
            TESTLOG("Hoge: setting bss = 0");
            bss = 0;
        }

        // nro をロード(シンボルは遅延解決)
        TESTLOG("Hoge: calling nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy)");
        result = nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy);
        NN_ASSERT(result.IsSuccess());
    }
    ~Hoge() {
        TESTLOG("~Hoge(): TP = 0x%08zX; app module __dso_handle = %p", (size_t)__get_tp(), __dso_handle);
        TESTLOG("~Hoge(): calling nn::ro::UnloadModule(&module)");
        nn::ro::UnloadModule(&module);
        TESTLOG("~Hoge(): calling nn::ro::UnregisterModuleInfo(&info)");
        nn::ro::UnregisterModuleInfo(&info);
        num_HogeDestructorCalls++;
    }
};


static void TestThread1(void *arg)
{
    NN_UNUSED(arg);
    int nro_var5_strcmp, nro_var6_strcmp;

    TESTLOG("TestThread1 in app module __dso_handle %p", __dso_handle);

    thread_local Hoge g_Hoge(NRR_NAME, NRO1_NAME);

    TESTLOG("nro_var1 = %d; expected 1 in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTCASE_MESSAGE(nro_var1 == 1, "expected 1 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTLOG("nro_var2 = %f; expected 2.0 in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTCASE_MESSAGE(nro_var2 == 2.0, "expected 2.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTLOG("nro_var3 = %f; expected 3.0 in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTCASE_MESSAGE(nro_var3 == 3.0, "expected 3.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTLOG("nro_var4 = %d; expected 4 in app module __dso_handle %p", nro_var4, __dso_handle);
    TESTCASE_MESSAGE(nro_var4 == 4, "expected 4 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var4, __dso_handle);
    nro_var5_strcmp = strcmp(nro_var5, "Five");
    TESTLOG("nro_var5_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var5_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    nro_var6_strcmp = strncmp(nro_var6, "six", 20);
    TESTLOG("nro_var6_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var6_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
}

static void TestThread2(void *arg)
{
    NN_UNUSED(arg);
    int nro_var5_strcmp, nro_var6_strcmp;

    TESTLOG("TestThread2 in app module __dso_handle %p", __dso_handle);

    thread_local Hoge g_Hoge(NRR_NAME, NRO2_NAME);

    TESTLOG("nro_var1 = %d; expected 11 in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTCASE_MESSAGE(nro_var1 == 11, "expected 11 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTLOG("nro_var2 = %f; expected 12.0 in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTCASE_MESSAGE(nro_var2 == 12.0, "expected 12.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTLOG("nro_var3 = %f; expected 13.0 in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTCASE_MESSAGE(nro_var3 == 13.0, "expected 13.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTLOG("nro_var4 = %d; expected 14 in app module __dso_handle %p", nro_var4, __dso_handle);
    TESTCASE_MESSAGE(nro_var4 == 14, "expected 14 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var4, __dso_handle);
    nro_var5_strcmp = strcmp(nro_var5, "Fifteen");
    TESTLOG("nro_var5_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var5_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    nro_var6_strcmp = strncmp(nro_var6, "sixteen", 20);
    TESTLOG("nro_var6_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var6_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
}

static void TestThread3(void *arg)
{
    NN_UNUSED(arg);
    int nro_var5_strcmp, nro_var6_strcmp;

    TESTLOG("TestThread3 in app module __dso_handle %p", __dso_handle);

    thread_local Hoge g_Hoge(NRR_NAME, NRO3_NAME);

    TESTLOG("nro_var1 = %d; expected 21 in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTCASE_MESSAGE(nro_var1 == 21, "expected 21 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTLOG("nro_var2 = %f; expected 22.0 in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTCASE_MESSAGE(nro_var2 == 22.0, "expected 22.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTLOG("nro_var3 = %f; expected 23.0 in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTCASE_MESSAGE(nro_var3 == 23.0, "expected 23.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTLOG("nro_var4 = %d; expected 24 in app module __dso_handle %p", nro_var4, __dso_handle);
    TESTCASE_MESSAGE(nro_var4 == 24, "expected 24 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var4, __dso_handle);
    nro_var5_strcmp = strcmp(nro_var5, "Twenty-Five");
    TESTLOG("nro_var5_strcmp= %d; expected 0 in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var5_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    nro_var6_strcmp = strncmp(nro_var6, "Twenty-Six", 20);
    TESTLOG("nro_var6_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var6_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
}

static void TestThread4(void *arg)
{
    NN_UNUSED(arg);
    int nro_var5_strcmp, nro_var6_strcmp;

    TESTLOG("TestThread4 in app module __dso_handle %p", __dso_handle);

    thread_local Hoge g_Hoge(NRR_NAME, NRO4_NAME);

    TESTLOG("nro_var1 = %d; expected 31 in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTCASE_MESSAGE(nro_var1 == 31, "expected 31 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTLOG("nro_var2 = %f; expected 32.0 in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTCASE_MESSAGE(nro_var2 == 32.0, "expected 32.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTLOG("nro_var3 = %f; expected 33.0 in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTCASE_MESSAGE(nro_var3 == 33.0, "expected 33.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTLOG("nro_var4 = %d; expected 34 in app module __dso_handle %p", nro_var4, __dso_handle);
    TESTCASE_MESSAGE(nro_var4 == 34, "expected 34 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var4, __dso_handle);
    nro_var5_strcmp = strcmp(nro_var5, "Thirty-Five");
    TESTLOG("nro_var5_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var5_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    nro_var6_strcmp = strncmp(nro_var6, "Thirty-Six", 20);
    TESTLOG("nro_var6_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var6_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
}

static void TestThread5(void *arg)
{
    NN_UNUSED(arg);
    int nro_var5_strcmp, nro_var6_strcmp;
    int i;

    TESTLOG("TestThread5 in app module __dso_handle %p", __dso_handle);

    thread_local Hoge g_Hoge(NRR_NAME, NRO5_NAME);

    TESTLOG("nro_var1 = %d; expected 41 in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTCASE_MESSAGE(nro_var1 == 41, "expected 41 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTLOG("nro_var2 = %f; expected 42.0 in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTCASE_MESSAGE(nro_var2 == 42.0, "expected 42.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTLOG("nro_var3 = %f; expected 43.0 in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTCASE_MESSAGE(nro_var3 == 43.0, "expected 43.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTLOG("nro_var4 = %d; expected 44 in app module __dso_handle %p", nro_var4, __dso_handle);
    TESTCASE_MESSAGE(nro_var4 == 44, "expected 44 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var4, __dso_handle);
    nro_var5_strcmp = strcmp(nro_var5, "Forty-Five");
    TESTLOG("nro_var5_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var5_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    nro_var6_strcmp = strncmp(nro_var6, "Forty-Six", 20);
    TESTLOG("nro_var6_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var6_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
}

static void TestThread6(void *arg)
{
    NN_UNUSED(arg);
    int nro_var5_strcmp, nro_var6_strcmp;

    TESTLOG("TestThread6 in app module __dso_handle %p", __dso_handle);

    thread_local Hoge g_Hoge(NRR_NAME, NRO6_NAME);

    TESTLOG("nro_var1 = %d; expected 51 in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTCASE_MESSAGE(nro_var1 == 51, "expected 51 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var1, __dso_handle);
    TESTLOG("nro_var2 = %f; expected 52.0 in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTCASE_MESSAGE(nro_var2 == 52.0, "expected 52.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var2, __dso_handle);
    TESTLOG("nro_var3 = %f; expected 53.0 in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTCASE_MESSAGE(nro_var3 == 53.0, "expected 53.0 but got %f in the NSP's TestThread in app module __dso_handle %p", nro_var3, __dso_handle);
    TESTLOG("nro_var4 = %d; expected 54 in app module __dso_handle %p", nro_var4, __dso_handle);
    TESTCASE_MESSAGE(nro_var4 == 54, "expected 54 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var4, __dso_handle);
    nro_var5_strcmp = strcmp(nro_var5, "Fifty-Five");
    TESTLOG("nro_var5_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var5_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var5_strcmp, __dso_handle);
    nro_var6_strcmp = strncmp(nro_var6, "Fifty-Six", 20);
    TESTLOG("nro_var6_strcmp = %d; expected 0 in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
    TESTCASE_MESSAGE(nro_var6_strcmp == 0, "expected 0 but got %d in the NSP's TestThread in app module __dso_handle %p", nro_var6_strcmp, __dso_handle);
}

/*---------------------------------------------------------------------------*
  Name:         main

  Description:  This is the main function of the demo.

                This demo displays Hello! on the main thread and sub-threads.

  Arguments:    None.

  Returns:      Always zero.

 *---------------------------------------------------------------------------*/
extern "C" void nnMain()
{
    nn::Result      result;
    size_t i;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);

    Initialize_Ro_And_Rom();

    // スレッドを生成する
    TESTLOG("begin creating thread");
    for (i = 0; i < NUM_THREADS; i++) {
        result = nn::os::CreateThread( &(g_Thread[i]), TestThread[i], NULL, g_ThreadStack[i].stack, ThreadStackSize, nn::os::DefaultThreadPriority );
        NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread[%zd].", i );
        TESTLOG("g_Thread[%zd] = 0x%08zX created", i, (size_t)&(g_Thread[i]));

        TESTLOG("begin starting thread");
        num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
        TESTLOG("There are %ld registered destructors before starting threads.", num_dtrs_to_call);
        TESTCASE_MESSAGE(num_dtrs_to_call == 0, "Before calling nn::os::StartThread we expected 0 uncalled Destructor but we got %ld.", num_dtrs_to_call);
        // スレッドの実行を開始する
        nn::os::StartThread( &(g_Thread[i]) );
        TESTLOG("g_Thread[%zd] = 0x%08zX started", i, (size_t)&(g_Thread[i]));

        TESTLOG("begin waiting thread");
        num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
        TESTLOG("There are %ld registered destructors before waiting threads.", num_dtrs_to_call);
        // スレッドが終了するのを待つ
        nn::os::WaitThread( &(g_Thread[i]) );
        TESTLOG("g_Thread[%zd] = 0x%08zX did wait", i, (size_t)&(g_Thread[i]));

        TESTLOG("begin destroying thread");
        num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
        TESTLOG("There are %ld registered destructors before destroying threads.", num_dtrs_to_call);
        // スレッドを破棄する
        nn::os::DestroyThread( &(g_Thread[i]) );
        TESTLOG("g_Thread[%zd] = 0x%08zX destroyed", i, (size_t)&(g_Thread[i]));

        TESTLOG("done destroying thread");
    }

    Finalize_Ro_And_Rom();

    TESTLOG("num_HogeConstructorCalls = %d", num_HogeConstructorCalls);
    TESTCASE_MESSAGE(num_HogeConstructorCalls == NUM_THREADS, "Before exiting nnMain we expected %d calls to Hoge() but we got %ld.",
        NUM_THREADS, num_HogeConstructorCalls);

    TESTLOG("num_HogeDestructorCalls = %d", num_HogeDestructorCalls);
    TESTCASE_MESSAGE(num_HogeDestructorCalls == NUM_THREADS, "Before exiting nnMain we expected %d calls to ~Hoge() but we got %ld.",
        NUM_THREADS, num_HogeDestructorCalls);

    TESTLOG("nnMain calling __nnmusl_get_tls_dtors_status() with 0x%08zX", __dso_handle);
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "expected 0 but there were %ld uncalled destructors", num_dtrs_to_call);

    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}

