﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <new>
#include <nn/fs.h>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/ro.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/os/os_SdkMemoryAllocatorForThreadLocal.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include <inttypes.h>
#include "ntd-test-tls.h"

typedef void* (*NroEntryPoint)(void);
extern "C" void *CallNro(void);

extern "C" void *__get_tp(void);

extern void *__dso_handle;  // this is defined in Chris/Sources/Libraries/rocrt/rocrt.cpp

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#if ULONG_MAX == 0xffffffff
#define TPOFF_K 8
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TPOFF_K 16
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif
#define NRR_NAME "rom:/.nrr/" TEST_ROOT_NAME ".nrr"

#define NRO_NAME "rom:/nro/" STRINGIZE_VALUE_OF(TLS_NRO_NAME) ".nro"

static const size_t MaxFileSize  = 0x400000;
static int64_t      num_dtrs_to_call = 0;
static char*        cacheBuffer;
static size_t       cacheSize = 0;
static int          num_HogeConstructorCalls = 0;
static int          num_HogeDestructorCalls = 0;

static size_t ReadAll(void** pOut, size_t bufferSize, const char* path, size_t alignment)
{
    void *p;
    nn::Result result;
    nn::fs::FileHandle file;
    result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    int64_t fileSize;
    int32_t test;
    result = nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_LESS(fileSize, static_cast<int64_t>(bufferSize));
    test = static_cast<int32_t>(fileSize);
    NN_ASSERT_EQUAL(static_cast<int64_t>(test), fileSize);

    p = aligned_alloc(alignment, test);
    NN_ASSERT_NOT_NULL(p);
    TESTLOG("ReadAll: path \"%s\" required %d byts to read its content", path, test);

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, file, 0, p, bufferSize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_EQUAL(static_cast<int64_t>(readSize), fileSize);
    *pOut = p;

    nn::fs::CloseFile(file);

    return readSize;
}

static void Initialize_Ro_And_Rom()
{
    nn::Result result;

    TESTLOG("Initialize_Ro_And_Rom calling nn::ro::Initialize()");
    nn::ro::Initialize();

    // ファイルシステムのメタデータキャッシュに必要なバッファサイズを修得
    TESTLOG("Initialize_Ro_And_Rom calling nn::fs::QueryMountRomCacheSize(&cacheSize)");
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());
    TESTLOG("Initialize_Ro_And_Rom cacheSize is %zd", cacheSize);

    // キャッシュバッファを確保
    TESTLOG("Initialize_Ro_And_Rom calling new(std::nothrow) char[cacheSize]");
    cacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(cacheBuffer);

    // ファイルシステムをマウント
    TESTLOG("Initialize_Ro_And_Rom calling nn::fs::MountRom(\"rom\", cacheBuffer, cacheSize)");
    result = nn::fs::MountRom("rom", cacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

static void Finalize_Ro_And_Rom()
{
    TESTLOG("Finalize_Ro_And_Rom calling nn::ro::Finalize()");
    nn::ro::Finalize();

    // アンマウントする
    TESTLOG("Finalize_Ro_And_Rom calling nn::fs::Unmount(\"rom\")");
    nn::fs::Unmount("rom");
    delete[] cacheBuffer;
    cacheBuffer = NULL;
}

class Hoge
{
public:
    nn::Result result;
    void* nro;
    void* bss;
    void* nrr;
    size_t imageSize;
    size_t bufferSize;
    size_t nrrSize;
    nn::ro::Module module;
    nn::ro::RegistrationInfo info;
    void *nro_dso_handle;
    const char *nrr_name;
    const char *nro_name;

    Hoge(const char *NrrName, const char *NroName) {
        nrr_name = NrrName;
        nro_name = NroName;

        num_HogeConstructorCalls++;
        nrr = 0;
        TESTLOG("Hoge: calling ReadAll(&nrr, MaxFileSize, \"%s\")", NrrName);
        nrrSize = ReadAll(&nrr, MaxFileSize, NrrName, nn::os::MemoryPageSize);
        TESTLOG("load_nro_module nrrSize is %zd", nrrSize);

        // nrr を登録
        TESTLOG("Hoge: calling nn::ro::RegisterModuleInfo(&info, nrr)");
        result = nn::ro::RegisterModuleInfo(&info, nrr);
        NN_ASSERT(result.IsSuccess());

        nro = 0;
        TESTLOG("Hoge: calling ReadAll(nro, MaxFileSize, \"%s\")", NroName);
        imageSize = ReadAll(&nro, MaxFileSize, NroName, nn::os::MemoryPageSize);
        TESTLOG("load_nro_module imageSize is %zd", imageSize);

        TESTLOG("Hoge: calling nn::ro::GetBufferSize(&bufferSize, nro)");
        result = nn::ro::GetBufferSize(&bufferSize, nro);
        NN_ASSERT(result.IsSuccess());
        TESTLOG("Hoge: bufferSize for bss is %zd", bufferSize);
        if (bufferSize != 0)
        {
            TESTLOG("Hoge: calling bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize)");
            bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize);
        }
        else
        {
            TESTLOG("Hoge: setting bss = 0");
            bss = 0;
        }

        // nro をロード(シンボルは遅延解決)
        TESTLOG("Hoge: calling nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy)");
        result = nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy);
        NN_ASSERT(result.IsSuccess());
    }
    ~Hoge() {
        TESTLOG("~Hoge(): calling nn::ro::UnloadModule(&module)");
        nn::ro::UnloadModule(&module);
        TESTLOG("~Hoge(): calling nn::ro::UnregisterModuleInfo(&info)");
        nn::ro::UnregisterModuleInfo(&info);
        num_HogeDestructorCalls++;
    }
};

/*---------------------------------------------------------------------------*
  Name:         main

  Description:  This is the main function of the demo.

                This demo displays Hello! on the main thread and sub-threads.

  Arguments:    None.

  Returns:      Always zero.

 *---------------------------------------------------------------------------*/
extern "C" void nnMain()
{
    nn::Result      result;
    int i;
    void *nro_dso_handle;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);

    TESTLOG("app module __dso_handle %p", __dso_handle);
    nn::os::ThreadType *main_thread = nn::os::GetCurrentThread();
    TESTLOG("main_thread in the app module = 0x%08zX", (size_t)main_thread);

    Initialize_Ro_And_Rom();

    num_dtrs_to_call = __nnmusl_get_number_uncalled_tls_dtors (__dso_handle);
    TESTLOG("There are %" PRId64 " registered destructors in the app module before calling a constructor to load the NRO.", num_dtrs_to_call);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "Before loading the NRO via the constructor we expected 0 uncalled Destructor but we got %" PRId64 ".", num_dtrs_to_call);

    thread_local Hoge g_Hoge(NRR_NAME, NRO_NAME);

    num_dtrs_to_call = __nnmusl_get_number_uncalled_tls_dtors (__dso_handle);
    TESTLOG("There are %" PRId64 " registered destructors in the app module after calling a constructor to load the NRO.", num_dtrs_to_call);
    TESTCASE_MESSAGE(num_dtrs_to_call == 1, "After loading the NRO via the constructor we expected 1 uncalled Destructor but we got %" PRId64 ".", num_dtrs_to_call);

    TESTLOG("num_HogeConstructorCalls = %d", num_HogeConstructorCalls);
    TESTCASE_MESSAGE(num_HogeConstructorCalls == 1, "Before calling NRO's entrypoint we expected 1 call to Hoge() but we got %d.", num_HogeConstructorCalls);

    TESTLOG("num_HogeDestructorCalls = %d", num_HogeDestructorCalls);
    TESTCASE_MESSAGE(num_HogeDestructorCalls == 0, "Before NRO's entrypoint we expected 0 call to ~Hoge() but we got %d.", num_HogeDestructorCalls);

    nro_dso_handle = CallNro();

    TESTLOG("NRO module __dso_handle %p", nro_dso_handle);

    num_dtrs_to_call = __nnmusl_get_number_uncalled_tls_dtors (nro_dso_handle);
    TESTLOG("There are %" PRId64 " registered destructors in the NRO module after returning from CallNro().", num_dtrs_to_call);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "After returning from the NRO we expected 0 uncalled Destructors but we got %" PRId64 ".", num_dtrs_to_call);

    num_dtrs_to_call = __nnmusl_get_number_uncalled_tls_dtors (__dso_handle);
    TESTLOG("There are %" PRId64 " registered destructors in the app module after returning from CallNro().", num_dtrs_to_call);
    TESTCASE_MESSAGE(num_dtrs_to_call == 1, "After returning from the NRO and back in the app module we expected 1 uncalled Destructor but we got %" PRId64 ".", num_dtrs_to_call);

    if (num_dtrs_to_call) {
        // I know that the outstanding destructor will not call any more constructors
        // unload the module via the destructor
        __nnmusl_call_tls_dtors_for_module (__dso_handle);
    }

    Finalize_Ro_And_Rom();

    TESTLOG("num_HogeConstructorCalls = %d", num_HogeConstructorCalls);
    TESTCASE_MESSAGE(num_HogeConstructorCalls == 1, "Before exiting nnMain we expected 1 call to Hoge() but we got %d.", num_HogeConstructorCalls);

    TESTLOG("num_HogeDestructorCalls = %d", num_HogeDestructorCalls);
    TESTCASE_MESSAGE(num_HogeDestructorCalls == 1, "Before exiting nnMain we expected 1 call to ~Hoge() but we got %d.", num_HogeDestructorCalls);

    TESTLOG("nnMain calling __nnmusl_get_number_uncalled_tls_dtors() with %p for app module", __dso_handle);
    num_dtrs_to_call = __nnmusl_get_number_uncalled_tls_dtors (__dso_handle);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "expected 0 but there were %" PRId64 " uncalled destructors", num_dtrs_to_call);

    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}

