﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <new>
#include <nn/fs.h>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/ro.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/os/os_SdkMemoryAllocatorForThreadLocal.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include "ntd-test-tls.h"

extern void *CallNro(void);
extern void *__dso_handle;  // this is defined in Chris/Sources/Libraries/rocrt/rocrt.cpp

#define MIN_ALIGN nn::os::ThreadStackAlignment

#define STACK_SIZE (3 * MIN_ALIGN)

const size_t            ThreadStackSize = STACK_SIZE;                 // スレッド操作スレッドのスタックサイズ
static NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック

static nn::os::ThreadType      g_Thread1;

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#define NRO_ROOT_NAME STRINGIZE_VALUE_OF(TLS_NRO_NAME)
#if ULONG_MAX == 0xffffffff
#define TPOFF_K 8
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TPOFF_K 16
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif
#define NRR_NAME "rom:/.nrr/" TEST_ROOT_NAME ".nrr"
#define NR0_NAME "rom:/nro/" NRO_ROOT_NAME ".nro"


static size_t ReadAll(void* pOut, size_t bufferSize, const char* path)
{
    nn::Result result;
    nn::fs::FileHandle file;
    result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_LESS(fileSize, static_cast<int64_t>(bufferSize));

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, file, 0, pOut, bufferSize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_EQUAL(static_cast<int64_t>(readSize), fileSize);

    nn::fs::CloseFile(file);

    return readSize;
}

static const size_t MaxFileSize  = 0x400000;
static void* nro;
static void* bss;
static void* nrr;
static size_t imageSize;
static size_t bufferSize;
static size_t nrrSize;
static nn::ro::Module module;
static nn::ro::RegistrationInfo info;
static long            num_dtrs_to_call = 0;
static char* cacheBuffer;

static void load_nro_module()
{
    nn::Result result;

    TESTLOG("load_nro_module calling nn::ro::Initialize()");
    nn::ro::Initialize();

    // ファイルシステムのメタデータキャッシュに必要なバッファサイズを修得
    size_t cacheSize = 0;
    TESTLOG("load_nro_module calling nn::fs::QueryMountRomCacheSize(&cacheSize)");
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());
    TESTLOG("load_nro_module cacheSize is %zd", cacheSize);

    // キャッシュバッファを確保
    TESTLOG("load_nro_module calling new(std::nothrow) char[cacheSize]");
    cacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(cacheBuffer);

    // ファイルシステムをマウント
    TESTLOG("load_nro_module calling nn::fs::MountRom(\"rom\", cacheBuffer, cacheSize)");
    result = nn::fs::MountRom("rom", cacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // nrr を読み込み
    TESTLOG("load_nro_module calling nrr = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize)");
    nrr = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize);
    TESTLOG("load_nro_module nrr is %p and was allocated to be MaxFileSize = %zd", nrr, MaxFileSize);
    TESTLOG("load_nro_module calling ReadAll(nrr, MaxFileSize, \"%s\")", NRR_NAME);
    nrrSize = ReadAll(nrr, MaxFileSize, NRR_NAME);
    TESTLOG("load_nro_module nrrSize is %zd", nrrSize);

    // nrr を登録
    TESTLOG("load_nro_module calling nn::ro::RegisterModuleInfo(&info, nrr)");
    result = nn::ro::RegisterModuleInfo(&info, nrr);
    NN_ASSERT(result.IsSuccess());

    TESTLOG("load_nro_module calling nro = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize)");
    nro = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize);
    TESTLOG("load_nro_module nro is %p and was allocated to be MaxFileSize = %zd", nro, MaxFileSize);
    TESTLOG("load_nro_module calling ReadAll(nro, MaxFileSize, \"%s\")", NR0_NAME);
    imageSize = ReadAll(nro, MaxFileSize, NR0_NAME);
    TESTLOG("load_nro_module imageSize is %zd", imageSize);

    TESTLOG("load_nro_module calling nn::ro::GetBufferSize(&bufferSize, nro)");
    result = nn::ro::GetBufferSize(&bufferSize, nro);
    NN_ASSERT(result.IsSuccess());
    TESTLOG("load_nro_module bufferSize is %zd", bufferSize);
    if (bufferSize != 0)
    {
    TESTLOG("load_nro_module calling bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize)");
        bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize);
    }
    else
    {
    TESTLOG("load_nro_module setting bss = 0");
        bss = 0;
    }

    // nro をロード(シンボルは遅延解決)
    TESTLOG("load_nro_module calling nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy)");
    result = nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy);
    NN_ASSERT(result.IsSuccess());
}

static void unload_nro_module()
{
    TESTLOG("unload_nro_module calling nn::ro::UnloadModule(&module)");
    nn::ro::UnloadModule(&module);
    TESTLOG("unload_nro_module calling nn::ro::UnregisterModuleInfo(&info)");
    nn::ro::UnregisterModuleInfo(&info);
    TESTLOG("unload_nro_module calling nn::ro::Finalize()");
    nn::ro::Finalize();

    // アンマウントする
    TESTLOG("unload_nro_module calling nn::fs::Unmount(\"rom\")");
    nn::fs::Unmount("rom");
    delete[] cacheBuffer;
    cacheBuffer = NULL;
}

static int num_HogeDestructorCalls = 0;
class Hoge
{
public:
    int x;
    nn::os::ThreadType *Hoge_thread;
    void *this_obj;
    char *name = (char *)"<unitialized name>";
    void *nro_dso_handle;
    long hoge_num_dtrs_to_call = 0;
    Hoge() {
        Hoge_thread = nn::os::GetCurrentThread();
        this_obj = (void*)this;
        TESTLOG("Hello from class Hoge (%s) %d thread address 0x%08zX; this = 0x%08zX", name, x, (size_t)Hoge_thread, (size_t)this_obj);
        TESTLOG("Hoge calling load_nro_module()");
        load_nro_module();
        TESTLOG("Hoge after calling load_nro_module()");
        TESTLOG("Hoge calling CallNro()");
        nro_dso_handle = CallNro();
        TESTLOG("Hoge after calling CallNro()");
        TESTLOG("Hoge calling __nnmusl_get_tls_dtors_status() with 0x%08zX from NRO", nro_dso_handle);
        hoge_num_dtrs_to_call = __nnmusl_get_tls_dtors_status (nro_dso_handle);
        TESTCASE_MESSAGE(hoge_num_dtrs_to_call == 0, "there were %ld uncalled destructors", hoge_num_dtrs_to_call);
    }
    Hoge(char *func_name, int T) {
        name = func_name;
        x = T;
        Hoge_thread = nn::os::GetCurrentThread();
        this_obj = (void*)this;
        TESTLOG("Hello from class Hoge (%s) %d thread address 0x%08zX; this = 0x%08zX", name, x, (size_t)Hoge_thread, (size_t)this_obj);
        TESTLOG("Hoge calling load_nro_module()");
        load_nro_module();
        TESTLOG("Hoge after calling load_nro_module()");
        TESTLOG("Hoge calling CallNro()");
        nro_dso_handle = CallNro();
        TESTLOG("Hoge after calling CallNro()");
        TESTLOG("Hoge calling __nnmusl_get_tls_dtors_status() with 0x%08zX from NRO", nro_dso_handle);
        hoge_num_dtrs_to_call = __nnmusl_get_tls_dtors_status (nro_dso_handle);
        TESTCASE_MESSAGE(hoge_num_dtrs_to_call == 0, "there were %ld uncalled destructors", hoge_num_dtrs_to_call);
    }
    ~Hoge() {
        TESTLOG("Good-bye from class Hoge (%s) %d thread address 0x%08zX; this = 0x%08zX!", name, x, (size_t)Hoge_thread, (size_t)this_obj);
        TESTLOG("Before calling unload_nro_module __nnmusl_get_tls_dtors_status() with 0x%08zX", nro_dso_handle);
        hoge_num_dtrs_to_call = __nnmusl_get_tls_dtors_status (nro_dso_handle);
        TESTLOG("in ~Hoge(): got %ld uncalled destructors in the NRO", hoge_num_dtrs_to_call);
        TESTCASE_MESSAGE(hoge_num_dtrs_to_call == 0, "expected 0 but there were %ld uncalled destructors in the NRO", hoge_num_dtrs_to_call);
        unload_nro_module();
        num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
        TESTLOG("in ~Hoge(): got %ld uncalled destructors in the NSP", num_dtrs_to_call);
        TESTCASE_MESSAGE(num_dtrs_to_call == 1, "expected 1 but there were %ld uncalled destructors", num_dtrs_to_call);
        num_HogeDestructorCalls++;
    }
};

/*---------------------------------------------------------------------------*
  Name:         TestThread

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    name of thread as string.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/


static void TestThread(void *arg)
{
    thread_local Hoge g_Hoge((char *)"g_Hoge_0", 47);
}


/*---------------------------------------------------------------------------*
  Name:         main

  Description:  This is the main function of the demo.

                This demo displays Hello! on the main thread and sub-threads.

  Arguments:    None.

  Returns:      Always zero.

 *---------------------------------------------------------------------------*/
extern "C" void nnMain()
{
    nn::Result      result;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);

    // スレッドを生成する
    result = nn::os::CreateThread( &g_Thread1, TestThread, (void *)"TestThreadOne", g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread1." );

    TESTLOG("g_Thread1 = 0x%08zX", (size_t)&g_Thread1);

    // スレッドの実行を開始する
    nn::os::StartThread( &g_Thread1 );
    TESTLOG("g_Thread1 started");

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );
    TESTLOG("g_Thread1 did wait");

    // スレッドを破棄する
    nn::os::DestroyThread( &g_Thread1 );
    TESTLOG("g_Thread1 destroyed");

    TESTLOG("num_HogeDestructorCalls = %d", num_HogeDestructorCalls);
    TESTCASE_MESSAGE(num_HogeDestructorCalls == 1, "Before exiting nnMain we expected 1 call to ~Hoge() but we got %ld.",
        num_HogeDestructorCalls);

    TESTLOG("nnMain calling __nnmusl_get_tls_dtors_status() with 0x%08zX", __dso_handle);
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "expected 0 but there were %ld uncalled destructors", num_dtrs_to_call);

    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}

