﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <new>
#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/ro.h>
#include <pthread.h>
#include "ntd-test-tls.h"

extern void TestTls(void);
extern void *__dso_handle;  // this is defined in Chris/Sources/Libraries/rocrt/rocrt.cpp

static size_t tdata_align_abs, tbss_align_abs, tdata_align_rel, tbss_align_rel, tls_segment_align;

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#define NRO_ROOT_NAME STRINGIZE_VALUE_OF(TLS_NRO_NAME)
#if ULONG_MAX == 0xffffffff
#define TPOFF_K 8
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TPOFF_K 16
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif
#define NRR_NAME "rom:/.nrr/" TEST_ROOT_NAME ".nrr"
#define NR0_NAME "rom:/nro/" NRO_ROOT_NAME ".nro"

size_t ReadAll(void* pOut, size_t bufferSize, const char* path)
{
    nn::Result result;
    nn::fs::FileHandle file;
    result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_LESS(fileSize, static_cast<int64_t>(bufferSize));

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, file, 0, pOut, bufferSize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_EQUAL(static_cast<int64_t>(readSize), fileSize);

    nn::fs::CloseFile(file);

    return readSize;
}

extern "C" void nnMain()
{
    nn::Result result;
    static const size_t MaxFileSize  = 0x400000;
    void* nro;
    void* bss;
    void* nrr;
    size_t imageSize;
    size_t bufferSize;
    size_t nrrSize;
    nn::ro::Module module;
    nn::ro::RegistrationInfo info;
    long            num_dtrs_to_call = 0;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);
    TESTLOG("__EX_start = %p", (unsigned char *)__EX_start);
    TESTLOG("__tdata_start = %p", (unsigned char *)__tdata_start);
    TESTLOG("__tdata_end = %p", (unsigned char *)__tdata_end);
    TESTLOG("__tdata_align_abs = %p", (unsigned char *)__tdata_align_abs);
    TESTLOG("__tdata_align_rel = %p", (unsigned char *)__tdata_align_rel);
    TESTLOG("__tbss_start = %p", (unsigned char *)__tbss_start);
    TESTLOG("__tbss_end = %p", (unsigned char *)__tbss_end);
    TESTLOG("__tbss_align_abs = %p", (unsigned char *)__tbss_align_abs);
    TESTLOG("__tbss_align_rel = %p", (unsigned char *)__tbss_align_rel);
    tdata_align_abs = (size_t)__tdata_align_abs - (size_t)__EX_start;
    tdata_align_rel = (size_t)__tdata_align_rel - (size_t)__tdata_start;
    TESTCASE_MESSAGE(tdata_align_abs == tdata_align_rel, "Inconsistent alignment for ,tdata abs and rel");
    tbss_align_abs = (size_t)__tbss_align_abs - (size_t)__EX_start;
    tbss_align_rel = (size_t)__tbss_align_rel - (size_t)__tbss_start;
    TESTCASE_MESSAGE(tbss_align_abs == tbss_align_rel, "Inconsistent alignment for ,tbss abs and rel");
    // ld and lld make an empty .tdata section  have alignment 1; gold does 0 alignment
    if (tdata_align_abs > 1) {
        TESTCASE_MESSAGE(tdata_align_abs >= tbss_align_abs, "Unexpected .tbss alignment greater than .tdata alignment");
    }
    tls_segment_align = (tdata_align_abs > tbss_align_abs) ? tdata_align_abs : tbss_align_abs;
    TESTLOG("tls_segment_align = %ld", tls_segment_align);

    TESTLOG("nnMain calling nn::ro::Initialize()");
    nn::ro::Initialize();

    // ファイルシステムのメタデータキャッシュに必要なバッファサイズを修得
    size_t cacheSize = 0;
    TESTLOG("nnMain calling nn::fs::QueryMountRomCacheSize(&cacheSize)");
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());

    // キャッシュバッファを確保
    TESTLOG("nnMain calling new(std::nothrow) char[cacheSize]");
    char* cacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(cacheBuffer);

    // ファイルシステムをマウント
    TESTLOG("nnMain calling nn::fs::MountRom(\"rom\", cacheBuffer, cacheSize)");
    result = nn::fs::MountRom("rom", cacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // nrr を読み込み
    TESTLOG("nnMain calling nrr = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize)");
    nrr = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize);
    TESTLOG("nnMain calling ReadAll(nrr, MaxFileSize, \"%s\")", NRR_NAME);
    nrrSize = ReadAll(nrr, MaxFileSize, NRR_NAME);

    // nrr を登録
    TESTLOG("nnMain calling nn::ro::RegisterModuleInfo(&info, nrr)");
    result = nn::ro::RegisterModuleInfo(&info, nrr);
    NN_ASSERT(result.IsSuccess());

    TESTLOG("nnMain calling nro = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize)");
    nro = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize);
    TESTLOG("nnMain calling ReadAll(nro, MaxFileSize, \"%s\")", NR0_NAME);
    imageSize = ReadAll(nro, MaxFileSize, NR0_NAME);

    TESTLOG("nnMain calling nn::ro::GetBufferSize(&bufferSize, nro)");
    result = nn::ro::GetBufferSize(&bufferSize, nro);
    NN_ASSERT(result.IsSuccess());
    if (bufferSize != 0)
    {
    TESTLOG("nnMain calling bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize)");
        bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize);
    }
    else
    {
    TESTLOG("nnMain setting bss = 0");
        bss = 0;
    }

    // nro をロード(シンボルは遅延解決)
    TESTLOG("nnMain calling nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy)");
    result = nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy);
    NN_ASSERT(result.IsSuccess());

    // 動的モジュール内のシンボルを呼び出す
    TESTLOG("nnMain calling TestTls()");
    TestTls();
    TESTLOG("nnMain after calling TestTls()");

    TESTLOG("nnMain calling nn::ro::UnloadModule(&module)");
    nn::ro::UnloadModule(&module);
    TESTLOG("nnMain calling nn::ro::UnregisterModuleInfo(&info)");
    nn::ro::UnregisterModuleInfo(&info);
    TESTLOG("nnMain calling nn::ro::Finalize()");
    nn::ro::Finalize();

    // アンマウントする
    TESTLOG("nnMain calling nn::fs::Unmount(\"rom\")");
    nn::fs::Unmount("rom");
    delete[] cacheBuffer;
    cacheBuffer = NULL;

    TESTLOG("nnMain calling __nnmusl_get_tls_dtors_status() with 0x%08zX", __dso_handle);
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "there were %ld uncalled destructors\n", num_dtrs_to_call);

    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}
