﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include "ntd-test-tls.h"

typedef struct ___oasis_nnmusl_dso_modules_
{
    int version;                    // This is our version number to specify how we are encoding this data.
                                    // Debugger will check this and bail if an unknown version is detected.
                                    // For example, Rynda 1.5.X has a new fancy TLS system and no longer uses the DTV, version is incremented,
                                    // and the debugger no longer attempts to locate NRO TLS variables.
    int nModules;                   // This will be used to know how many modules are listed in the table.
                                    // E.g. if the table includes [MainMod, NRO1, NRO2] then nModules = 3.
    int nElements;                  // This will be used to know how big modAddresses.  Note that modAddresses[0] holds the index of the last NSO.
                                    // modAddresses[1] will hold the first module, the app.
    size_t reserved1;               // Reserved for future use. Maybe a pointer to another structure...
    size_t reserved2;               // Reserved for future use. Maybe a pointer to another structure...
    size_t modAddresses[];          // This is the current array structure that contains each modules load address (and it matches the DTV index).
} ___oasis_nnmusl_dso_modules_;

extern ___oasis_nnmusl_dso_modules_ *__oasis_nnmusl_dso_modules;

extern void *__dso_handle;  // this is defined in Chris/Sources/Libraries/rocrt/rocrt.cpp

#define NRO_INT_NAME STRINGIZE_VALUE_OF(NRO_INT)
#define NRO_ENTRY_NAME STRINGIZE_VALUE_OF(NRO_ENTRY)
#define NRO_HOGE_CLASS_NAME STRINGIZE_VALUE_OF(NRO_HOGE_CLASS)
#define NRO_HOGE_OBJ_NAME STRINGIZE_VALUE_OF(NRO_HOGE_OBJ)
#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_NRO_NAME)
#if ULONG_MAX == 0xffffffff
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif

extern "C" void *__get_tp(void);

extern "C" void *NRO_ENTRY(void);

#define MIN_ALIGN nn::os::ThreadStackAlignment

#define STACK_SIZE (3 * MIN_ALIGN)

static const size_t            ThreadStackSize = STACK_SIZE;                 // スレッド操作スレッドのスタックサイズ
static NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
static nn::os::ThreadType      g_Thread1;

thread_local int Dummy = 1;
thread_local int NRO_INT = NRO_INT_VALUE;

extern thread_local int nsp_int; // defined to 123456789;

static int num_HogeDestructorCalls = 0;
class NRO_HOGE_CLASS
{
public:
    int x;
    nn::os::ThreadType *Hoge_thread;
    void *this_obj;
    char *name = (char *)"<unitialized name>";
    long hoge_num_dtrs_to_call = 0;
    NRO_HOGE_CLASS() {
        Hoge_thread = nn::os::GetCurrentThread();
        this_obj = (void*)this;
        TESTLOG("Hello from class %s (%s) %d thread address 0x%08zX; this = 0x%08zX; __dso_handle = %p", NRO_HOGE_CLASS_NAME, name, x, (size_t)Hoge_thread, (size_t)this_obj, __dso_handle);
    }
    NRO_HOGE_CLASS(char *func_name, int T) {
        name = func_name;
        x = T;
        Hoge_thread = nn::os::GetCurrentThread();
        this_obj = (void*)this;
        TESTLOG("Hello from class %s (%s) %d thread address 0x%08zX; this = 0x%08zX; __dso_handle = %p", NRO_HOGE_CLASS_NAME, name, x, (size_t)Hoge_thread, (size_t)this_obj, __dso_handle);
    }
    ~NRO_HOGE_CLASS() {
        TESTLOG("Good-bye from class %s (%s) %d thread address 0x%08zX; this = 0x%08zX!  __dso_handle = %p", NRO_HOGE_CLASS_NAME, name, x, (size_t)Hoge_thread, (size_t)this_obj, __dso_handle);
        TESTLOG("~%s(): TP = 0x%08zX, __dso_handle = %p", NRO_HOGE_CLASS_NAME, __get_tp(), __dso_handle);
        hoge_num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
        TESTLOG("in ~%s(): got %ld uncalled destructors in the NRO in __dso_handle %p", NRO_HOGE_CLASS_NAME, hoge_num_dtrs_to_call, __dso_handle);
        TESTCASE_MESSAGE(hoge_num_dtrs_to_call == 1, "Before exiting ~%s() we expected 1 uncalled Destructor but we got %ld.  __dso_handle = %p", NRO_HOGE_CLASS_NAME, hoge_num_dtrs_to_call, __dso_handle);
        num_HogeDestructorCalls++;
    }
};

static void test_nro_tls_constructors(void *arg)
{
    TESTLOG("test_nro_tls_constructors entered with __dso_handle = %p", __dso_handle);
    NN_UNUSED(arg);
    long            num_dtrs_to_call = 0;

    TESTLOG("test_nro_tls_constructors calling __nnmusl_get_tls_dtors_status() for module with __dso_handle = %p", __dso_handle);
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "Before entering test_nro_tls_constructors we expected 0 uncalled Destructor but we got %ld.  __dso_handle = %p", num_dtrs_to_call, __dso_handle);

    TESTLOG("calling constructor for %s when __dso_handle = %p", NRO_HOGE_OBJ_NAME, __dso_handle);
    thread_local NRO_HOGE_CLASS NRO_HOGE_OBJ((char *)NRO_HOGE_OBJ_NAME, 47);

    TESTCASE_MESSAGE(Dummy == 1, "expected 1 for Dummy but got %d in %s's test_nro_tls_constructors.  __dso_handle = %p", Dummy, TEST_NAME, __dso_handle);

    TESTLOG("nsp_int = %d; expected 123456789; __dso_handle = %p", nsp_int, __dso_handle);
    TESTCASE_MESSAGE(nsp_int == 123456789, "expected 123456789 for nsp_int but got %d in %s's test_nro_tls_constructors.  __dso_handle = %p", nsp_int, TEST_NAME, __dso_handle);
    TESTLOG("%s = %d; expected %d; __dso_handle = %p", NRO_INT_NAME, NRO_INT, NRO_INT_VALUE, __dso_handle);
    TESTCASE_MESSAGE(NRO_INT == NRO_INT_VALUE, "expected %d for %s but got %d in %s's test_nro_tls_constructors.  __dso_handle = %p", NRO_INT_VALUE, NRO_INT_NAME, NRO_INT_VALUE, TEST_NAME, __dso_handle);

    TESTLOG("test_nro_tls_constructors calling __nnmusl_get_tls_dtors_status() for NRO with __dso_handle = %p", __dso_handle);
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTCASE_MESSAGE(num_dtrs_to_call == 1, "Before exiting nnMain we expected 1 uncalled Destructor but we got %ld.  __dso_handle = %p", num_dtrs_to_call, __dso_handle);
}

void *NRO_ENTRY()
{
    int i;
    TESTLOG("Start %s", TEST_NAME);

    TESTLOG("__EX_start = %p", (unsigned char *)__EX_start);
    TESTLOG("%s: __oasis_nnmusl_dso_modules->version = %d", TEST_NAME, __oasis_nnmusl_dso_modules->version);
    TESTLOG("%s: __oasis_nnmusl_dso_modules->nModules = %d", TEST_NAME, __oasis_nnmusl_dso_modules->nModules);
    TESTLOG("%s: __oasis_nnmusl_dso_modules->nElements = %d", TEST_NAME, __oasis_nnmusl_dso_modules->nElements);
    for (i = 1; i < __oasis_nnmusl_dso_modules->nElements; i++) {
        if (__oasis_nnmusl_dso_modules->modAddresses[i] == (size_t)__EX_start) {
            TESTLOG("%s is modAddresses element %d", TEST_NAME, i);
            break;
        }
    }

    nn::Result      result;
    long            num_dtrs_to_call = 0;
    int atexit_result;
    nn::os::ThreadType *main_thread = nn::os::GetCurrentThread();
    TESTLOG("1: main_thread = 0x%08zX", (size_t)main_thread);

    result = nn::os::CreateThread( &g_Thread1, test_nro_tls_constructors, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread1." );

    TESTLOG("g_Thread1 = 0x%08zX", (size_t)&g_Thread1);

    TESTLOG("2: main_thread = 0x%08zX", (size_t)main_thread);

    // スレッドの実行を開始する
    nn::os::StartThread( &g_Thread1 );
    TESTLOG("g_Thread1 started");

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );
    TESTLOG("g_Thread1 did wait");

    // スレッドを破棄する
    nn::os::DestroyThread( &g_Thread1 );
    TESTLOG("g_Thread1 destroyed");

    TESTLOG("~%s() got num_HogeDestructorCalls = %d", NRO_HOGE_CLASS_NAME, num_HogeDestructorCalls);
    TESTCASE_MESSAGE(num_HogeDestructorCalls == 1, "Before exiting %s we expected 1 call to ~%s() but we got %ld. __dso_handle = %p",
        NRO_ENTRY_NAME, NRO_HOGE_CLASS_NAME, num_HogeDestructorCalls, __dso_handle);

    TESTLOG("%s calling __nnmusl_get_tls_dtors_status() for module with __dso_handle = 0x%08zX", NRO_ENTRY_NAME, __dso_handle);
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "Before exiting %s we expected 0 uncalled Destructors but we got %ld.", NRO_ENTRY_NAME, num_dtrs_to_call);
    TESTLOG("End %s", TEST_NAME);
    return __dso_handle;
}


