﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include "ntd-test-tls.h"

#define MIN_ALIGN nn::os::ThreadStackAlignment

#define STACK_SIZE (3 * MIN_ALIGN)

const size_t            ThreadStackSize = STACK_SIZE;                 // スレッド操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック

nn::os::ThreadType      g_Thread1;
nn::os::ThreadType      g_Thread2;

int                     GameMinus;
thread_local short      GamePlus;
thread_local char       GamePlus1 = 1;
thread_local int        GamePlus100 = 100;
thread_local double     GamePlusDouble = 2.0;
thread_local float      GamePlusFloat;

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#if ULONG_MAX == 0xffffffff
#define TPOFF_K 8
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TPOFF_K 16
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif

static void GameUpdatePlusMinus()
{
    GameMinus -= 5;;
    GamePlus += 5;
    GamePlus1 += 1;
    GamePlus100 += 100;
    GamePlusDouble += 5.0;
    GamePlusFloat -= 5.0;
}

/*---------------------------------------------------------------------------*
  Name:         TestThreadOne

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    name of thread as string.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/


static void TestThreadOne(void *arg)
{
    int i;

    TESTLOG("TestThreadOne arg is %s", (char *)arg);

    TESTCASE_MESSAGE(GamePlus == 0, "GamePlus when initialized: %d (expected 0)", GamePlus);
    TESTCASE_MESSAGE(GamePlus1 == 1, "GamePlus1 when initialized: %d (expected 1)", GamePlus1);
    TESTCASE_MESSAGE(GamePlus100 == 100, "GamePlus100 when initialized: %d (expected 100)", GamePlus100);
    TESTCASE_MESSAGE(GamePlusDouble == 2.0, "GamePlusDouble when initialized: %d (expected 2.0)", GamePlusDouble);
    TESTCASE_MESSAGE(GamePlusFloat == 0.0, "GamePlusFloat when initialized: %f (expected 0.0)", GamePlusFloat);

    for (i = 0; i < 5; i++)
    {
        TESTLOG("Hello! from TestThreadOne.");
        // normal global data
        GameMinus--;
        // thread local storage data
        GamePlus++;
        GamePlus100++;
        GameUpdatePlusMinus();
    }
    TESTCASE_MESSAGE(GamePlus == 30, "GamePlus at thread exit: %d (expected 30)", GamePlus);
    TESTCASE_MESSAGE(GamePlus1 == 6, "GamePlus1 at thread exit: %d (expected 6)", GamePlus1);
    TESTCASE_MESSAGE(GamePlus100 == 605, "GamePlus100 at thread exit: %d (expected 605)", GamePlus100);
    TESTCASE_MESSAGE(GamePlusDouble == 27.0, "GamePlusDouble at thread exit: %f (expected 27.0)", GamePlusDouble);
    TESTCASE_MESSAGE(GamePlusFloat == -25.0, "GamePlusFloat at thread exit: %f (expected -25.0)", GamePlusFloat);
}


/*---------------------------------------------------------------------------*
  Name:         TestThreadTwo

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    name of thread as string.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/

static void TestThreadTwo(void *arg)
{
    int i;

    TESTLOG("TestThreadTwo arg is %s", (char *)arg);

    TESTCASE_MESSAGE(GamePlus == 0, "GamePlus when initialized: %d (expected 0)", GamePlus);
    TESTCASE_MESSAGE(GamePlus1 == 1, "GamePlus1 when initialized: %d (expected 1)", GamePlus1);
    TESTCASE_MESSAGE(GamePlus100 == 100, "GamePlus100 when initialized: %d (expected 100)", GamePlus100);
    TESTCASE_MESSAGE(GamePlusDouble == 2.0, "GamePlusDouble when initialized: %d (expected 2.0)", GamePlusDouble);
    TESTCASE_MESSAGE(GamePlusFloat == 0.0, "GamePlusFloat when initialized: %f (expected 0.0)", GamePlusFloat);

    for (i = 0; i < 5; i++)
    {
        TESTLOG("Hello! from TestThreadTwo.");
        // normal global data
        GameMinus--;
        // thread local storage data
        GamePlus++;
        GamePlus100++;
        GameUpdatePlusMinus();
    }
    TESTCASE_MESSAGE(GamePlus == 30, "GamePlus at thread exit: %d (expected 30)", GamePlus);
    TESTCASE_MESSAGE(GamePlus1 == 6, "GamePlus1 at thread exit: %d (expected 6)", GamePlus1);
    TESTCASE_MESSAGE(GamePlus100 == 605, "GamePlus100 at thread exit: %d (expected 605)", GamePlus100);
    TESTCASE_MESSAGE(GamePlusDouble == 27.0, "GamePlusDouble at thread exit: %f (expected 27.0)", GamePlusDouble);
    TESTCASE_MESSAGE(GamePlusFloat == -25.0, "GamePlusFloat at thread exit: %f (expected -25.0)", GamePlusFloat);
}

/*---------------------------------------------------------------------------*
  Name:         main

  Description:  This is the main function of the demo.

                This demo displays Hello! on the main thread and sub-threads.

  Arguments:    None.

  Returns:      Always zero.

 *---------------------------------------------------------------------------*/
extern "C" void nnMain()
{
    nn::Result      result;
    void *          p;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);

    p = malloc( 16 );
    TESTCASE_MESSAGE(p != 0, "malloc expected to return non-zero result");
    free( p );

    TESTCASE_MESSAGE(GameMinus == 0, "GameMinus when initialized: %d (expected 0)", GameMinus);
    // スレッドを生成する
    result = nn::os::CreateThread( &g_Thread1, TestThreadOne, (void *)"1", g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread1." );

    TESTLOG("g_Thread1 = 0x%08zX", (size_t)&g_Thread1);

    result = nn::os::CreateThread( &g_Thread2, TestThreadTwo, (void *)"2", g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread2." );

    TESTLOG("g_Thread2 = 0x%08zX", (size_t)&g_Thread2);

    TestThreadOne((void *)"main_thread");
    // スレッドの実行を開始する
    nn::os::StartThread( &g_Thread1 );
    TESTLOG("g_Thread1 started");
    nn::os::StartThread( &g_Thread2 );
    TESTLOG("g_Thread2 started");

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );
    TESTLOG("g_Thread1 did wait");
    nn::os::WaitThread( &g_Thread2 );
    TESTLOG("g_Thread2 did wait");

    // スレッドを破棄する
    nn::os::DestroyThread( &g_Thread1 );
    TESTLOG("g_Thread1 destroyed");
    nn::os::DestroyThread( &g_Thread2 );
    TESTLOG("g_Thread2 destroyed");

    TESTCASE_MESSAGE(GameMinus == -90, "GameMinus at main exit: %d (expected -90)", GameMinus);

    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}

