﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "thread_test.h"
/* The header above includes other headers this file needs:
     #include <stdio.h>
*/
#include <pthread.h>
#include <errno.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */
  #include "ntd-tests/ntd-tests.h"
#ifdef __cplusplus
}
#endif /* __cplusplus */

/*
  This example simulates a number of threads working on a parallel
  problem. The threads use pthread_mutex_trylock() so that
  they do not spend time blocking on a mutex and instead spend more
  of the time making progress towards the final solution. When
  trylock fails, the processing is done locally, eventually to
  be merged with the final parallel solution.

  This example should complete faster than the example for
  pthread_mutex_lock() in which threads solve the same parallel
  problem but spend more time waiting in resource contention.
  */
#define            LOOPCONSTANT     100000
#define            THREADS          10
typedef struct thread_info {
    pthread_t           threadid;
    char                name[16];
} thread_info;
static thread_info      threads[THREADS];

#if SUPPORT_STATIC_INITALIZATION
static pthread_mutex_t    mutex = PTHREAD_MUTEX_INITIALIZER;
#else
static pthread_mutex_t    mutex;
#endif
static int                i,j,k,l;

static void *threadfunc(void *parm)
{
  int   loop = 0, iter;
  int   localProcessingCompleted = 0;
  int   numberOfLocalProcessingBursts = 0;
  int   processingCompletedThisBurst = 0;
  int   numberOfGlobalProcessingBursts = 0;
  int   rc;

  pthread_t me = pthread_self();
  char *name = (char *)"Can not find thread";
  for (iter = 0; iter < THREADS; iter++) {
    if (threads[iter].threadid == me) {
        name = threads[iter].name;
        break;
    }
  }

  TESTWARN("%s entering threadfunc\n", name);

  for (loop=0; loop<LOOPCONSTANT; ++loop) {
    rc = pthread_mutex_trylock(&mutex);
    if (rc == EBUSY) {
      /* Process continue processing the part of the problem   */
      /* that we can without the lock. We do not want to waste */
      /* time blocking. Instead, we'll count locally.          */
      ++localProcessingCompleted;
      ++numberOfLocalProcessingBursts;
      continue;
    }
    /* We acquired the lock, so this part of the can be global*/
    TESTCASE_MESSAGE(rc == 0, "pthread_mutex_trylock() returned %d expected 0", rc);
    /* Processing completed consist of last local processing  */
    /* plus the 1 unit of processing this time through        */
    processingCompletedThisBurst = 1 + localProcessingCompleted;
    localProcessingCompleted = 0;
    i+=processingCompletedThisBurst; j+=processingCompletedThisBurst;
    k+=processingCompletedThisBurst; l+=processingCompletedThisBurst;
    numberOfGlobalProcessingBursts++;

    rc = pthread_mutex_unlock(&mutex);
    TESTCASE_MESSAGE(rc == 0, "pthread_mutex_unlock() returned %d expected 0", rc);

    /* Horizon threads seem to be FIFO; yield current thread to let other threads have a chance to work */
    sched_yield(); /* mea added */
  }
  /* If any local processing remains, merge it with the global*/
  /* problem so our part of the solution is accounted for     */
  if (localProcessingCompleted) {
    rc = pthread_mutex_lock(&mutex);
    TESTCASE_MESSAGE(rc == 0, "final pthread_mutex_lock() returned %d expected 0", rc);

    i+=localProcessingCompleted; j+=localProcessingCompleted;
    k+=localProcessingCompleted; l+=localProcessingCompleted;

    rc = pthread_mutex_unlock(&mutex);
    TESTCASE_MESSAGE(rc == 0, "final pthread_mutex_unlock() returned %d expected 0", rc);
  }

  TESTWARN("%s processed about %d%% of the problem locally\n",
         name, (numberOfLocalProcessingBursts * 100) / LOOPCONSTANT);
  TESTWARN("%s processed about %d%% of the problem globably\n\n",
         name, (numberOfGlobalProcessingBursts * 100) / LOOPCONSTANT);

  return NULL;
}

void test_pthread_mutex_trylock_yield()
{
  int                   rc=0;
  int                   loop=0;
  pthread_attr_t        pta;
  const char *testcase = __FUNCTION__;

  NTD_TEST_GROUP_START("mutex_trylock_yield", 3);

  pthread_attr_init(&pta);
  /* pthread_attr_setdetachstate(&pta, PTHREAD_CREATE_JOINABLE); */

#if !SUPPORT_STATIC_INITALIZATION
  rc = pthread_mutex_init(&mutex, NULL);
  TESTCASE_MESSAGE(rc == 0, "pthread_mutex_init(NULL) returned %d expected 0", rc);
#endif
  rc = pthread_mutex_lock(&mutex);
  TESTCASE_MESSAGE(rc == 0, "pthread_mutex_lock() returned %d expected 0", rc);

  TESTWARN("Creating %d threads\n", THREADS);
  for (loop=0; loop<THREADS; ++loop) {
    rc = pthread_create(&(threads[loop].threadid), &pta, threadfunc, NULL);
    sprintf(threads[loop].name, "thread %d", loop);
    TESTCASE_MESSAGE(rc == 0, "pthread_create() returned %d expected 0", rc);
  }

  sleep(1); /* mea added */
  rc = pthread_mutex_unlock(&mutex);
  TESTCASE_MESSAGE(rc == 0, "pthread_mutex_unlock() returned %d expected 0", rc);

  TESTWARN("Wait for results\n");
  for (loop=0; loop<THREADS; ++loop) {
    rc = pthread_join(threads[loop].threadid, NULL);
    TESTCASE_MESSAGE(rc == 0, "pthread_join() returned %d expected 0", rc);
  }

  TESTWARN("Cleanup and show results\n");
  pthread_attr_destroy(&pta);
  pthread_mutex_destroy(&mutex);

  TESTWARN("\nUsing %d threads and LOOPCONSTANT = %d\n",
         THREADS, LOOPCONSTANT);
  TESTWARN("Values are: (should be %d)\n", THREADS * LOOPCONSTANT);
  TESTWARN("  ==>%d, %d, %d, %d\n", i, j, k, l);

  NTD_TEST_GROUP_END("mutex_trylock_yield", 3);
}

#if 0
Debug output:

Entering testcase
Creating 10 threads
Wait for results
Thread processed about 100% of the problem locally
Thread processed about 90% of the problem locally
Thread processed about 88% of the problem locally
Thread processed about 94% of the problem locally
Thread processed about 93% of the problem locally
Thread processed about 96% of the problem locally
Thread processed about 90% of the problem locally
Thread processed about 91% of the problem locally
Thread processed about 81% of the problem locally
Thread processed about 76% of the problem locally
Cleanup and show results

Using 10 threads and LOOPCONSTANT = 100000
Values are: (should be 1000000)
  ==>1000000, 1000000, 1000000, 1000000
Main completed
#endif
