﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "thread_test.h"
/* The header above includes other headers this file needs:
     #include <stdio.h>
*/
#include <pthread.h>
#include <sched.h>

static void validate_policy(pthread_attr_t *a, int set_policy, char *set_policy_str)
{
    int                 rc=0;
    int                 policy;
    char *              policy_str;
    bool                test;

    rc = pthread_attr_getschedpolicy(a, &policy);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_getschedpolicy() returned %d expected 0", rc);

    if (!rc) {
        if (policy == SCHED_FIFO) {
            policy_str = (char *)"SCHED_FIFO";
        } else if (policy == SCHED_OTHER) {
            policy_str = (char *)"SCHED_OTHER";
        } else {
            policy_str = (char *)"UNKNOWN_POLICY";
        }
        TESTCASE_MESSAGE(policy == set_policy,
            "Failure: expected scheduling policy %s but pthread_attr_getschedpolicy returned %s",
            set_policy_str, policy_str);
    }
}

void test_pthread_attr_setschedpolicy()
{
    int                   rc=0;
    pthread_attr_t        pta;
    bool                  test;
    const char *testcase = __FUNCTION__;

    NTD_TEST_GROUP_START("attr_setschedpolicy", 3);

    TESTWARN("Create a thread attributes object with default scheduling policy SCHED_OTHER\n");
    rc = pthread_attr_init(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

    if (!rc) {
        validate_policy(&pta, 0, (char *)"SCHED_OTHER");
    } else {
        TESTWARN("Failed calling pthread_attr_init\n");
    }

    rc = pthread_attr_init(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

    TESTWARN("Setting scheduling parameters to SCHED_FIFO\n");
    rc = pthread_attr_setschedpolicy(&pta, SCHED_FIFO);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setschedpolicy() returned %d expected 0", rc);

    if (!rc) {
        validate_policy(&pta, SCHED_FIFO, (char *)"SCHED_FIFO");
    } else {
        TESTWARN("Failed passing %s for policy\n", "SCHED_FIFO");
    }

    rc = pthread_attr_init(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

    TESTWARN("Setting scheduling parameters to SCHED_OTHER\n");
    rc = pthread_attr_setschedpolicy(&pta, SCHED_OTHER);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setschedpolicy() returned %d expected 0", rc);

    if (!rc) {
        validate_policy(&pta, SCHED_OTHER, (char *)"SCHED_OTHER");
    } else {
        TESTWARN("Failed passing %s for policy\n", "SCHED_OTHER");
    }

    rc = pthread_attr_init(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

#if TEST_UNSUPPORTED_FEATURES
    TESTWARN("Setting scheduling parameters to SCHED_RR\n");
    rc = pthread_attr_setschedpolicy(&pta, SCHED_RR);
    test = (rc == ENOTSUP);
    TESTCASE_MESSAGE(rc == ENOTSUP,
        "pthread_attr_setschedpolicy(&pta, SCHED_RR) returned %d expected ENOTSUP(%d)",
        rc, ENOTSUP);
#endif

    TESTWARN("Destroy thread attributes object\n");
    rc = pthread_attr_destroy(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_destroy() returned %d expected 0", rc);

    NTD_TEST_GROUP_END("attr_setschedpolicy", 3);
}
