﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "thread_test.h"
/* The header above includes other headers this file needs:
     #include <stdio.h>
     #include <string.h>
     #include <stdlib.h>
*/
#include <pthread.h>
#include <sched.h>

#if defined(__NX__)
  #include <nnc/os.h>
  #include <nnc/os/os_ThreadApi.h>

  #define MIN_ALIGN NN_OS_THREAD_STACK_ALIGNMENT
#else
  #define MIN_ALIGN 0x1000    /* 4K */
#endif /* __NX__ */

void test_pthread_attr_set_and_get()
{
    int                     rc=0;
    pthread_attr_t          pta;
    int                     policy;
    struct sched_param      in_param, out_param;
    int                     inheritsched;
    size_t                  mystacksize = 2 * MIN_ALIGN;
    void                    *mystackaddr;
    void                    *myallocation = 0;
    void                    *stackaddr;
    size_t                  stacksize;

    NTD_TEST_GROUP_START("attr_set_and_get", 3);

    TESTWARN("Create a thread attributes object\n");
    rc = pthread_attr_init(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

    rc = pthread_attr_setschedpolicy(&pta, SCHED_FIFO);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setschedpolicy() 1 returned %d expected 0", rc);

    rc = pthread_attr_setinheritsched(&pta, PTHREAD_EXPLICIT_SCHED);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setinheritsched() 1 returned %d expected 0", rc);

    memset(&in_param, 0, sizeof(in_param));
    memset(&out_param, 0, sizeof(out_param));
    in_param.sched_priority = -16;
    rc = pthread_attr_setschedparam(&pta, &in_param);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setschedparam() returned %d expected 0", rc);

    mystackaddr = myallocation = malloc(mystacksize + (MIN_ALIGN - 1));
    if (mystackaddr != NULL) {
        TESTWARN("Align mystackaddr to nearest 0x%x boundary.\n", MIN_ALIGN);
        mystackaddr = (void *)(((size_t)mystackaddr + (MIN_ALIGN - 1)) & ~(MIN_ALIGN - 1));
        TESTCASE_MESSAGE(!((size_t)mystackaddr & (MIN_ALIGN - 1)),
            "mystackaddr = %p missaligned by %ld",
            mystackaddr, (size_t)mystackaddr & (MIN_ALIGN - 1));
    } else {
        TESTWARN("Unable to allocate stack.");
        return;
    }

    rc = pthread_attr_setstack(&pta, mystackaddr, mystacksize);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setstack() returned %d expected 0", rc);

    rc = pthread_attr_getschedpolicy(&pta, &policy);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_getschedpolicy() returned %d expected 0", rc);
    TESTCASE_MESSAGE(policy == SCHED_FIFO,
        "error: pthread_attr_getschedpolicy expected %d but got %d\n", SCHED_FIFO, policy
    );

    rc = pthread_attr_getinheritsched(&pta, &inheritsched);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_getinheritsched() returned %d expected 0", rc);
    TESTCASE_MESSAGE(inheritsched == PTHREAD_EXPLICIT_SCHED,
        "error: pthread_attr_getinheritsched expected %d but got %d\n",
        PTHREAD_EXPLICIT_SCHED, inheritsched);

    rc = pthread_attr_getschedparam(&pta, &out_param);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_getschedparam() returned %d expected 0", rc);
    TESTCASE_MESSAGE(out_param.sched_priority == -16,
        "error: pthread_attr_getschedparam expected %d but got %d\n", -16, out_param.sched_priority);

    rc = pthread_attr_getstack(&pta, &stackaddr, &stacksize);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_setstack() returned %d expected 0", rc);
    if (rc == 0) {
        TESTCASE_MESSAGE(stackaddr == mystackaddr,
            "pthread_attr_getstack returned stackaddr 0x%x but expected 0x%x\n",
            stackaddr, mystackaddr);

        TESTCASE_MESSAGE(stacksize == mystacksize,
            "pthread_attr_getstack returned stackaddr 0x%x but expected 0x%x\n",
            stacksize, mystacksize);
    }

    TESTWARN("Destroy thread attributes object\n");
    rc = pthread_attr_destroy(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_destroy() returned %d expected 0", rc);

    if (myallocation)
        free(myallocation);

    NTD_TEST_GROUP_END("attr_set_and_get", 3);
}
