﻿/*******************************************************************************

  This file contains a series of tests for remove()

*******************************************************************************/

#include "ntd_extended_test.h"
#include <string.h> /* strstr() */
#include <sys/stat.h>

void test_remove_failure() __attribute__ ((optnone));

void test_remove_failure() __attribute__ ((optnone))
{
    FILE *file;
    int remove_ret_val, unlink_ret_val; /* Return values from remove() and unlink()*/
    const char *filename, *dirname, *dir_non_empty_name,
        *file_in_dir_name, *file2_in_dir_name, *file3_in_dir_name, *file_in_subdir2_name,
        *subdir_in_dir_name, *subdir2_in_dir_name, *subdir3_in_dir_name, *subsubdir_in_dir_name;
#if __NX__
    filename = "host:/remove.c_test_remove_failure.txt";
    dirname = "host:/dir_to_remove";
    dir_non_empty_name = "host:/dir_to_remove_non_empty";
    file_in_dir_name = "host:/dir_to_remove_non_empty/remove.c_test_remove_failure.txt";
    file2_in_dir_name = "host:/dir_to_remove_non_empty/remove2.c_test_remove_failure.txt";
    file3_in_dir_name = "host:/dir_to_remove_non_empty/remove3.c_test_remove_failure.txt";
    file_in_subdir2_name = "host:/dir_to_remove_non_empty/subdir2/remove.c_test_remove_failure.txt";
    subdir_in_dir_name = "host:/dir_to_remove_non_empty/subdir";
    subsubdir_in_dir_name = "host:/dir_to_remove_non_empty/subdir/subsubdir";
    subdir2_in_dir_name = "host:/dir_to_remove_non_empty/subdir2";
    subdir3_in_dir_name = "host:/dir_to_remove_non_empty/subdir3";
#else
    filename = "remove.c_test_remove_failure.txt";
    dirname = "dir_to_remove";
    dir_non_empty_name = "dir_to_remove_non_empty";
    file_in_dir_name = "dir_to_remove_non_empty/remove.c_test_remove_failure.txt";
    file2_in_dir_name = "dir_to_remove_non_empty/remove2.c_test_remove_failure.txt";
    file3_in_dir_name = "dir_to_remove_non_empty/remove3.c_test_remove_failure.txt";
    file_in_subdir2_name = "dir_to_remove_non_empty/subdir2/remove.c_test_remove_failure.txt";
    subdir_in_dir_name = "dir_to_remove_non_empty/subdir";
    subsubdir_in_dir_name = "dir_to_remove_non_empty/subdir/subsubdir";
    subdir2_in_dir_name = "dir_to_remove_non_empty/subdir2";
    subdir3_in_dir_name = "dir_to_remove_non_empty/subdir3";
#endif /* if __NX__ */

    /* Create the file if it doesn't already exist */
    file = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(file, filename);
    file = NULL;

    remove_ret_val = remove(filename);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        filename, remove_ret_val);

    remove_ret_val = remove(filename);
    TESTCASE_MESSAGE(remove_ret_val != 0, "remove(\"%s\") returned zero expected failure",
        filename);

    /* Create the directory if it doesn't already exist */
    mkdir(HOST_FILENAME("dir_to_remove"), S_IRWXU | S_IRWXG | S_IRWXO);

    remove_ret_val = remove(dirname);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        dirname, remove_ret_val);

    remove_ret_val = remove(dirname);
    TESTCASE_MESSAGE(remove_ret_val != 0, "remove(\"%s\") returned zero expected failure",
        dirname);

    /* Test unlink */
    /* Create the file if it doesn't already exist */
    file = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(file, filename);
    file = NULL;

    unlink_ret_val = unlink(filename);
    TESTCASE_MESSAGE(unlink_ret_val == 0, "unlink(\"%s\") returned %d expected 0",
        filename, unlink_ret_val);

     unlink_ret_val = unlink(filename);
    TESTCASE_MESSAGE(unlink_ret_val == -1 && errno==ENOENT, "unlink non-existing (\"%s\") returned %d expected -1 errno = %d(%s) expected ENOENT",
        filename, unlink_ret_val, errno, strerror(errno));

    /* Test rmdir a file */
    /* Create the file if it doesn't already exist */
    file = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(file, filename);
    file = NULL;

    remove_ret_val = rmdir(filename);
    TESTCASE_MESSAGE(remove_ret_val == -1 && errno==ENOTDIR, "rmdir file (\"%s\") returned %d expected -1  errno = %d(%s) expected ENOTDIR",
        filename, remove_ret_val, errno, strerror(errno));

    remove_ret_val = remove(filename);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove file (\"%s\") returned %d expected 0",
        filename, remove_ret_val);

    remove_ret_val = rmdir(filename);
    TESTCASE_MESSAGE(remove_ret_val == -1 && errno==ENOENT, "rmdir non-existing directory (\"%s\") returned %d expected -1 errno = %d(%s) expected ENOENT",
        filename, remove_ret_val, errno, strerror(errno));

    /* Create the directory if it doesn't already exist */
    mkdir(HOST_FILENAME("dir_to_remove"), S_IRWXU | S_IRWXG | S_IRWXO);

    unlink_ret_val = unlink(dirname);
    TESTCASE_MESSAGE(unlink_ret_val ==-1 && errno==EISDIR, "unlink(\"%s\") returned %d expected -1 errno = %d(%s) expected EISDIR",
        dirname, unlink_ret_val, errno, strerror(errno));

    /* Test for removing non-empty directory */
    /* 1. Remove directory with a file in it */
    /* Create the directory if it doesn't already exist */
    mkdir(HOST_FILENAME("dir_to_remove_non_empty"), S_IRWXU | S_IRWXG | S_IRWXO);

    /* Create the file if it doesn't already exist */
    file = FOPEN_TEST(file_in_dir_name, "w");
    FCLOSE_TEST(file, file_in_dir_name);
    file = NULL;

    /* Remove directory with a file in it */
    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    /* Remove the file, then remove the directory */
    remove_ret_val = remove(file_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        file_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        dir_non_empty_name, remove_ret_val);

    /* 2. Remove directory with a subdirectory in it */
    /* Create the directory if it doesn't already exist */
    mkdir(HOST_FILENAME("dir_to_remove_non_empty"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir"), S_IRWXU | S_IRWXG | S_IRWXO);

    /* Remove directory with a subdirectory in it */
    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    /* Remove the subdir, then remove the directory */
    remove_ret_val = remove(subdir_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subdir_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        dir_non_empty_name, remove_ret_val);

    /* 3. Remove directory with a subdirectory and a file in it */
    /* Create the directory if it doesn't already exist */
    mkdir(HOST_FILENAME("dir_to_remove_non_empty"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir"), S_IRWXU | S_IRWXG | S_IRWXO);
    /* Create the file if it doesn't already exist */
    file = FOPEN_TEST(file_in_dir_name, "w");
    FCLOSE_TEST(file, file_in_dir_name);
    file = NULL;

    /* Remove directory with a subdirectory and a file in it */
    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    /* Remove the subdir and the file, then remove the directory */
    remove_ret_val = remove(subdir_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subdir_in_dir_name, remove_ret_val);
    remove_ret_val = remove(file_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        file_in_dir_name, remove_ret_val);
    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        dir_non_empty_name, remove_ret_val);

    /* 4. Remove directory with multiple files in it */
    /* Create the directory if it doesn't already exist */
    mkdir(HOST_FILENAME("dir_to_remove_non_empty"), S_IRWXU | S_IRWXG | S_IRWXO);

    /* Create the files if it doesn't already exist */
    file = FOPEN_TEST(file_in_dir_name, "w");
    FCLOSE_TEST(file, file_in_dir_name);
    file = NULL;

    file = FOPEN_TEST(file2_in_dir_name, "w");
    FCLOSE_TEST(file, file2_in_dir_name);
    file = NULL;

    file = FOPEN_TEST(file3_in_dir_name, "w");
    FCLOSE_TEST(file, file3_in_dir_name);
    file = NULL;

    /* Remove directory with multiple files in it */
    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    /* Remove the files, then remove the directory */
    remove_ret_val = remove(file_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        file_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(file2_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        file2_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(file3_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        file3_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        dir_non_empty_name, remove_ret_val);

    /* 5. Remove directory with multiple subdirectories in it */
    /* Create the directory if it doesn't already exist */
    mkdir(HOST_FILENAME("dir_to_remove_non_empty"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir2"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir3"), S_IRWXU | S_IRWXG | S_IRWXO);

    /* Remove directory with multiple subdirectories in it */
    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    /* Remove the subdir, then remove the directory */
    remove_ret_val = remove(subdir_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subdir_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(subdir2_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subdir2_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(subdir3_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subdir_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        dir_non_empty_name, remove_ret_val);

    /* 6. Remove directory with multiple subdirectories and files in it */
    /* Create the directory if it doesn't already exist */
    mkdir(HOST_FILENAME("dir_to_remove_non_empty"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir2"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir3"), S_IRWXU | S_IRWXG | S_IRWXO);

    /* Create the files if it doesn't already exist */
    file = FOPEN_TEST(file_in_dir_name, "w");
    FCLOSE_TEST(file, file_in_dir_name);
    file = NULL;

    file = FOPEN_TEST(file2_in_dir_name, "w");
    FCLOSE_TEST(file, file2_in_dir_name);
    file = NULL;

    file = FOPEN_TEST(file3_in_dir_name, "w");
    FCLOSE_TEST(file, file3_in_dir_name);
    file = NULL;

    /* Remove directory with multiple subdirectories in it */
    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    /* Remove the files, then remove the directory */
    remove_ret_val = remove(file_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        file_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(file2_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        file2_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(file3_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        file3_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    /* Remove the subdir, then remove the directory */
    remove_ret_val = remove(subdir_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subdir_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(subdir2_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subdir2_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(subdir3_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subdir3_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        dir_non_empty_name, remove_ret_val);

    /* 7. Remove directory with multiple subdirectories and files of different levels in it*/
    /* Create the directory if it doesn't already exist */
    mkdir(HOST_FILENAME("dir_to_remove_non_empty"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir/subsubdir"), S_IRWXU | S_IRWXG | S_IRWXO);
    mkdir(HOST_FILENAME("dir_to_remove_non_empty/subdir2"), S_IRWXU | S_IRWXG | S_IRWXO);

    /* Create the files if it doesn't already exist */
    file = FOPEN_TEST(file_in_dir_name, "w");
    FCLOSE_TEST(file, file_in_dir_name);
    file = NULL;

    file = FOPEN_TEST(file_in_subdir2_name, "w");
    FCLOSE_TEST(file, file_in_subdir2_name);
    file = NULL;

    /* Remove directory with multiple subdirectories and files in it */
    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    /* Remove the files, then remove the directory */
    remove_ret_val = remove(file_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        file_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(file_in_subdir2_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        file_in_subdir2_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    /* Remove the subdir, then remove the directory */
    remove_ret_val = remove(subdir_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        subdir_in_dir_name, errno, strerror(errno));

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(subsubdir_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subsubdir_in_dir_name, remove_ret_val);

    remove_ret_val = remove(subdir_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subdir_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val != 0 && errno == ENOTEMPTY,
        "remove(\"%s\") returned zero when removing a non-empty directory"
        "errno = %d (%s) expected ENOTEMPTY",
        dir_non_empty_name, errno, strerror(errno));

    remove_ret_val = remove(subdir2_in_dir_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        subdir2_in_dir_name, remove_ret_val);

    remove_ret_val = remove(dir_non_empty_name);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0",
        dir_non_empty_name, remove_ret_val);

}
/*------------------------------------------------------------------------}}}}*/



void test_open_removed_file_helper(const char *mode,
                                   int is_read_mode)
{
    FILE *file;
    int remove_ret_val; /* Return value from from remove() */
    char *str_ret_val; /* Return value from strtr() */
    const char *filename;
#if __NX__
    filename = "host:/remove.c_test_opening_removed_file.txt";
#else
    filename = "remove.c_test_opening_removed_file.txt";
#endif /* if __NX__ */

    /* Create the file if it didn't already exist */
    file = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(file, filename);
    file = NULL;

    remove_ret_val = remove(filename);
    TESTCASE_MESSAGE(remove_ret_val == 0, "remove(\"%s\") returned %d expected 0\n",
        filename, remove_ret_val);

    file = fopen(filename, mode);

    if (file != NULL) /* If (opened deleted file for reading) */
    {
        TESTCASE_MESSAGE(!is_read_mode,
            "fopen('%s', '%s') returned %p, expected to fail for file that was just deleted.",
            filename, mode, file);
    }
    else
    {
        TESTCASE_MESSAGE(is_read_mode,
            "fopen('%s', '%s') returned %p, expected to succeed, for file that was just deleted.",
            filename, mode, file);
    }
    FCLOSE_DELETE_TEST(file, filename);
    file = NULL;
}

void test_open_removed_file()
{
    /* Reading modes */
    test_open_removed_file_helper("r",   1);
    test_open_removed_file_helper("r+",  1);
    test_open_removed_file_helper("rb",  1);
    test_open_removed_file_helper("rb+", 1);
    test_open_removed_file_helper("r+b", 1);

    /* Writing modes */
    test_open_removed_file_helper("w",   0);
    test_open_removed_file_helper("w+",  0);
    test_open_removed_file_helper("wb",  0);
    test_open_removed_file_helper("wb+", 0);
    test_open_removed_file_helper("w+b", 0);
    test_open_removed_file_helper("a",   0);
    test_open_removed_file_helper("a+",  0);
    test_open_removed_file_helper("ab",  0);
    test_open_removed_file_helper("ab+", 0);
    test_open_removed_file_helper("a+b", 0);

    /* !!!---> C2011 feature below. Comment these out if not supported */
    /* Writing "x" modes. These modes fail if the file exists */
    test_open_removed_file_helper("wx",   0);
    test_open_removed_file_helper("wx+",  0);
    test_open_removed_file_helper("w+x",  0);
    test_open_removed_file_helper("wbx",  0);
    test_open_removed_file_helper("w+bx", 0);
    test_open_removed_file_helper("wb+x", 0);
    test_open_removed_file_helper("ax",   0);
    test_open_removed_file_helper("ax+",  0);
    test_open_removed_file_helper("a+x",  0);
    test_open_removed_file_helper("abx",  0);
    test_open_removed_file_helper("a+bx", 0);
    test_open_removed_file_helper("ab+x", 0);
}

void test_open_unlink_file_helper(const char *mode,
                                   int is_read_mode)
{
    FILE *file;
    int unlink_ret_val; /* Return value from from remove() */
    char *str_ret_val; /* Return value from strtr() */
    const char *filename;
#if __NX__
    filename = "host:/remove.c_test_opening_removed_file.txt";
#else
    filename = "remove.c_test_opening_removed_file.txt";
#endif /* if __NX__ */

    /* Create the file if it didn't already exist */
    file = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(file, filename);
    file = NULL;

    unlink_ret_val = unlink(filename);
    TESTCASE_MESSAGE(unlink_ret_val == 0, "unlink(\"%s\") returned %d expected 0\n",
        filename, unlink_ret_val);

    file = fopen(filename, mode);

    if (file != NULL) /* If (opened deleted file for reading) */
    {
        TESTCASE_MESSAGE(!is_read_mode,
            "fopen('%s', '%s') returned %p, expected to fail for file that was just deleted.",
            filename, mode, file);
    }
    else
    {
        TESTCASE_MESSAGE(is_read_mode,
            "fopen('%s', '%s') returned %p, expected to succeed, for file that was just deleted.",
            filename, mode, file);
    }
    FCLOSE_DELETE_TEST(file, filename);
    file = NULL;
}

void test_open_unlinked_file()
{
    /* Reading modes */
    test_open_unlink_file_helper("r",   1);
    test_open_unlink_file_helper("r+",  1);
    test_open_unlink_file_helper("rb",  1);
    test_open_unlink_file_helper("rb+", 1);
    test_open_unlink_file_helper("r+b", 1);

    /* Writing modes */
    test_open_unlink_file_helper("w",   0);
    test_open_unlink_file_helper("w+",  0);
    test_open_unlink_file_helper("wb",  0);
    test_open_unlink_file_helper("wb+", 0);
    test_open_unlink_file_helper("w+b", 0);
    test_open_unlink_file_helper("a",   0);
    test_open_unlink_file_helper("a+",  0);
    test_open_unlink_file_helper("ab",  0);
    test_open_unlink_file_helper("ab+", 0);
    test_open_unlink_file_helper("a+b", 0);

    /* !!!---> C2011 feature below. Comment these out if not supported */
    /* Writing "x" modes. These modes fail if the file exists */
    test_open_unlink_file_helper("wx",   0);
    test_open_unlink_file_helper("wx+",  0);
    test_open_unlink_file_helper("w+x",  0);
    test_open_unlink_file_helper("wbx",  0);
    test_open_unlink_file_helper("w+bx", 0);
    test_open_unlink_file_helper("wb+x", 0);
    test_open_unlink_file_helper("ax",   0);
    test_open_unlink_file_helper("ax+",  0);
    test_open_unlink_file_helper("a+x",  0);
    test_open_unlink_file_helper("abx",  0);
    test_open_unlink_file_helper("a+bx", 0);
    test_open_unlink_file_helper("ab+x", 0);
}



int ntd_extended_remove(void)
{
    NTD_TEST_GROUP_START("remove", 2);
    test_remove_failure();
    test_open_removed_file();
    test_open_unlinked_file();
    return NTD_TEST_GROUP_END("remove", 2);
}
