﻿/*******************************************************************************

  This file contains a series of tests for fwrite()

*******************************************************************************/

#include "ntd_extended_test.h"
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>

static void test_rewind_append_fopen()
{
    FILE *file;
    const char *filename;
    char *fread_buffer;
#if __NX__
    filename = "host:/fwrite_rewind_append.txt";
#else
    filename = "fwrite_rewind_append.txt";
#endif /* if __NX__ */

    file = FOPEN_TEST(filename, "w");
    FWRITE_STRING_TEST("1", file);
    FCLOSE_TEST(file, filename);
    file = NULL;

    file = FOPEN_TEST(filename, "a");
    FWRITE_STRING_TEST("2", file);
    rewind(file);
    FWRITE_STRING_TEST("3", file);
    FCLOSE_TEST(file, filename);
    file = NULL;

    file = FOPEN_TEST(filename, "r");
    FREAD_FILE_TEST(&fread_buffer, file);
    TESTCASE_STRINGS_MATCH(fread_buffer, "123");
    free(fread_buffer);
    FCLOSE_DELETE_TEST(file, filename);
    file = NULL;
}

static void test_rewind_append_freopen()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    char *fread_buffer;
#if __NX__
    filename = "host:/fwrite_freopen_append.txt";
#else
    filename = "fwrite.c_test_rewind_append_freopen.txt";
#endif /* if __NX__ */

    fopen_ptr = FOPEN_TEST(filename, "w");
    FWRITE_STRING_TEST("1", fopen_ptr);


#if __NX__
    freopen_ptr = freopen(filename, "a", fopen_ptr);
    TESTCASE_MESSAGE(freopen_ptr == NULL && errno == EBUSY,
       "freopen(\"%s\",\"r\") returns %p file is already open for write "
       "errno = %d (%s) expected EBUSY on horzion",
       filename, freopen_ptr, errno, strerror(errno));
#else
    freopen_ptr = FREOPEN_TEST(filename,"a",fopen_ptr);
#endif
    if (freopen_ptr == NULL) return;
    FWRITE_STRING_TEST("2", freopen_ptr);
    rewind(freopen_ptr);
    FWRITE_STRING_TEST("3", freopen_ptr);
    FCLOSE_TEST(freopen_ptr, filename);
    freopen_ptr = NULL;
    fopen_ptr = NULL;

    fopen_ptr = FOPEN_TEST(filename, "r");
    FREAD_FILE_TEST(&fread_buffer, fopen_ptr);
    TESTCASE_STRINGS_MATCH(fread_buffer, "123");
    free(fread_buffer);
    FCLOSE_DELETE_TEST(fopen_ptr, filename);
    freopen_ptr = NULL;
    fopen_ptr = NULL;
}


static void test_rewind_write()
{
  FILE *file;
  const char *filename;
  char *fread_buffer;
  #if __NX__
    filename = "host:/rewind_simple.txt";
  #else
    filename = "rewind_simple.txt";
  #endif /* if __NX__ */

  file = FOPEN_TEST(filename, "w");
  FWRITE_STRING_TEST("3", file);
  rewind(file);
  FWRITE_STRING_TEST("2", file);
  rewind(file);
  FWRITE_STRING_TEST("1", file);
  FCLOSE_TEST(file, filename);

  /* Read in the file and make sure only "1" was written and not "123" */
  file = FOPEN_TEST(filename, "r");
  FREAD_FILE_TEST(&fread_buffer, file);
  TESTCASE_STRINGS_MATCH(fread_buffer, "1");
  free(fread_buffer);
  FCLOSE_DELETE_TEST(file, filename);
}

static void test_fopen_and_write_with_each_mode_helper(const char *mode,
                                                int is_read_mode)
{
    FILE *file;
    char *fwrite_text;
    char *fread_buffer;
#if __NX__
    char *filename = "host:/fwrite_test_modes.txt";
#else
    char *filename = "fwrite_test_modes.txt";
#endif /* if __NX__ */


    fwrite_text = "Testing fwrite()";

    /* If the file needs to exist before it's written to, then create it */
    if (is_read_mode == 1)
    {
        file = FOPEN_TEST(filename, "w");
        FCLOSE_TEST(file, filename);
        file = NULL;
    }

    /* Write content to the file */
    file = FOPEN_TEST(filename, mode);
    FWRITE_STRING_TEST(fwrite_text, file);
    FCLOSE_TEST(file, filename);
    file = NULL;

    /* Read from the file */
    file = FOPEN_TEST(filename, "r");
    FREAD_FILE_TEST(&fread_buffer, file);
    FCLOSE_TEST(file, filename);
    file = NULL;

    /* Make sure the right content was written */
    TESTCASE_STRINGS_MATCH(fread_buffer, fwrite_text);

    DELETE_IF_EXISTS(filename);

    free(fread_buffer);
}

static void test_fopen_and_write_with_each_mode()
{
    /* Reading (plus writing) modes */
    test_fopen_and_write_with_each_mode_helper("r+",  1);
    test_fopen_and_write_with_each_mode_helper("rb+", 1);
    test_fopen_and_write_with_each_mode_helper("r+b", 1);
    /* (Can't write to modes "r" or "rb", so no need to test those) */

    /* Writing modes */
    test_fopen_and_write_with_each_mode_helper("w",   0);
    test_fopen_and_write_with_each_mode_helper("w+",  0);
    test_fopen_and_write_with_each_mode_helper("wb",  0);
    test_fopen_and_write_with_each_mode_helper("wb+", 0);
    test_fopen_and_write_with_each_mode_helper("w+b", 0);
    test_fopen_and_write_with_each_mode_helper("a",   0);
    test_fopen_and_write_with_each_mode_helper("a+",  0);
    test_fopen_and_write_with_each_mode_helper("ab",  0);
    test_fopen_and_write_with_each_mode_helper("ab+", 0);
    test_fopen_and_write_with_each_mode_helper("a+b", 0);

    /* !!!---> C2011 feature below. Comment these out if not supported */
    /* Writing "x" modes. These modes fail if the file exists */
    test_fopen_and_write_with_each_mode_helper("wx",   0);
    test_fopen_and_write_with_each_mode_helper("wx+",  0);
    test_fopen_and_write_with_each_mode_helper("w+x",  0);
    test_fopen_and_write_with_each_mode_helper("wbx",  0);
    test_fopen_and_write_with_each_mode_helper("w+bx", 0);
    test_fopen_and_write_with_each_mode_helper("wb+x", 0);
    test_fopen_and_write_with_each_mode_helper("ax",   0);
    test_fopen_and_write_with_each_mode_helper("ax+",  0);
    test_fopen_and_write_with_each_mode_helper("a+x",  0);
    test_fopen_and_write_with_each_mode_helper("abx",  0);
    test_fopen_and_write_with_each_mode_helper("a+bx", 0);
    test_fopen_and_write_with_each_mode_helper("ab+x", 0);
}

static void test_write_to_console()
{
    char buf[18]={"    console test\n\0"};
    int ret;

#if __NX__
    // writing to stdin seems to appear on stdout on linux
    ret = write(STDIN_FILENO, buf, 18);
    TESTCASE_MESSAGE(ret == -1 && errno == EBADF,
        "read(STDIN_FILENO) return %d expected -1 errno = %d (%s) expected EBADF on horizon",
        ret, errno, strerror(errno));
#endif
    ret = write(STDOUT_FILENO, buf, 18);
    TESTCASE_MESSAGE(ret == 18,
        "read(STDOUT_FILENO) return %d expected 18",
        ret);
    ret = write(STDERR_FILENO, buf, 18);
    TESTCASE_MESSAGE(ret == 18,
        "read(STDOUT_FILENO) return %d expected 18",
        ret);

}

int ntd_extended_fwrite(void)
{
    NTD_TEST_GROUP_START("fwrite", 2);
    test_rewind_append_fopen();
    test_rewind_append_freopen();
    test_rewind_write();
    test_fopen_and_write_with_each_mode();
    test_write_to_console();
    return NTD_TEST_GROUP_END("fwrite", 2);
}
