﻿/*---------------------------------------------------------------------------*
  Copyright (C) 2016 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "jira-regression-tests.h"
#include "ntd-test-libc.h"
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdarg.h>
#include <wchar.h>
#include <math.h>
#include <float.h>
#include <fenv.h>

#define DEBUG_ROUNDING 0

static void rynda_635_sprintf_test_double(int line, const char * expect, const char *fmt, double d)
{
    char stack_buffer[512] = "";
    size_t exp_len = strlen(expect);

    int actual_rc = snprintf(stack_buffer, sizeof(stack_buffer), fmt, d);
    if ((actual_rc != exp_len)
    || ((strncmp( stack_buffer, expect, sizeof(stack_buffer)) != 0)))
    {
        TESTCASE_FAILURE();
        fprintf( stderr,
            "FAILED: %s, line %d\n"
            "        format string \"%s\", value = %a\n"
            "        expected %2d, \"%s\"\n"
            "        actual   %2d, \"%s\"\n",
            short_path(__FILE__), line,
            fmt, d,
            exp_len, expect,
            actual_rc, stack_buffer);
    } else {
        TESTCASE_SUCCESS();
    }
}

#if DEBUG_ROUNDING
static void print_hex_double(const char *title, double d)
{
    uint64_t *d_ptr = &d;
    uint64_t ll = *d_ptr;
    printf("%-12.12s %16.16llx: mant 1.%13.13llx exp % +4d (%a)\n",
        title,
        ll,
        ll & 0x000fffffffffffffull,
        (int) ((ll >> 52LL) & 0x7ff) - 1023,
        d);
}

static void testround_d(double d, int p)
{
    double y = d;
    int e2 = 0;
    int neg = 0;
    // only print absolute value (prefix has - in neg case)

    printf("\ntestround_d(%a (%f), %d)\n", d, d, p);

    print_hex_double("d = :", d);
    if (signbit(y)) {
        y=-y;
        print_hex_double("neg y=-y:", y);
        neg = 1;
    }
    y = frexp(y, &e2);
    print_hex_double("y = frexp():", y);
    printf("e2 = %d\n", e2);
    y = y * 2;
    print_hex_double("y=y*2:", y);
    if (y) e2--;
    printf("after if (y) e2--; e2 = %d\n", e2);

#if 1
    // proposed fix
    double round = 1.0;
    int re;

    if (p<0 || p>=DBL_MANT_DIG/4) re=0;
    else re=DBL_MANT_DIG/4-p;

    if (p<0 || p>=DBL_MANT_DIG/4) re=0;
    else re=DBL_MANT_DIG/4-p;
#else
    double round = 8.0;
    int re;

    if (p<0 || p>=DBL_MANT_DIG/4-1) re=0;
    else re=DBL_MANT_DIG/4-1-p;
#endif

    printf("re = %d\n", re);

    if (re) {
        while (re--) round*=16;
        print_hex_double("round = ", round);
        if (neg) {
            printf("neg case\n");
            y=-y;
            print_hex_double("y=-y:", y);
            y-=round;
            print_hex_double("y+=round:", y);
            y+=round;
            print_hex_double("y-=round:", y);
            y=-y;
            print_hex_double("y=-y:", y);
        } else {
            y+=round;
            print_hex_double("y+=round:", y);
            y-=round;
            print_hex_double("y-=round:", y);
       }
       print_hex_double("after round y:", y);
    }
}
#endif

void rynda_635()
{
    NTD_TEST_GROUP_START("RYNDA-635", 1);
    int rounding_mode = fegetround();
    TESTCASE_MESSAGE(fesetround(FE_TONEAREST) == 0, "fesetround(FE_TONEAREST) failed");

    /* %a */
    /* 4 values:
     *  1.2345
     * -0.003
     * 3,270,901,760
     * 1,020,376,855
     * the last two values were intended to be:
     * -123.0
     * 0.0256
     */

#if DEBUG_ROUNDING
   // testround_d(1.2345, 0);
    testround_d(0.0256, 0);
    testround_d(0.0256, 1);
    testround_d(-0.003, 3);
#endif

    rynda_635_sprintf_test_double(__LINE__, "0x1.3c083126e978dp+0", "%a",    1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1p+0",               "%.0a",  1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1.p+0",              "%#.0a", 1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1.4p+0",             "%.1a",  1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1.3cp+0",            "%.2a",  1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1.3c1p+0",           "%.3a",  1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1.3c08p+0",          "%.4a",  1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1.3c083p+0",         "%.5a",  1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1.3c0831p+0",        "%.6a",  1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1.3c08312p+0",       "%.7a",  1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1.3c083127p+0",      "%.8a",  1.2345);
    rynda_635_sprintf_test_double(__LINE__, "0x1.3c083126fp+0",     "%.9a",  1.2345);

    rynda_635_sprintf_test_double(__LINE__, "-0x1.89374bc6a7efap-9", "%a",    -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1p-8",               "%.0a",  -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.p-8",              "%#.0a", -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.9p-9",             "%.1a",  -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.89p-9",            "%.2a",  -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.893p-9",           "%.3a",  -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.893p-9",           "%.3za", -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.8937p-9",          "%.4a",  -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.89375p-9",         "%.5a",  -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.89374cp-9",        "%.6a",  -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.89374bcp-9",       "%.7a",  -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.89374bc7p-9",      "%.8a",  -0.003);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.89374bc6ap-9",     "%.9a",  -0.003);

    rynda_635_sprintf_test_double(__LINE__, "0x1.85ecp+31",          "%a",    3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1p+32",               "%.0a",  3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1.p+32",              "%#.0a", 3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1.8p+31",             "%.1a",  3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1.86p+31",            "%.2a",  3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1.85fp+31",           "%.3a",  3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1.85ecp+31",          "%.4a",  3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1.85ec0p+31",         "%.5a",  3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1.85ec00p+31",        "%.6a",  3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1.85ec000p+31",       "%.7a",  3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1.85ec0000p+31",      "%.8a",  3270901760);
    rynda_635_sprintf_test_double(__LINE__, "0x1.85ec00000p+31",     "%.9a",  3270901760);

    rynda_635_sprintf_test_double(__LINE__, "0x1.e68db8b8p+29",      "%a",    1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1p+30",               "%.0a",  1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1.p+30",              "%#.0a", 1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1.ep+29",             "%.1a",  1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1.e7p+29",            "%.2a",  1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1.e69p+29",           "%.3a",  1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1.e68ep+29",          "%.4a",  1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1.e68dcp+29",         "%.5a",  1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1.e68db9p+29",        "%.6a",  1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1.e68db8cp+29",       "%.7a",  1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1.e68db8b8p+29",      "%.8a",  1020376855);
    rynda_635_sprintf_test_double(__LINE__, "0x1.e68db8b80p+29",     "%.9a",  1020376855);

    rynda_635_sprintf_test_double(__LINE__, "-0x1.ecp+6",            "%a",    -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1p+7",               "%.0a",  -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.p+7",              "%#.0a", -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.fp+6",             "%.1a",  -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.ecp+6",            "%.2a",  -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.ec0p+6",           "%.3a",  -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.ec00p+6",          "%.4a",  -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.ec000p+6",         "%.5a",  -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.ec0000p+6",        "%.6a",  -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.ec00000p+6",       "%.7a",  -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.ec000000p+6",      "%.8a",  -123.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.ec0000000p+6",     "%.9a",  -123.0);

    rynda_635_sprintf_test_double(__LINE__, "0x1.a36e2eb1c432dp-6",  "%a",    0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1p-5",                "%.0a",  0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1.p-5",               "%#.0a", 0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1.ap-6",              "%.1a",  0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1.a3p-6",             "%.2a",  0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1.a37p-6",            "%.3a",  0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1.a36ep-6",           "%.4a",  0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1.a36e3p-6",          "%.5a",  0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1.a36e2fp-6",         "%.6a",  0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1.a36e2ebp-6",        "%.7a",  0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1.a36e2eb2p-6",       "%.8a",  0.0256);
    rynda_635_sprintf_test_double(__LINE__, "0x1.a36e2eb1cp-6",      "%.9a",  0.0256);

    rynda_635_sprintf_test_double(__LINE__, "0x0p+0",  "%a",    0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0p+0",                "%.0a",  0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0.p+0",               "%#.0a", 0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0.0p+0",              "%.1a",  0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0.00p+0",             "%.2a",  0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0.000p+0",            "%.3a",  0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0.0000p+0",           "%.4a",  0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0.00000p+0",          "%.5a",  0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0.000000p+0",         "%.6a",  0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0.0000000p+0",        "%.7a",  0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0.00000000p+0",       "%.8a",  0.0);
    rynda_635_sprintf_test_double(__LINE__, "0x0.000000000p+0",      "%.9a",  0.0);

    rynda_635_sprintf_test_double(__LINE__, "-0x0p+0",  "%a",    -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0p+0",                "%.0a",  -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0.p+0",               "%#.0a", -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0.0p+0",              "%.1a",  -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0.00p+0",             "%.2a",  -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0.000p+0",            "%.3a",  -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0.0000p+0",           "%.4a",  -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0.00000p+0",          "%.5a",  -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0.000000p+0",         "%.6a",  -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0.0000000p+0",        "%.7a",  -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0.00000000p+0",       "%.8a",  -0.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x0.000000000p+0",      "%.9a",  -0.0);

    rynda_635_sprintf_test_double(__LINE__, "0x1p+1",  "%a",    2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1p+1",                "%.0a",  2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1.p+1",               "%#.0a", 2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1.0p+1",              "%.1a",  2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1.00p+1",             "%.2a",  2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1.000p+1",            "%.3a",  2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1.0000p+1",           "%.4a",  2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1.00000p+1",          "%.5a",  2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1.000000p+1",         "%.6a",  2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1.0000000p+1",        "%.7a",  2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1.00000000p+1",       "%.8a",  2.0);
    rynda_635_sprintf_test_double(__LINE__, "0x1.000000000p+1",      "%.9a",  2.0);

    rynda_635_sprintf_test_double(__LINE__, "0x1p-3",  "%a",    0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1p-3",                "%.0a",  0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1.p-3",               "%#.0a", 0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1.0p-3",              "%.1a",  0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1.00p-3",             "%.2a",  0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1.000p-3",            "%.3a",  0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1.0000p-3",           "%.4a",  0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1.00000p-3",          "%.5a",  0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1.000000p-3",         "%.6a",  0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1.0000000p-3",        "%.7a",  0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1.00000000p-3",       "%.8a",  0.125);
    rynda_635_sprintf_test_double(__LINE__, "0x1.000000000p-3",      "%.9a",  0.125);

    rynda_635_sprintf_test_double(__LINE__, "-0x1p+8",  "%a",    -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1p+8",                "%.0a",  -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.p+8",               "%#.0a", -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.0p+8",              "%.1a",  -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.00p+8",             "%.2a",  -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.000p+8",            "%.3a",  -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.0000p+8",           "%.4a",  -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.00000p+8",          "%.5a",  -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.000000p+8",         "%.6a",  -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.0000000p+8",        "%.7a",  -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.00000000p+8",       "%.8a",  -256.0);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.000000000p+8",      "%.9a",  -256.0);

    rynda_635_sprintf_test_double(__LINE__, "-0x1.b5c56465b22d1p+28","%a",    -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1p+29",                "%.0a",  -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.p+29",               "%#.0a", -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.bp+28",              "%.1a",  -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.b6p+28",             "%.2a",  -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.b5cp+28",            "%.3a",  -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.b5c5p+28",           "%.4a",  -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.b5c56p+28",          "%.5a",  -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.b5c564p+28",         "%.6a",  -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.b5c5646p+28",        "%.7a",  -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.b5c56466p+28",       "%.8a",  -459036230.356);
    rynda_635_sprintf_test_double(__LINE__, "-0x1.b5c56465bp+28",      "%.9a",  -459036230.356);

    TESTCASE_MESSAGE(fesetround(rounding_mode) == 0, "fesetround(FE_TONEAREST) failed");
    NTD_TEST_GROUP_END("RYNDA-635", 1);
}
