﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>

#include <sys/stat.h>

#include <nnt/nntest.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_BitUtil.h>
#include <nn/fs.h>
#include <nn/os.h>

#include "tests.h"

// Used by certain tests that require multiple mallocs
void* bytes[MAX_MALLOC_SIZE];

const int EXTRA_MEMORY_SIZE = 64 * 1024 * 1024;
char extraMemory[EXTRA_MEMORY_SIZE];
char* curMemory = extraMemory;

/******************************************************
Function Name: Allocate
Description  : Allocates memory without using malloc
******************************************************/
void* Allocate(size_t size)
{
    char* old = curMemory;
    curMemory += size;

    NN_ASSERT(curMemory <= extraMemory + EXTRA_MEMORY_SIZE && "Used up all the extra memory (not malloc-related)\n");

    return old;
}

/******************************************************
Function Name: Deallocate
Description  : Doesn't actually do anything, but is needed to match Allocate
******************************************************/
void Deallocate(void* p, size_t size)
{
    // We don't actually free this memory... we just keep a bunch of it for simplicity
}

/******************************************************
Function Name: FreeMallocMemory
Description  : Gives some memory back to malloc for use by a test
******************************************************/
void FreeMallocMemory(int size)
{
    void** ptr = &bytes[size];
    NN_LOG("Giving malloc an extra %i(%i) bytes\n", size, malloc_usable_size(*ptr));
    EXPECT_NE((void*)NULL, *ptr);
    free(*ptr);
    *ptr = NULL;
}

/******************************************************
Function Name: ReallocMallocMemory
Description  : Takes back the memory given to malloc earlier
******************************************************/
void ReallocMallocMemory(int size)
{
    void** ptr = &bytes[size];
    EXPECT_EQ((void*)NULL, *ptr);
    *ptr = malloc(size);
    NN_LOG("Took the extra %i(%i) bytes back\n", size, malloc_usable_size(*ptr));
    EXPECT_NE((void*)NULL, *ptr);
}

/******************************************************
Function Name: MallocAndVerify
Description  : Takes some memory from malloc for future use and verifies it's the expected size
******************************************************/
void MallocAndVerify(void** pBytes, int memoryAllowed, int expectedSize)
{
    *pBytes = malloc(memoryAllowed);
    int usableSize = malloc_usable_size(*pBytes);

    // Verify the usable size matches our expectations
    ASSERT_EQ(expectedSize, usableSize) << "Verification failed for " << memoryAllowed << "...\n";
}

/******************************************************
Function Name: Environment
Description  : The environment that grabs all the malloc memory before the tests run
******************************************************/
struct Environment : public ::testing::Environment
{
    // Override this to define how to set up the environment.
    virtual void SetUp()
    {
        // Make sure nn::fs allocations don't go to malloc
        nn::fs::SetAllocator(Allocate, Deallocate);
        nn::fs::MountHost("host", HOST_MOUNT_PATH);

        // Used by various tests
        NN_LOG("Creating folders/files...\n");
        mkdir(TEST_PATH, 0777);
        FILE *f1 = fopen(TEST_PATH_FILE1, "w");
        NN_ASSERT(f1);
        fclose(f1);
        FILE *f2 = fopen(TEST_PATH_FILE2, "w");
        NN_ASSERT(f2);
        fclose(f2);

        memset(bytes, 0, sizeof(bytes));

        // This memory will be used by various libc functions
        // Note that 480 is as high as you can go with pure 16-byte alignment (you can't allocate 496 bytes)
        NN_LOG("Allocating some memory...\n");

#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
        for (int i = 1; i <= 480; ++i)
            MallocAndVerify(&bytes[i], i, nn::util::align_up(i, 16));
        for (int i = 497; i <= 528; ++i)
            MallocAndVerify(&bytes[i], i, nn::util::align_up(i, 16));
        for (int i = 1009; i <= 1024; ++i)
            MallocAndVerify(&bytes[i], i, 1024);

        MallocAndVerify(&bytes[1072], 1072, 1104);
        MallocAndVerify(&bytes[1192], 1192, 1216);
        MallocAndVerify(&bytes[1256], 1256, 1360);
        MallocAndVerify(&bytes[1296], 1296, 1360);
        MallocAndVerify(&bytes[2096], 2096, 2448);
        MallocAndVerify(&bytes[2104], 2104, 2448);
        MallocAndVerify(&bytes[2721], 2721, 4096); // Additional allocation for open_memstream and open_wmemstream
        MallocAndVerify(&bytes[4096], 4096, 4096);
        MallocAndVerify(&bytes[90112], 90112, 90112);
#elif NN_BUILD_TARGET_PLATFORM_ADDRESS_32
        // 32-bit actually has different alignments
        for (int i = 1; i <= 368; ++i)
            MallocAndVerify(&bytes[i], i, nn::util::align_up(i, 8));
        for (int i = 369; i <= 384; ++i)
            MallocAndVerify(&bytes[i], i, 384);
        for (int i = 385; i <= 392; ++i)
            MallocAndVerify(&bytes[i], i, 392);
        for (int i = 393; i <= 408; ++i)
            MallocAndVerify(&bytes[i], i, 408);
        for (int i = 409; i <= 424; ++i)
            MallocAndVerify(&bytes[i], i, nn::util::align_up(i, 8));
        for (int i = 425; i <= 448; ++i)
            MallocAndVerify(&bytes[i], i, nn::util::align_up(i, 16));
        for (int i = 449; i <= 472; ++i)
            MallocAndVerify(&bytes[i], i, 472);
        for (int i = 473; i <= 480; ++i)
            MallocAndVerify(&bytes[i], i, 480);
        for (int i = 497; i <= 528; ++i)
            MallocAndVerify(&bytes[i], i, nn::util::align_up(i, 16));
        for (int i = 1009; i <= 1024; ++i)
            MallocAndVerify(&bytes[i], i, 1024);

        MallocAndVerify(&bytes[1072], 1072, 1112);
        MallocAndVerify(&bytes[1176], 1176, 1224);
        MallocAndVerify(&bytes[1192], 1192, 1224);
        MallocAndVerify(&bytes[1256], 1256, 1360);
        MallocAndVerify(&bytes[1296], 1296, 1360);
        MallocAndVerify(&bytes[2048], 2048, 4096);
        MallocAndVerify(&bytes[2096], 2096, 4096);
        MallocAndVerify(&bytes[2104], 2104, 4096);
        MallocAndVerify(&bytes[90112], 90112, 90112);
#else
#error Unsupported platform type
#endif

        // Allocate as much memory as possible
        // For large sizes, just reduce by half until there's nothing left
        // For small amounts of memory that might potentially be missed by this, go by the alignment amount (16)
        NN_LOG("Allocating the rest of the memory...\n");
        for (int i = 1024 * 1024 * 1024; i > MAX_MALLOC_SIZE; i /= 2)
        {
            void* p = malloc(i);
            while (p)
                p = malloc(i);
        }
        for (int i = MAX_MALLOC_SIZE; i > 0; i -= 16)
        {
            void* p = malloc(i);
            while (p)
                p = malloc(i);
        }
    }
};

::testing::Environment* const env = ::testing::AddGlobalTestEnvironment(new Environment);
