﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifdef _WIN32
#include <cstdio>
#include <io.h>
#endif

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/cstd/cstd_CStdArg.h>
#include <nn/util/util_FormatString.h>

#include <nnt/gtest/detail/gtest-cstdio.h>
#include <nnt/gtest/detail/gtest-heap.h>
#include <nnt/gtest/detail/gtest-host.h>
#include <nnt/gtest/detail/gtest-sstream.h>
#include <nnt/gtest/detail/gtest-string.h>

namespace nnt { namespace testing { namespace detail {

namespace {

StringStream g_Stream;

enum FileType : int
{
    FileType_Unknown,
    FileType_System,
    FileType_User,
};

File g_Stderr = {
    FileType_System,
#ifdef _WIN32
    stderr,
#else
    nullptr,
#endif
};

File g_Stdout = {
    FileType_System,
#ifdef _WIN32
    stdout,
#else
    nullptr,
#endif
};

struct OutputFunctionArg final
{
    int count;
    StringStream* pStream;
};

void FormatStringOutput(uintptr_t arg, const char* pCharacters, int count)
{
    auto outputFunctionArg = reinterpret_cast<OutputFunctionArg*>(arg);
    outputFunctionArg->count += count;
    *(outputFunctionArg->pStream) << String(pCharacters, pCharacters + count);
}

} // namespace

File* const CStdio::Stderr = &g_Stderr;

File* const CStdio::Stdout = &g_Stdout;

int CStdio::Fclose(File* stream) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stream);

    switch (stream->_type)
    {
    case FileType_User:
        {
            Host::CloseFile(stream->_impl);
            Heap::Free(stream);
            return 0;
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

int CStdio::Fflush(File* stream) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stream);

    switch (stream->_type)
    {
    case FileType_System:
        {
            NN_LOG(g_Stream.str().c_str());
            g_Stream.str("");
            return 0;
        }
    case FileType_User:
        {
            if (Host::FlushFile(stream->_impl))
            {
                return 0;
            }
            else
            {
                return -1;
            }
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

File* CStdio::Fopen(const char* filename, const char* mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(filename);
    NN_SDK_REQUIRES_NOT_NULL(mode);

    void* impl = Host::OpenFile(filename, mode);

    if (impl == nullptr)
    {
        return nullptr;
    }

    auto pFile = reinterpret_cast<File*>(Heap::Allocate(sizeof(File)));
    NN_SDK_ASSERT_NOT_NULL(pFile);
    pFile->_type = FileType_User;
    pFile->_impl = impl;

    return pFile;
}

int CStdio::Fprintf(File* stream, const char* format, ...) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stream);
    NN_SDK_REQUIRES_NOT_NULL(format);

    int rv = 0;

    va_list args;
    va_start(args, format);

    switch (stream->_type)
    {
    case FileType_System:
        {
            rv = CStdio::Vprintf(format, args);
            break;
        }
    case FileType_User:
        {
            StringStream stringStream;

            OutputFunctionArg arg = { 0, &stringStream };

            ::nn::util::VFormatString(
                FormatStringOutput, reinterpret_cast<uintptr_t>(&arg),
                format, args);

            const String string = stringStream.str();

            void* const handle = stream->_impl;

            int offset = 0;
            if (!Host::GetFileSize(handle, &offset))
            {
                rv = -1;
                break;
            }

            int size = offset + arg.count;
            if (!Host::SetFileSize(handle, size))
            {
                rv = -1;
                break;
            }

            if (!Host::WriteFile(handle, offset, string.c_str(), arg.count))
            {
                rv = -1;
                break;
            }

            rv = arg.count;
            break;
        }
    default: NN_UNEXPECTED_DEFAULT;
    }

    va_end(args);

    return rv;
}

int CStdio::Fileno(File* stream) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stream);

    switch (stream->_type)
    {
    case FileType_System:
        {
#ifdef _WIN32
            return _fileno(static_cast<::std::FILE*>(stream->_impl));
#else
            return 0;
#endif
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

int CStdio::Isatty(int fd) NN_NOEXCEPT
{
#ifdef _WIN32
    return _isatty(fd);
#else
    NN_UNUSED(fd);
    return true;
#endif
}

int CStdio::Printf(const char *format, ...) NN_NOEXCEPT
{
    va_list args;
    va_start(args, format);
    int rv = CStdio::Vprintf(format, args);
    va_end(args);
    return rv;
}

int CStdio::Vprintf(const char* format, va_list args) NN_NOEXCEPT
{
    OutputFunctionArg arg = { 0, &g_Stream };
    ::nn::util::VFormatString(
        FormatStringOutput, reinterpret_cast<uintptr_t>(&arg), format, args);
    return arg.count;
}

}}} // namespace nnt::testing::detail
