﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>

#include <nnt/gtest/detail/gtest-heap.h>

namespace nnt { namespace testing { namespace detail {

template <typename T>
class Vector final
{
public:
    NNT_TESTING_DETAIL_HEAP_IS_ALLOCATABLE();

private:
    class Entry final
    {
    public:
        NNT_TESTING_DETAIL_HEAP_IS_ALLOCATABLE();

    private:
        T m_Value;

    public:
        explicit Entry(const T& value) NN_NOEXCEPT : m_Value(value) {}

        T& GetValue() NN_NOEXCEPT
        {
            return this->m_Value;
        }

        const T& GetValue() const NN_NOEXCEPT
        {
            return this->m_Value;
        }

        void SetValue(const T& value) NN_NOEXCEPT
        {
            this->m_Value = value;
        }

        T* GetPointer() NN_NOEXCEPT
        {
            return &this->m_Value;
        }

    private:
        NN_DISALLOW_COPY(Entry);
        NN_DISALLOW_MOVE(Entry);
    };

public:
    typedef size_t size_type;
    typedef T value_type;

public:
    class const_iterator
    {
    protected:
        Entry** m_Ptr;

    public:
        typedef ptrdiff_t difference_type;
        typedef T value_type;

    public:
        explicit const_iterator(Entry** ptr) NN_NOEXCEPT : m_Ptr(ptr) {}

        const_iterator(const const_iterator& other) NN_NOEXCEPT
            : m_Ptr(other.m_Ptr) {}

        const_iterator& operator=(const const_iterator& other) NN_NOEXCEPT
        {
            if (this != &other)
            {
                this->m_Ptr = other.m_Ptr;
            }
            return *this;
        }

        const T& operator*() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            NN_SDK_REQUIRES_NOT_NULL(*(this->m_Ptr));
            return (*(this->m_Ptr))->GetValue();
        }

        const T* operator->() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            NN_SDK_REQUIRES_NOT_NULL(*(this->m_Ptr));
            return (*(this->m_Ptr))->GetPointer();
        }

        const_iterator& operator++() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            ++(this->m_Ptr);
            return *this;
        }

        const_iterator operator++(int) NN_NOEXCEPT
        {
            const_iterator iter(*this);
            operator++();
            return iter;
        }

        const_iterator& operator--() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            --(this->m_Ptr);
            return *this;
        }

        const_iterator& operator+=(size_t n) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            this->m_Ptr += n;
            return *this;
        }

        const_iterator operator+(size_t n) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            return const_iterator(this->m_Ptr + n);
        }

        ptrdiff_t operator-(const_iterator other) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            NN_SDK_REQUIRES_NOT_NULL(other.m_Ptr);
            return this->m_Ptr - other.m_Ptr;
        }

        bool operator==(const const_iterator& other) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            return this->m_Ptr == other.m_Ptr;
        }

        bool operator!=(const const_iterator& other) const NN_NOEXCEPT
        {
            return !(*this == other);
        }
    };

    class iterator : public const_iterator
    {
    public:
        explicit iterator(Entry** ptr) NN_NOEXCEPT : const_iterator(ptr) {}

        iterator(const iterator& other) NN_NOEXCEPT
            : const_iterator(other.m_Ptr) {}

        T& operator*() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            NN_SDK_REQUIRES_NOT_NULL(*(this->m_Ptr));
            return (*(this->m_Ptr))->GetValue();
        }

        T* operator->() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            NN_SDK_REQUIRES_NOT_NULL(*(this->m_Ptr));
            return (*(this->m_Ptr))->GetPointer();
        }

        iterator& operator=(const iterator& other) NN_NOEXCEPT
        {
            const_iterator::operator=(other);
            return *this;
        }

        iterator operator+(size_t n) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            return iterator(this->m_Ptr + n);
        }
    };

    class const_reverse_iterator
    {
    protected:
        Entry** m_Ptr;

    public:
        explicit const_reverse_iterator(Entry** ptr) NN_NOEXCEPT : m_Ptr(ptr) {}

        const_reverse_iterator(const const_reverse_iterator& other) NN_NOEXCEPT
            : m_Ptr(other.m_Ptr) {}

        const_reverse_iterator& operator=(
            const const_reverse_iterator& other) NN_NOEXCEPT
        {
            if (this != &other)
            {
                this->m_Ptr = other.m_Ptr;
            }
            return *this;
        }

        const T& operator*() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            NN_SDK_REQUIRES_NOT_NULL(*(this->m_Ptr - 1));
            return (*(this->m_Ptr - 1))->GetValue();
        }

        const T* operator->() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            NN_SDK_REQUIRES_NOT_NULL(*(this->m_Ptr - 1));
            return (*(this->m_Ptr - 1))->GetPointer();
        }

        const_reverse_iterator& operator++() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            --(this->m_Ptr);
            return *this;
        }

        bool operator==(const const_reverse_iterator& other) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_Ptr);
            return this->m_Ptr == other.m_Ptr;
        }

        bool operator!=(const const_reverse_iterator& other) const NN_NOEXCEPT
        {
            return !(*this == other);
        }
    };

private:
    Entry **m_ppBuffer;

    size_t m_Count;

public:
    Vector() NN_NOEXCEPT : m_ppBuffer(Vector::Allocate(0)), m_Count(0) {}

    Vector(const Vector& other) NN_NOEXCEPT
    {
        this->Initialize(other);
    }

    Vector(Vector&& other) NN_NOEXCEPT
    {
        if (this != &other)
        {
            this->Finalize();
            this->MoveFrom(other);
        }
    }

    explicit Vector(size_t n) NN_NOEXCEPT
        : m_ppBuffer(Vector::Allocate(0))
        , m_Count(0)
    {
        while (0 < n--)
        {
            this->push_back(T());
        }
    }

    Vector(size_t n, const T& value) NN_NOEXCEPT
        : m_ppBuffer(Vector::Allocate(0))
        , m_Count(0)
    {
        while (0 < n--)
        {
            this->push_back(T(value));
        }
    }

    template<class InputIt>
    Vector(InputIt head, InputIt tail) NN_NOEXCEPT
        : m_ppBuffer(Vector::Allocate(0))
        , m_Count(0)
    {
        for (InputIt iter = head; iter != tail; ++iter)
        {
            this->push_back(*iter);
        }
    }

    ~Vector() NN_NOEXCEPT { this->Finalize(); }

    Vector& operator=(const Vector& other) NN_NOEXCEPT
    {
        if (this != &other)
        {
            this->Finalize();
            this->Initialize(other);
        }

        return *this;
    }

    Vector& operator=(Vector&& other) NN_NOEXCEPT
    {
        if (this != &other)
        {
            this->Finalize();
            this->MoveFrom(other);
        }

        return *this;
    }

    size_t size() const NN_NOEXCEPT
    {
        return this->m_Count;
    }

    bool empty() const NN_NOEXCEPT
    {
        return m_Count == 0;
    }

    void reserve(size_t size = 0) NN_NOEXCEPT
    {
        NN_UNUSED(size);
    }

    iterator begin() NN_NOEXCEPT
    {
        return iterator(this->m_ppBuffer);
    }

    const_iterator begin() const NN_NOEXCEPT
    {
        return const_iterator(this->m_ppBuffer);
    }

    iterator end() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!(this->m_ppBuffer[this->m_Count]));
        return iterator(this->m_ppBuffer + this->m_Count);
    }

    const_iterator end() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!(this->m_ppBuffer[this->m_Count]));
        return const_iterator(this->m_ppBuffer + this->m_Count);
    }

    const_reverse_iterator rbegin() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!(this->m_ppBuffer[this->m_Count]));
        return const_reverse_iterator(this->m_ppBuffer + this->m_Count);
    }

    const_reverse_iterator rend() const NN_NOEXCEPT
    {
        return const_reverse_iterator(this->m_ppBuffer);
    }

    bool operator==(const Vector& other) const NN_NOEXCEPT
    {
        if (this->size() != other.size())
        {
            return false;
        }

        const_iterator head = this->begin();
        const_iterator tail = this->end();
        const_iterator peer = other.begin();

        while (head != tail)
        {
            if (*head != *peer)
            {
                return false;
            }

            ++head;
            ++peer;
        }

        return true;
    }

    T& operator[](size_t index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0u, this->m_Count);
        return this->m_ppBuffer[index]->GetValue();
    }

    const T& operator[](size_t index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0u, this->m_Count);
        return this->m_ppBuffer[index]->GetValue();
    }

    T& at(size_t index)
    {
        NN_SDK_REQUIRES_RANGE(index, 0u, this->m_Count);
        return this->m_ppBuffer[index]->GetValue();
    }

    const T& at(size_t index) const
    {
        NN_SDK_REQUIRES_RANGE(index, 0u, this->m_Count);
        return this->m_ppBuffer[index]->GetValue();
    }

    const T& back() const
    {
        NN_SDK_REQUIRES_LESS(0u, m_Count);
        return this->m_ppBuffer[m_Count - 1]->GetValue();
    }

    void push_back(const T& value) NN_NOEXCEPT
    {
        auto pp = Vector::Allocate(this->m_Count + 1);
        for (size_t i = 0; i < this->m_Count; ++i)
        {
            pp[i] = this->m_ppBuffer[i];
        }
        pp[this->m_Count] = new Entry(value);
        NN_SDK_ASSERT_NOT_NULL(pp[this->m_Count]);
        Vector::Free(this->m_ppBuffer);
        this->m_ppBuffer = pp;
        this->m_Count = this->m_Count + 1;
    }

    void pop_back() NN_NOEXCEPT
    {
        this->erase(const_iterator(this->m_ppBuffer + this->m_Count - 1));
    }

    iterator erase(const_iterator position) NN_NOEXCEPT
    {
        auto pp = Vector::Allocate(this->m_Count - 1);
        size_t i = 0;
        while (i < this->m_Count &&
               const_iterator(this->m_ppBuffer + i) != position)
        {
            NN_SDK_ASSERT_RANGE(i, 0u, this->m_Count - 1);
            pp[i] = this->m_ppBuffer[i];
            ++i;
        }
        NN_SDK_ASSERT_RANGE(i, 0u, this->m_Count);
        delete this->m_ppBuffer[i];
        for (size_t j = i + 1; j < this->m_Count; ++j)
        {
            pp[j - 1] = this->m_ppBuffer[j];
        }
        Vector::Free(this->m_ppBuffer);
        this->m_ppBuffer = pp;
        this->m_Count = this->m_Count - 1;
        return iterator(this->m_ppBuffer + i);
    }

    void clear() NN_NOEXCEPT
    {
        this->Finalize();
        this->Initialize(Vector());
    }

    void assign(size_t n, const T& value) NN_NOEXCEPT
    {
        this->clear();

        while (0 < n--)
        {
            this->push_back(T(value));
        }
    }

    void swap(Vector& other) NN_NOEXCEPT
    {
        {
            Entry **ppBuffer = m_ppBuffer;
            m_ppBuffer = other.m_ppBuffer;
            other.m_ppBuffer = ppBuffer;
        }

        {
            size_t count = m_Count;
            m_Count = other.m_Count;
            other.m_Count = count;
        }
    }

private:
    void Initialize(const Vector& other) NN_NOEXCEPT
    {
        this->m_ppBuffer = Vector::Allocate(other.m_Count);
        this->m_Count = other.m_Count;
        for (size_t i = 0; i < this->m_Count; ++i)
        {
            this->m_ppBuffer[i] = new Entry(other.m_ppBuffer[i]->GetValue());
            NN_SDK_ASSERT_NOT_NULL(this->m_ppBuffer[i]);
        }
    }

    void Finalize()
    {
        if (this->m_ppBuffer)
        {
            for (size_t i = 0; i < this->m_Count; ++i)
            {
                delete this->m_ppBuffer[i];
            }
            Vector::Free(this->m_ppBuffer);
        }
    }

    void MoveFrom(Vector& other) NN_NOEXCEPT
    {
        this->m_ppBuffer = other.m_ppBuffer;
        this->m_Count = other.m_Count;
        other.m_ppBuffer = nullptr;
        other.m_Count = 0;
    }

    static Entry **Allocate(size_t n) NN_NOEXCEPT
    {
        auto p = static_cast<Entry**>(Heap::Allocate((n + 1) * sizeof(Entry*)));
        NN_SDK_ASSERT_NOT_NULL(p);
        p[n] = nullptr;
        return p;
    }

    static void Free(Entry **p) NN_NOEXCEPT
    {
        Heap::Free(p);
    }
};

}}} // namespace nnt::testing::detail
