﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nnt/gtest/detail/gtest-cstring.h>
#include <nnt/gtest/detail/gtest-heap.h>

namespace nnt { namespace testing { namespace detail {

class String final
{
public:
    NNT_TESTING_DETAIL_HEAP_IS_ALLOCATABLE();

public:
    static const size_t npos = static_cast<size_t>(-1);

public:
    typedef char value_type;

public:
    class const_iterator
    {
    protected:
        char* m_Ptr;

    public:
        explicit const_iterator(char* ptr) NN_NOEXCEPT : m_Ptr(ptr)
        {
            NN_SDK_REQUIRES_NOT_NULL(ptr);
        }

        const char& operator*() const NN_NOEXCEPT
        {
            return *m_Ptr;
        }

        const_iterator& operator++() NN_NOEXCEPT
        {
            ++m_Ptr;
            return *this;
        }

        const_iterator& operator--() NN_NOEXCEPT
        {
            --m_Ptr;
            return *this;
        }

        const_iterator operator+(ptrdiff_t n) const NN_NOEXCEPT
        {
            return const_iterator(m_Ptr + n);
        }

        bool operator==(const const_iterator& other) const NN_NOEXCEPT
        {
            return m_Ptr == other.m_Ptr;
        }

        bool operator!=(const const_iterator& other) const NN_NOEXCEPT
        {
            return !(*this == other);
        }

        const char* base() const NN_NOEXCEPT
        {
            return m_Ptr;
        }
    };

    class iterator final : public const_iterator
    {
    public:
        explicit iterator(char* ptr) NN_NOEXCEPT : const_iterator(ptr) {}

        iterator& operator++() NN_NOEXCEPT
        {
            const_iterator::operator++();
            return *this;
        }

        iterator& operator--() NN_NOEXCEPT
        {
            const_iterator::operator--();
            return *this;
        }

        iterator operator+(ptrdiff_t n) const NN_NOEXCEPT
        {
            return iterator(this->m_Ptr + n);
        }
    };

    class const_reverse_iterator
    {
    protected:
        char* m_Ptr;

    public:
        explicit const_reverse_iterator(char* ptr) NN_NOEXCEPT : m_Ptr(ptr)
        {
            NN_SDK_REQUIRES_NOT_NULL(ptr);
        }

        const char& operator*() const NN_NOEXCEPT
        {
            return *(m_Ptr - 1);
        }
    };

private:
    char* m_pBuffer;

    size_t m_Count;

public:
    String() NN_NOEXCEPT
    {
        this->Initialize("", 0);
    }

    String(const String& other) NN_NOEXCEPT
    {
        this->Initialize(other.m_pBuffer, other.m_Count);
    }

    String(const String& other, size_t pos, size_t len = npos) NN_NOEXCEPT
    {
        if (other.m_Count <= pos)
        {
            this->Initialize("", 0);
        }
        else if (len == npos || pos + len >= other.m_Count)
        {
            this->Initialize(other.m_pBuffer + pos, other.m_Count - pos);
        }
        else
        {
            this->Initialize(other.m_pBuffer + pos, len);
        }
    }

    String(const char* str) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(str);
        this->Initialize(str, CString::Strlen(str));
    }

    String(const char* str, size_t n) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(str);
        this->Initialize(str, n);
    }

    String(size_t n, char c) NN_NOEXCEPT;

    String(String&& other) NN_NOEXCEPT
    {
        this->MoveFrom(other);
    }

    String(const char* head, const char* tail) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(head);
        NN_SDK_REQUIRES_NOT_NULL(tail);
        this->Initialize(head, tail - head > 0 ? tail - head : 0);
    }

    ~String() NN_NOEXCEPT
    {
        this->Finalize();
    }

    String& operator=(const String& other) NN_NOEXCEPT
    {
        return this->assign(other);
    }

    String& operator=(const char* str) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(str);
        return this->assign(String(str));
    }

    String& operator=(String&& other) NN_NOEXCEPT
    {
        if (this != &other)
        {
            this->Finalize();
            this->MoveFrom(other);
        }
        return *this;
    }

    size_t size() const NN_NOEXCEPT { return m_Count; }

    size_t length() const NN_NOEXCEPT { return m_Count; }

    void reserve(size_t size = 0) NN_NOEXCEPT
    {
        NN_UNUSED(size);
    }

    bool empty() const NN_NOEXCEPT { return m_Count == 0; }

    const char& operator[](size_t pos) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(pos, m_Count);
        return this->m_pBuffer[pos];
    }

    char& operator[](size_t pos) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(pos, m_Count);
        return this->m_pBuffer[pos];
    }

    String& operator+=(const String& other) NN_NOEXCEPT
    {
        return this->append(other);
    }

    String& operator+=(const char* str) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(str);
        return this->append(str);
    }

    String& operator+=(char c) NN_NOEXCEPT
    {
        this->push_back(c);
        return *this;
    }

    String& append(const String& other) NN_NOEXCEPT
    {
        return this->Append(other.m_pBuffer, other.m_Count);
    }

    String& append(const char* str) NN_NOEXCEPT
    {
        return this->Append(str, CString::Strlen(str));
    }

    String& append(const char* str, size_t n) NN_NOEXCEPT
    {
        return this->Append(str, n);
    }

    String& append(size_t n, char c) NN_NOEXCEPT
    {
        return this->append(String(n, c));
    }

    iterator begin() NN_NOEXCEPT
    {
        return iterator(m_pBuffer);
    }

    const_iterator begin() const NN_NOEXCEPT
    {
        return const_iterator(m_pBuffer);
    }

    iterator end() NN_NOEXCEPT
    {
        return iterator(m_pBuffer + m_Count);
    }

    const_iterator end() const NN_NOEXCEPT
    {
        return const_iterator(m_pBuffer + m_Count);
    }

    const_reverse_iterator rbegin() const NN_NOEXCEPT
    {
        return const_reverse_iterator(m_pBuffer + m_Count);
    }

    void push_back(char c) NN_NOEXCEPT
    {
        this->Append(&c, 1);
    }

    String& assign(const String& other) NN_NOEXCEPT
    {
        if (this != &other)
        {
            this->Finalize();
            this->Initialize(other.m_pBuffer, other.m_Count);
        }
        return *this;
    }

    String& assign(String&& other) NN_NOEXCEPT
    {
        if (this != &other)
        {
            this->Finalize();
            this->MoveFrom(other);
        }
        return *this;
    }

    String& assign(const char* str) NN_NOEXCEPT
    {
        return this->assign(String(str));
    }

    iterator erase(iterator p) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(
            reinterpret_cast<uintptr_t>(p.base()),
            reinterpret_cast<uintptr_t>(this->begin().base()),
            reinterpret_cast<uintptr_t>(this->end().base()));
        const ptrdiff_t pos = p.base() - this->begin().base();
        this->Erase(static_cast<size_t>(pos), 1);
        return iterator(m_pBuffer + pos);
    }

    const char* c_str() const NN_NOEXCEPT { return m_pBuffer; }

    const char* data() const NN_NOEXCEPT { return m_pBuffer; }

    String substr(size_t pos = 0, size_t len = npos) const NN_NOEXCEPT
    {
        return String(*this, pos, len);
    }

    int compare(const String& other) const NN_NOEXCEPT
    {
        return this->compare(other.m_pBuffer);
    }

    int compare(const char* str) const NN_NOEXCEPT
    {
        return CString::Strcmp(m_pBuffer, str);
    }

    size_t find(const String& str, size_t pos = 0) const NN_NOEXCEPT
    {
        return this->find(str.c_str(), pos);
    }

    size_t find(char c, size_t pos = 0) const NN_NOEXCEPT
    {
        for (size_t i = pos; i < m_Count; ++i)
        {
            if (m_pBuffer[i] == c) { return i; }
        }

        return npos;
    }

    size_t find(const char* str, size_t pos = 0) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(str);
        NN_SDK_REQUIRES_MINMAX(pos, 0u, m_Count);
        const char* ptr = CString::Strstr(&m_pBuffer[pos], str);
        return (ptr == nullptr) ? npos : static_cast<size_t>(ptr - m_pBuffer);
    }

    size_t find_first_of(const char* str, size_t pos = 0) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(str);
        NN_SDK_REQUIRES_MINMAX(pos, 0u, m_Count);

        for (size_t i = pos; i < m_Count; ++i)
        {
            const char needle[] = { m_pBuffer[i], '\0' };

            const char* ptr = CString::Strstr(str, needle);

            if (ptr != nullptr) { return i; }
        }

        return npos;
    }

private:
    void Initialize(const char* str, size_t num) NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    void MoveFrom(String& other) NN_NOEXCEPT
    {
        m_Count = other.m_Count;
        m_pBuffer = other.m_pBuffer;
        other.m_Count = 0;
        other.m_pBuffer = nullptr;
    }

    String& Append(const char* str, size_t num) NN_NOEXCEPT;

    String& Erase(size_t pos, size_t len) NN_NOEXCEPT;
};

inline ::nnt::testing::detail::String operator+(
    const ::nnt::testing::detail::String& lhs,
    const ::nnt::testing::detail::String& rhs) NN_NOEXCEPT
{
    return ::nnt::testing::detail::String(lhs).append(rhs);
}

inline ::nnt::testing::detail::String operator+(
    const char* lhs, const ::nnt::testing::detail::String& rhs) NN_NOEXCEPT
{
    return ::nnt::testing::detail::String(lhs).append(rhs);
}

inline ::nnt::testing::detail::String operator+(
    char lhs, const ::nnt::testing::detail::String& rhs) NN_NOEXCEPT
{
    return ::nnt::testing::detail::String(1, lhs).append(rhs);
}

inline ::nnt::testing::detail::String operator+(
    const ::nnt::testing::detail::String& lhs, const char* rhs) NN_NOEXCEPT
{
    return ::nnt::testing::detail::String(lhs).append(rhs);
}

inline ::nnt::testing::detail::String operator+(
    const ::nnt::testing::detail::String& lhs, char rhs) NN_NOEXCEPT
{
    return ::nnt::testing::detail::String(lhs).append(1, rhs);
}

inline bool operator==(const ::nnt::testing::detail::String& lhs,
                       const ::nnt::testing::detail::String& rhs) NN_NOEXCEPT
{
    return lhs.compare(rhs) == 0;
}

inline bool operator==(const char* lhs,
                       const ::nnt::testing::detail::String& rhs) NN_NOEXCEPT
{
    return rhs.compare(lhs) == 0;
}

inline bool operator==(const ::nnt::testing::detail::String& lhs,
                       const char* rhs) NN_NOEXCEPT
{
    return lhs.compare(rhs) == 0;
}

inline bool operator!=(const ::nnt::testing::detail::String& lhs,
                       const ::nnt::testing::detail::String& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

inline bool operator!=(const char* lhs,
                       const ::nnt::testing::detail::String& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

inline bool operator!=(const ::nnt::testing::detail::String& lhs,
                       const char* rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

inline bool operator<(const ::nnt::testing::detail::String& lhs,
                      const ::nnt::testing::detail::String& rhs) NN_NOEXCEPT
{
    return lhs.compare(rhs) < 0;
}

}}} // namespace nnt::testing::detail
