﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nnt/gtest/detail/gtest-heap.h>

namespace nnt { namespace testing { namespace detail {

template <typename T>
class List final
{
public:
    NNT_TESTING_DETAIL_HEAP_IS_ALLOCATABLE();

private:
    class Value final
    {
    public:
        NNT_TESTING_DETAIL_HEAP_IS_ALLOCATABLE();

    private:
        T m_Value;

    public:
        explicit Value(const T& value) NN_NOEXCEPT : m_Value(value) {}

        T& Get() NN_NOEXCEPT { return m_Value; }

        const T& Get() const NN_NOEXCEPT { return m_Value; }
    };

    class Node final
    {
    public:
        NNT_TESTING_DETAIL_HEAP_IS_ALLOCATABLE();

    private:
        Value* m_pValue;
        Node* m_pBefore;
        Node* m_pBehind;

    public:
        Node() NN_NOEXCEPT
            : m_pValue(nullptr)
            , m_pBefore(nullptr)
            , m_pBehind(nullptr)
        {
        }

        ~Node() NN_NOEXCEPT
        {
            if (m_pValue != nullptr) { delete m_pValue; }
        }

        T& GetValue() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pValue);
            return m_pValue->Get();
        }

        const T& GetValue() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pValue);
            return m_pValue->Get();
        }

        void SetValue(const T& value) NN_NOEXCEPT
        {
            if (m_pValue != nullptr) { delete m_pValue; }
            m_pValue = new Value(value);
        }

        Node* GetBefore() NN_NOEXCEPT
        {
            return m_pBefore;
        }

        const Node* GetBefore() const NN_NOEXCEPT
        {
            return m_pBefore;
        }

        void SetBefore(Node* pNode) NN_NOEXCEPT
        {
            m_pBefore = pNode;
        }

        Node* GetBehind() NN_NOEXCEPT
        {
            return m_pBehind;
        }

        const Node* GetBehind() const NN_NOEXCEPT
        {
            return m_pBehind;
        }

        void SetBehind(Node* pNode) NN_NOEXCEPT
        {
            m_pBehind = pNode;
        }

        void Insert(Node* pHead, Node* pTail) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pHead);
            NN_SDK_REQUIRES_NOT_NULL(pTail);
            if (m_pBehind != nullptr) { m_pBehind->m_pBefore = pTail; }
            pHead->m_pBefore = this;
            pTail->m_pBehind = m_pBehind;
            m_pBehind = pHead;
        }
    };

public:
    typedef T value_type;

public:
    class const_iterator
    {
    public:
        typedef ptrdiff_t difference_type;
        typedef T value_type;

    protected:
        Node* m_pNode;

    public:
        explicit const_iterator(Node* pNode) NN_NOEXCEPT : m_pNode(pNode) {}

        const_iterator(const const_iterator& other) NN_NOEXCEPT
            : m_pNode(other.m_pNode) {}

        const T& operator*() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pNode);
            return m_pNode->GetValue();
        }

        const T* operator->() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pNode);
            return &m_pNode->GetValue();
        }

        const_iterator& operator++() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pNode);
            m_pNode = m_pNode->GetBehind();
            return *this;
        }

        bool operator==(const const_iterator& other) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pNode);
            return m_pNode == other.m_pNode;
        }

        bool operator!=(const const_iterator& other) const NN_NOEXCEPT
        {
            return !(*this == other);
        }
    };

    class iterator final : public const_iterator
    {
    public:
        explicit iterator(Node* pNode) NN_NOEXCEPT : const_iterator(pNode) {}

        iterator(const iterator& other) NN_NOEXCEPT
            : const_iterator(other.m_pNode) {}
    };

private:
    Node *m_pHead;

    Node *m_pTail;

    size_t m_Count;

public:
    List() NN_NOEXCEPT
    {
        this->Initialize();
    }

    List(const List& other) NN_NOEXCEPT
    {
        this->Initialize();

        this->Append(other.begin(), other.end());
    }

    template<class InputIt>
    List(InputIt head, InputIt tail) NN_NOEXCEPT
    {
        this->Initialize();

        this->Append(head, tail);
    }

    ~List() NN_NOEXCEPT
    {
        this->Finalize();
    }

    List& operator=(const List& other) NN_NOEXCEPT
    {
        if (this != &other)
        {
            this->Finalize();

            this->Initialize();

            this->Append(other.begin(), other.end());
        }
    }

    size_t size() const NN_NOEXCEPT
    {
        return m_Count;
    }

    const_iterator begin() const NN_NOEXCEPT
    {
        return const_iterator(m_pHead);
    }

    iterator begin() NN_NOEXCEPT
    {
        return iterator(m_pHead);
    }

    const_iterator end() const NN_NOEXCEPT
    {
        return const_iterator(m_pTail);
    }

    iterator end() NN_NOEXCEPT
    {
        return iterator(m_pTail);
    }

    bool operator==(const List& other) const NN_NOEXCEPT
    {
        if (this->size() != other.size())
        {
            return false;
        }

        const_iterator head = this->begin();
        const_iterator tail = this->end();
        const_iterator peer = other.begin();

        while (head != tail)
        {
            if (*head != *peer)
            {
                return false;
            }

            ++head;
            ++peer;
        }

        return true;
    }

    void push_back(const T& value) NN_NOEXCEPT
    {
        Node* pNode = new Node();
        pNode->SetValue(value);

        Node* pBefore = m_pTail->GetBefore();
        if (pBefore != nullptr)
        {
            pBefore->Insert(pNode, pNode);
        }
        else
        {
            m_pHead = pNode;
            pNode->Insert(m_pTail, m_pTail);
        }

        ++m_Count;
    }

    void splice(iterator iter, List& list) NN_NOEXCEPT
    {
        Node* pNode = m_pHead;

        while (pNode != m_pTail && iter != iterator(pNode))
        {
            pNode = pNode->GetBehind();
        }

        NN_SDK_ASSERT_EQUAL(iter, iterator(pNode));

        if (list.m_pHead == list.m_pTail)
        {
            return;
        }

        if (pNode->GetBefore() != nullptr)
        {
            pNode->GetBefore()->Insert(list.m_pHead, list.m_pTail->GetBefore());
        }
        else
        {
            m_pHead = list.m_pHead;
            Node* pBefore = list.m_pTail->GetBefore();
            pBefore->SetBehind(pNode);
            pNode->SetBefore(pBefore);
        }

        m_Count += list.m_Count;

        list.m_Count = 0;
        list.m_pHead = list.m_pTail;
        list.m_pTail->SetBefore(nullptr);
    }

private:
    void Initialize() NN_NOEXCEPT
    {

        m_pTail = new Node();
        m_pHead = m_pTail;
        m_Count = 0;
    }

    void Finalize() NN_NOEXCEPT
    {
        for (Node* pNode = m_pHead; pNode != nullptr; )
        {
            Node* pBehind = pNode->GetBehind();
            delete pNode;
            pNode = pBehind;
        }
    }

    template<typename InputIt>
    void Append(InputIt head, InputIt tail) NN_NOEXCEPT
    {
        for (InputIt iter = head; iter != tail; ++iter)
        {
            this->push_back(*iter);
        }
    }
};

}}} // namespace nnt::testing::detail
